# incoming.py = deal with incoming email
#
##########################################################
# Copyright (c) 2001 Philip Hunt. See COPYING for details
# of licensing information.
##########################################################

# Last altered: 10-Aug-2001
# History:
# 1-Jun-2001 PH: created
# 9-Aug-2001 PH: copy the values from the plaintext-header into
# the real header of the decrypted incoming email.

"""***
Notes here about the incoming.py module...

***"""

#***** python standard libraries:
import rfc822   # understands mail headers
import sys
import os
import string
import copy

#***** PH libraries
import utility
import mailheader

#***** part of Herbrip:
from herb_globals import *
import startup
import encwrap

debug = 0  # debugging this module?


#---------------------------------------------------------------------
"""***
process incoming email, reading from filename (fnEmailIn) and
writing plaintext to filename (fnEmailPt).
***"""

def processIn(fnEmailIn, fnEmailPt):
   # get a file handle for the incoming mail
   try:
      mailIn = open(fnEmailIn, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting"\
         % fnEmailIn
      sys.exit(1)   
   
   # read the header lines
   inm = mailheader.Mail()
   inm.readFromFileHandle(mailIn)
   
   processedMail = processInMail(inm)
 
   # output result
   utility.writeFile(fnEmailPt, processedMail.asString())
      
#---------------------------------------------------------------------

"""***
returns a mailheader.Mail object containing the result
***"""

def processInMail(inm):
   
   xHerbivore = inm.header.getheader(X_HV)
   if xHerbivore == None:
      print "herbrip: Incoming email <%s> has no '%s:' header, exiting" \
         % (fnEmailIn, X_HV)
      sys.exit(0)
   
   xHerbivoreVersion = inm.header.getheader(X_HV_VERSION)
   # for now, ignore version. Later we will check it for backwards
   # compatibility
   
   if debug: print "** xHerbivore: %s" % xHerbivore
   if debug: print "** xHerbivore-Version: %s" % xHerbivoreVersion
   
   xHerbivoreKey = inm.header.getheader(X_HV_KEY)
   if debug: print """--- Herbivore Key ---
%s
--- end Herbivore Key ---""" % xHerbivoreKey

   #get email address, because we need to store that with the key
   fromName,fromAddr = inm.header.getaddr("From")
   if debug: print "** from \"%s\" <%s>" % (fromName, fromAddr)
   
   dictAdd(fromAddr, fromName, xHerbivoreKey)

   result = mailheader.Mail()
   result.header = copy.deepcopy(inm.header) 
   
   # decode it if we need to
   if xHerbivore == X_HV_ENCRYPTED:
      decryptForInput(inm, result)
   else:
      result.body.readFromString(inm.body.asString())
      
   return result  

#---------------------------------------------------------------------

"""***
Decrypt, and process headers-in-decrypted-plaintext.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) the unprocessed message
mResult (a mailheader.Mail) the same message, when processed.

On input (mResult) has a copy of (m)'s header.

***"""

def decryptForInput(m, mResult):
   myPrivateKey = startup.secretKeys['private']
   plainText = encwrap.pkDec(myPrivateKey, m.body.asString())
      
   # extract the "headers" from (plainText):
   ptAsMailMessage = mailheader.Mail()
   ptAsMailMessage.readFromString(plainText)
   pseudoHeader = ptAsMailMessage.header
   
   #??? make this a constant in herb_globals.py ???
   interestingFields = ["Subject", "In-Reply-To", "References", 
      "Keywords", "Comments"]
      
   for f in interestingFields:
      v = pseudoHeader.getheader(f)
      if v != None:
         mResult.header.put(f,v)
         if debug: print "*** adding header %s: %s" % (f,v)
         
   if debug:
      for f in pseudoHeader.getheadernames():
         if f not in interestingFields:
            print "### ignoring header %s: %s" % (f, pseudoHeader.get(f))      

   # (plainText) is just the message body, so we must add
   # the message headers to it:

   mResult.body = ptAsMailMessage.body




#---------------------------------------------------------------------

"""***
Add the sender's public key to our dictionary of public keys.
***"""

def dictAdd(fromAddr, fromName, xHerbivoreKey):
   if debug and 0:
      print ("** dictAdd(fromAddr='%s', fromName='%s', ...)" 
         % (fromAddr, fromName))

   startup.publicKeys.addRecord(fromAddr, fromName, xHerbivoreKey)
   startup.publicKeys.save()
   

#---------------------------------------------------------------------


#---------------------------------------------------------------------


#end incoming.py
