# test_herbrip.py = Test the herbrip command-line client
#
##########################################################
# Copyright (c) 2001 Philip Hunt. See COPYING for details
# of licensing information.
##########################################################

# Last altered: 28-Nov-2001
# History:
# 12-Jul-2001 PH: created
# 12-Nov-2001 PH: modifying for new directory structure

"""***
test_herbrip.py is a regression test suite for herbrip.

To run it, go into the directory it is in (./test/) and run from
the command line:

   $ python test_herbrip.py
   
The program will create several files in its own directory, and
several subdirectories (this is why it is important to run it with
its own directory as the current directory).   
   
***"""

#***** python standard libraries:
import sys
import os, os.path
import commands
import rfc822
import string

srcDir = os.path.abspath("../src")
sys.path.append(srcDir)
import utility
import mailheader

debug = 0     # debugging this module?
interact = 0  # is this module interactive?
verbose = 1   # verbose means output more information

dataDir = os.path.abspath("../testtmp")


#---------------------------------------------------------------------

__debug__ = 1 #this is so the 'assert' statements work

# doesn't work with herbrip anymore because its executable isn't in 
# the path
def run(comment, command):
   if comment: 
      print "\n" + comment, 
      if interact:
         dummy = raw_input(">")
      else:
         print
   print "CMD { %s }..." % command
   os.system(command)
   
   
   
      
def runHerbrip(comment, herbripArguments):
   if comment: 
      print "\n" + comment, 
      if interact:
         dummy = raw_input(">")
      else:
         print
   cmd2 = "%s/herbrip %s" % (srcDir, herbripArguments)
   print "CMD { %s }..." % cmd2
   os.system(cmd2)
     
      
def comment(com):
   print "==== %s ====" % com

#---------------------------------------------------------------------

def _getBody(mailFn):
   """return a string containing the body of a mail file"""
   fh = open(mailFn)
   mess = rfc822.Message(fh)
   return fh.read()
   
def _getHeader(mailFn):
   """Return a mailheader.MailHeader object containing the headers
      of the mail file stored in filename (mailFn)"""
   fh = open(mailFn)
   mess = mailheader.makeMailFromFileHandle(fh) 
   return mess.header
   

#---------------------------------------------------------------------

def strContains(s, sub):
   """does string (s) contain (sub)?"""
   if sub == "": return 1
   if s == "": return 0
   return string.find(s, sub) != -1

#---------------------------------------------------------------------

def assertExists(filename):
   "test that a file exists and is readable"
   if not os.access(filename, os.F_OK):
      raise AssertionError, "File <%s> doesn't exist" % filename
   if not os.access(filename, os.R_OK):
      raise AssertionError, "File <%s> isn't readable" % filename
      
   if debug or verbose:
      print "<%s> exists, GOOD." % filename
 
def assertContains(filename, data, sub):
   "test that (filename) contains a substring"
   if not strContains(data, sub):
      raise AssertionError, "File <%s> has no '%s'" % (filename, sub)
        
def assertEnc(filename): 
   "test that (filename) exists and is an encrypted email"
   assertExists(filename)
   data = utility.readFile(filename)
   assertContains(filename, data, "X-Herbivore: enc")
   assertContains(filename, data, "----- BEGIN HERBIVORE ENCRYPTED -----")
   assertContains(filename, data, "-----")
   assertContains(filename, data, "----- END HERBIVORE ENCRYPTED -----")
   if debug or verbose:
      print "<%s> is encrypted mail, GOOD." % filename
   
         
def assertHerbPlain(filename): 
   """test that (filename) exists and is an unencrypted (plaintext) email
      with a 'Herbivore: plain' header"""
   assertExists(filename)
   data = utility.readFile(filename)
   assertContains(filename, data, "X-Herbivore: plain")
   if debug or verbose:
      print "<%s> is Herbivore-plaintext mail, GOOD." % filename
      
def assertNotEnc(filename): 
   """test that (filename) exists and doesn't have Herbivore encryption"""
   assertExists(filename)
   data = utility.readFile(filename)
   try:
      assertEnc(filename)
   except:
      pass # should be an exception, as it shouldn't be encypted   
   else:   
      raise AssertionError, \
         "File <%s> is herbivore-encrypted when it shouldn't be" \
            % filename
   if debug or verbose:
      print "<%s> is non-Herbivore-encrypted mail, GOOD." % filename
     
def assertSame(fn1, fn2):
   "test that two files are the same"
   cmd = "diff %s %s" % (fn1, fn2)
   result = commands.getoutput(cmd)
   if result != "":
      print "--- result of diff ---"
      print result
      print "--- end result -------"
      raise AssertionError, "Files <%s>, <%s> are different" % (fn1, fn2)

   if debug or verbose:
      print "<%s> is the same as <%s>, GOOD." & (fn1, fn2)


"""***
(mf1) and (mf2) are filenames of email files. This function 
asserts that they have the same bodies; the headers may be different.
***"""

def assertSameBody(mf1, mf2):
   body1 = _getBody(mf1)
   body2 = _getBody(mf2)
   if body1 != body2:
      print "Bodies are different!"
      raise AssertionError, "Mail files <%s>, <%s> have different bodies"\
         % (mf1, mf2)
         
   if debug or verbose:
      print "<%s> has the same body as <%s>, GOOD." % (mf1, mf2)
  
  
 
"""***
(mf1) and (mf2) are filenames of email files. This function 
asserts that they have the same value for one of more header
fields (in subsequent arguments).
***"""

def assertSameHeader(mf1, mf2, *hfs):
   header1 = _getHeader(mf1)
   header2 = _getHeader(mf2)
   for hf in hfs:
      #print "!!! testing header field '%s' !!!" % hf
      if header1.get(hf) != header2.get(hf):
         errorTxt = ("Mail files <%s>, <%s> have different values for"
            + "\nheader %s; Values: [%s], [%s]") % \
               (mf1, mf2, hf, header1.get(hf), header2.get(hf))
         raise AssertionError, errorTxt
         
   if debug or verbose:
      plural = ""
      if len(hfs) > 1: plural = "s"
      headersStr = ""
      for hf in hfs: headersStr = headersStr + hf + ", "
      headersStr = headersStr[:-2]
      print "<%s>, <%s> have same %s header%s GOOD." % (mf1, mf2,
         headersStr, plural)


#---------------------------------------------------------------------
# emails to send

messAB = """From: alice@nowhere.org
To: bob@nowhere.org
Subject: message from Alice to Bob

Hi, Bob, how's it going

--
Alice
"""
   
messBA = """From: <bob@nowhere.org> (Robert Xavier Roberto)
To: alice@nowhere.org
Subject: message from Bob to Alice


. (testing multiple empty lines before start of text)
I'm fine, Alice. How are you?

--
Bob
"""

messCA = """From: cathy@nowhere.org
To: alice@nowhere.org
Subject: part of the GPL

  The licenses for most software are designed to take away your
freedom to share and change it.  By contrast, the GNU General Public
License is intended to guarantee your freedom to share and change free
software--to make sure the software is free for all its users.  This
General Public License applies to most of the Free Software
Foundation's software and to any other program whose authors commit to
using it.  (Some other Free Software Foundation software is covered by
the GNU Library General Public License instead.)  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
this service if you wish), that you receive source code or can get it
if you want it, that you can change the software or use pieces of it
in new free programs; and that you know you can do these things.

--
*********************************************************
**-----------------------------------------------------**
**  Cathy, cathy@nowhere.org                           **
**                                                     **
**  Look at my sig, it's...                            **
**                        R A T H E R   B I G   ! !    **
**-----------------------------------------------------**
*********************************************************
"""

messABCD = """From: alice@nowhere.org
To: bob@nowhere.org, cathy@nowhere.org, daniel@nowhere.org
Subject: look at cathy's sig!!!

Cathy wrote:
> --
> *********************************************************
> **-----------------------------------------------------**
> **  Cathy, cathy@nowhere.org                           **
> **                                                     **
> **  Look at my sig, it's...                            **
> **                        R A T H E R   B I G   ! !    **
> **-----------------------------------------------------**
> *********************************************************

oooh, pretty!!!

-- 
Alice
"""

messD_ccABC = """From: daniel@nowhere.org
Cc: alice@nowhere.org, bob@nowhere.org, cathy@nowhere.org
Subject: message from D, cc'd to A B and C

D to A,B.C

-- 
Daniel
"""



#---------------------------------------------------------------------

"""***
Test with 4 users, Alice, Bob, Cathy and Daniel. All have addresses
<name>@nowhere.org.

Tests preformed:

* initialize alice
* initialize bob

* send message alice -> bob
  * outbound processing: --out messab messab_out 
  * inbound processing: --in messab_out messab_in

* send message bob -> alice
  * outbound processing: --out messba messba_out 
  * inbound processing: --in messba_out messba_in

* initialize cathy
* send message cathy -> alice
  * outbound processing (using --outd option):
    --dest "cp %(fn)s messca_out" --outd messca
  * inbound processing

* initialize daniel
* send message alice -> bob, cathy, daniel (Note that A has B and
  C's keys, but not D's key).
  * outbound processing using --outp, also uses "oft" program
    creates messo_ab, messo_ac, messo_ad
  * inbound processing by bob -> messi_ab  
  * inbound processing by cathy -> messi_ac  
  * inbound processing by daniel -> messi_ad  
 
  

***"""


def test_start():
   run("delete old message files", "rm -rf mess*")
   run("", "rm -rf alice")
   run("", "rm -rf bob")
   run("", "rm -rf cathy")
   run("", "rm -rf daniel")
   runHerbrip("Initialize Alice", "--dir alice --create")
   assertExists("alice/secret_keys")
   #assertExists("alice/public_keys")
   runHerbrip("Initialize Bob", "--dir bob --create")
   assertExists("bob/secret_keys")
   #assertExists("bob/public_keys")
      
def test_AB():
   comment("sending a message Alice --> Bob")
   utility.writeFile("messab", messAB)
   assertExists("messab")

   runHerbrip("Alice prepares her outgoing message", 
      "--dir alice --out messab messab_out")
   assertExists("messab_out")
   
   runHerbrip("Bob reads it, extracting Alice's public key",
      "--dir bob --in messab_out messab_in")
   assertExists("messab_in")
   assertExists("bob/public_keys")
   assertSameBody("messab", "messab_in")
  
def test_BA():
   comment("sending reply Bob --> Alice")
   utility.writeFile("messba", messBA)
   assertExists("messba")
   assertExists("messba")

   runHerbrip("Bob prepares his outgoing message (should encrypt it)", 
      "--dir bob --out messba messba_out")
   assertEnc("messba_out")
     
   runHerbrip("Alice reads it, decrypting it with her private key",
      "--dir alice --in messba_out messba_in")
   assertExists("messba_in")
   assertExists("alice/public_keys")
   assertSameBody("messba", "messba_in")

def test_CA():
   runHerbrip("Initialize Cathy", "--dir cathy --create")
   
   comment("sending message Cathy --> Alice")
   utility.writeFile("messca", messCA)
   assertExists("messca")
      
   runHerbrip("Cathy prepares her outgoing message (should encrypt it)", 
      '--dir cathy --dest "cp %(fn)s messca_out" --outp messca')
   assertExists("messca_out")  
    
   runHerbrip("Alice reads it, decrypting it with her private key",
      "--dir alice --in messca_out messca_in")
   assertExists("messca_in")
   assertSameBody("messca", "messca_in")
 
def test_ABCD():
   runHerbrip("Initialize Daniel", "--dir daniel --create")
   
   comment("sending message Alice --> Bob, Cathy, Daniel")
   utility.writeFile("mess_abcd", messABCD)
   assertExists("mess_abcd")
   
   runHerbrip("Alice prepares her outgoing message", 
      '--dir alice --dest "oft %(fn)s" --outp mess_abcd')
   assertExists("messo_ab")  
   assertExists("messo_ac")  
   assertExists("messo_ad")  
 
   runHerbrip("Bob reads Alice's message", 
      '--dir bob --in messo_ab messi_ab')
   assertExists("messi_ab")
   assertSameBody("mess_abcd", "messi_ab")
   
   runHerbrip("Cathy reads Alice's message", 
      '--dir cathy --in messo_ac messi_ac')
   assertExists("messi_ac")
   assertSameBody("mess_abcd", "messi_ac")
   
   runHerbrip("Daniel reads Alice's message", 
      '--dir daniel --in messo_ad messi_ad')
   assertExists("messi_ad")
   assertSameBody("mess_abcd", "messi_ad")
   
 
def test_DccABC():
   """testing when D CC:s a message to A B and C"""
   comment("sending message Daniel -->(CC:) Alice, Bob, Cathy")
   utility.writeFile("mess5_dabc", messD_ccABC)
   assertNotEnc("mess5_dabc")
   
   runHerbrip("Daniel prepares his outgoing message", 
      '--dir daniel --dest "opft mess5o %(fn)s" --outp mess5_dabc')
   assertEnc("mess5o_da")  
   assertHerbPlain("mess5o_db")  
   assertHerbPlain("mess5o_dc")  
   
   runHerbrip("Alice reads Daniel's message", 
      '--dir alice --in mess5o_da mess5i_da')
   assertExists("mess5i_da")
   assertSameBody("mess5_dabc", "mess5i_da")
   # The message going to alice is the only encrypted one, so it is
   # the only one that should preserve the Cc: header 
   assertSameHeader("mess5_dabc", "mess5i_da", "Cc", "Subject")
   
   runHerbrip("Bob reads Daniel's message", 
      '--dir bob --in mess5o_db mess5i_db')
   assertExists("mess5i_db")
   assertSameBody("mess5_dabc", "mess5i_db")
    
   runHerbrip("Cathy reads Daniel's message", 
      '--dir alice --in mess5o_dc mess5i_dc')
   assertExists("mess5i_dc")
   assertSameBody("mess5_dabc", "mess5i_dc")
  
   
   
   
def testAll():
   test_start()
   test_AB()
   test_BA()
   test_CA()
   test_ABCD()
   test_DccABC()
   print "***** TESTS COMPLETED SUCCESSFULLY ! *****"
         


#---------------------------------------------------------------------


if __name__ == "__main__":
   testAll()

#end test_herbrip.py
