# logging.py = herbrip logging (for debugging etc)
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 23-Jan-2002
# History:
# 23-Jan-2002 PH: created

"""***
This module enables Herbrip logging. Log files are written in the
~/.herbivore/ directory and are intended to be useful for debugging.
***"""

#***** python standard libraries:
import sys
import string
import time

#***** PH library:
import mailheader

#***** part of Herbrip:

debug = 0 # debugging this module?

#---------------------------------------------------------------------

logAllItems = ("o", "op", "i", "ip", "cmd", "comment")

class Log:
   def __init__(self):
      self.logItems = logAllItems

   def write(self, text): 
      """(implemented by subclass)"""  
      
   def writeln(self, text): 
      self.write(text + "\n")

   def invoked(self, argv):
      """a herbrip program was invoked"""
      tup = time.localtime(time.time())
      timestamp = time.strftime("%a %Y-%b-%d %H:%M:%S", tup)
      cmd = string.join(argv, " ")
      text = "\n!!!!!!!!!! %s, Herbrip was invoked:\n%s\n" % (timestamp, cmd)
      self.write(text)
   
   def message(self, type, mess):
      type = string.lower(type)
      if type not in self.logItems:
         # we aren't logging this type of message
         return
      t = "mail"
      if type == "i": t = "INCOMING mail"
      if type == "ip": t = "INCOMING mail, processed result"
      if type == "o": t = "OUTGOING mail"
      if type == "op": t = "OUTGOING mail, processed result"
      top = "########## " + t + " "
      top2 = top + ("#"*(70 - len(top)))
      text = top2 + "\n" + str(mess)
      self.write(text)
   
   def invoking(self, cmd):
      if "cmd" in self.logItems:
         self.writeln("CMD { %s }" % cmd)
   
   def comment(self, text):
      """a comment"""
      if "comment" in self.logItems:
         self.writeln("# " + text)
   

#---------------------------------------------------------------------

class NullLog(Log):
   """ an example of the Null Object design pattern """
   def write(self, text): pass
   
   # these aren't really necessary, but add them too:
   def invoked(self, argv): pass
   def message(self, type, mess): pass
   def invoking(self, cmd): pass
   def comment(self, text): pass


class FileLog(Log):
   def __init__(self, filename, stuffToLog =logAllItems):
      self.filename = filename
      self.logItems = stuffToLog
      
   def write(self, text):
      f = open(self.filename, "a+")
      f.write(text)
      f.close()   
      
class PrintLog(Log):
   def write(self, text):
      print text,
     

#---------------------------------------------------------------------
# testing this module:

msg = """From: philh@comuno.freeserve.co.uk
To: alice@nowhere.org
Subject: hello, world

Hello, world!!

-- 
phil hunt
"""

def testme():
   print "***** testing logging.py *****"
   log = PrintLog()
   log.comment("this is a comment")
   log.invoked(sys.argv)
   log.comment("comment2")
   log.invoking("cp hello goodbye")
   log.message("I", msg)
   msgMail = mailheader.makeMailFromString(msg)
   msgMail.header.put("X-Herbivore-Proc", "yes")
   log.message("OP", msgMail)
   log.comment("comment3")

#---------------------------------------------------------------------

if __name__ == "__main__": 
   testme()

#end logging.py
