# test_herbrip.py = Test the herbrip command-line client
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file  COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 24-Jan-2002
# History:
# 12-Jul-2001 PH: created
# 12-Nov-2001 PH: modifying for new directory structure

"""***
test_herbrip.py is a regression test suite for herbrip.

To run it, go into the directory it is in (./test/) and run from
the command line:

   $ python test_herbrip.py
   
The program will create several files in its own directory, and
several subdirectories (this is why it is important to run it with
its own directory as the current directory).   
   
***"""

#***** python standard libraries:
import sys
import os, os.path
import commands
import rfc822
import string
import time

#***** modules in ../src directory:
srcDir = os.path.abspath("../src")
sys.path.append(srcDir)
import utility
import mailheader

#***** modules for testing:
import tmessages

debug = 0     # debugging this module?
interact = 0  # is this module interactive?
verbose = 1   # verbose means output more information

dataDir = os.path.abspath("../testtmp")


#---------------------------------------------------------------------

__debug__ = 1 #this is so the 'assert' statements work

# doesn't work with herbrip anymore because its executable isn't in 
# the path
def run(comment, command):
   if comment: 
      print "\n" + comment, 
      if interact:
         dummy = raw_input(">")
      else:
         print
   print "CMD { %s }..." % command
   os.system(command)
  
      
def runHerbrip(comment, herbripArguments, herbripExeName ="herbrip"):
   if comment: 
      print "\n" + comment, 
      if interact:
         dummy = raw_input(">")
      else:
         print      
   cmd2 = "%s/%s %s" % (srcDir, herbripExeName, herbripArguments)
   print "CMD { %s }..." % cmd2
   os.system(cmd2)
     
def runHerbrip_csm(comment, herbripArguments):
   runHerbrip(comment, herbripArguments, "herbrip_csm")
   
def runHerbrip_pin(comment, herbripArguments):
   runHerbrip(comment, herbripArguments, "herbrip_pin")
     
def comment(com):
   print "==== %s ====" % com

#---------------------------------------------------------------------

def _getBody(mailFn):
   """return a string containing the body of a mail file"""
   fh = open(mailFn)
   mess = rfc822.Message(fh)
   return fh.read()
   
def _getHeader(mailFn):
   """Return a mailheader.MailHeader object containing the headers
      of the mail file stored in filename (mailFn)"""
   fh = open(mailFn)
   mess = mailheader.makeMailFromFileHandle(fh) 
   return mess.header
   

#---------------------------------------------------------------------

def strContains(s, sub):
   """does string (s) contain (sub)?"""
   if sub == "": return 1
   if s == "": return 0
   return string.find(s, sub) != -1

#---------------------------------------------------------------------

def assertExists(filename):
   "test that a file exists and is readable"
   if not os.access(filename, os.F_OK):
      raise AssertionError, "File <%s> doesn't exist" % filename
   if not os.access(filename, os.R_OK):
      raise AssertionError, "File <%s> isn't readable" % filename
      
   if debug or verbose:
      print "<%s> exists, GOOD." % filename
 
def assertContains(filename, data, sub):
   "test that (filename) contains a substring"
   if not strContains(data, sub):
      raise AssertionError, "File <%s> has no '%s'" % (filename, sub)
        
def assertEnc(filename): 
   "test that (filename) exists and is an encrypted email"
   assertExists(filename)
   data = utility.readFile(filename)
   assertContains(filename, data, "X-Herbivore: enc")
   assertContains(filename, data, "----- BEGIN HERBIVORE ENCRYPTED -----")
   assertContains(filename, data, "-----")
   assertContains(filename, data, "----- END HERBIVORE ENCRYPTED -----")
   if debug or verbose:
      print "<%s> is encrypted mail, GOOD." % filename
   
         
def assertHerbPlain(filename): 
   """test that (filename) exists and is an unencrypted (plaintext) email
      with a 'Herbivore: plain' header"""
   assertExists(filename)
   data = utility.readFile(filename)
   assertContains(filename, data, "X-Herbivore: plain")
   if debug or verbose:
      print "<%s> is Herbivore-plaintext mail, GOOD." % filename
      
def assertNotEnc(filename): 
   """test that (filename) exists and doesn't have Herbivore encryption"""
   assertExists(filename)
   data = utility.readFile(filename)
   try:
      assertEnc(filename)
   except:
      pass # should be an exception, as it shouldn't be encypted   
   else:   
      raise AssertionError, \
         "File <%s> is herbivore-encrypted when it shouldn't be" \
            % filename
   if debug or verbose:
      print "<%s> is non-Herbivore-encrypted mail, GOOD." % filename
     
def assertSame(fn1, fn2):
   "test that two files are the same"
   cmd = "diff %s %s" % (fn1, fn2)
   result = commands.getoutput(cmd)
   if result != "":
      print "--- result of diff ---"
      print result
      print "--- end result -------"
      raise AssertionError, "Files <%s>, <%s> are different" % (fn1, fn2)

   if debug or verbose:
      print "<%s> is the same as <%s>, GOOD." & (fn1, fn2)


"""***
(mf1) and (mf2) are filenames of email files. This function 
asserts that they have the same bodies; the headers may be different.
***"""

def assertSameBody(mf1, mf2):
   body1 = _getBody(mf1)
   body2 = _getBody(mf2)
   if body1 != body2:
      print "Bodies are different!"
      raise AssertionError, "Mail files <%s>, <%s> have different bodies"\
         % (mf1, mf2)
         
   if debug or verbose:
      print "<%s> has the same body as <%s>, GOOD." % (mf1, mf2)
  
  
 
"""***
(mf1) and (mf2) are filenames of email files. This function 
asserts that they have the same value for one of more header
fields (in subsequent arguments).
***"""

def assertSameHeader(mf1, mf2, *hfs):
   header1 = _getHeader(mf1)
   header2 = _getHeader(mf2)
   for hf in hfs:
      #print "!!! testing header field '%s' !!!" % hf
      if header1.get(hf) != header2.get(hf):
         errorTxt = ("Mail files <%s>, <%s> have different values for"
            + "\nheader %s; Values: [%s], [%s]") % \
               (mf1, mf2, hf, header1.get(hf), header2.get(hf))
         raise AssertionError, errorTxt
         
   if debug or verbose:
      plural = ""
      if len(hfs) > 1: plural = "s"
      headersStr = ""
      for hf in hfs: headersStr = headersStr + hf + ", "
      headersStr = headersStr[:-2]
      print "<%s>, <%s> have same %s header%s; GOOD." % (mf1, mf2,
         headersStr, plural)


"""***
Assert that in the message in file (mf), the header field (hf)
has the value (hfValue)
***"""

def assertHeaderValue(mf, hf, hfValue):
   header = _getHeader(mf)
   headerFieldStr = header.get(hf)
   if headerFieldStr != hfValue:
      errorTxt = ("Mail file <%s>, header %s: value is [%s] should be [%s]"
         % (mf, hf, headerFieldStr, hfValue))
      raise AssertionError, errorTxt  
      
   if debug or verbose:
      print "<%s> header %s: %s; GOOD." % (mf, hf, hfValue)

#---------------------------------------------------------------------

"""***
Test with 4 users, Alice, Bob, Cathy and Daniel. All have addresses
<name>@nowhere.org.

Tests preformed:

* initialize alice
* initialize bob

* send message alice -> bob
  * outbound processing: --out messab messab_out 
  * inbound processing: --in messab_out messab_in

* send message bob -> alice
  * outbound processing: --out messba messba_out 
  * inbound processing: --in messba_out messba_in

* initialize cathy
* send message cathy -> alice
  * outbound processing (using --outd option):
    --dest "mif:cp %(fn)s messca_out" --outd messca
  * inbound processing

* initialize daniel
* send message alice -> bob, cathy, daniel (Note that A has B and
  C's keys, but not D's key).
  * outbound processing using --outp, also uses "oft.py" program
    creates messo_ab, messo_ac, messo_ad
  * inbound processing by bob -> messi_ab  
  * inbound processing by cathy -> messi_ac  
  * inbound processing by daniel -> messi_ad  
 
* send message daniel -> alice, bob, cathy (Cc:'d to all three)
 
* send message cathy -> To: alice, Cc: bob and Bcc: daniel
  (check that handling of To/Cc/Bcc is working). Message is --out
  using the file: protocol.
   
* send message alice -> bob, using herbrip_csm. Bob decodes it,
  with the message going through herbrip_pin     

***"""


def test_start():
   run("delete old message files", "rm -rf mess*")
   run("", "rm -rf alice")
   run("", "rm -rf bob")
   run("", "rm -rf cathy")
   run("", "rm -rf daniel")
   runHerbrip("Initialize Alice", "--dir alice --create")
   assertExists("alice/secret_keys")
   #assertExists("alice/public_keys")
   runHerbrip("Initialize Bob", "--dir bob --create")
   assertExists("bob/secret_keys")
   #assertExists("bob/public_keys")
      
def test_AB():
   comment("sending a message Alice --> Bob")
   utility.writeFile("messab", tmessages.AB)
   assertExists("messab")

   runHerbrip("Alice prepares her outgoing message", 
      "--dir alice --out messab messab_out")
   assertExists("messab_out")
   
   runHerbrip("Bob reads it, extracting Alice's public key",
      "--dir bob --in messab_out messab_in")
   assertExists("messab_in")
   assertExists("bob/public_keys")
   assertSameBody("messab", "messab_in")
  
def test_BA():
   comment("sending reply Bob --> Alice")
   utility.writeFile("messba", tmessages.BA)
   assertExists("messba")
   assertExists("messba")

   runHerbrip("Bob prepares his outgoing message (should encrypt it)", 
      "--dir bob --out messba messba_out")
   assertEnc("messba_out")
     
   runHerbrip("Alice reads it, decrypting it with her private key",
      "--dir alice --in messba_out messba_in")
   assertExists("messba_in")
   assertExists("alice/public_keys")
   assertSameBody("messba", "messba_in")

def test_CA():
   runHerbrip("Initialize Cathy", "--dir cathy --create")
   
   comment("sending message Cathy --> Alice")
   utility.writeFile("messca", tmessages.CA)
   assertExists("messca")
      
   runHerbrip("Cathy prepares her outgoing message (should encrypt it)", 
      '--dir cathy --dest "mif:cp %(fn)s messca_out" --outd messca')
   assertExists("messca_out")  
    
   runHerbrip("Alice reads it, decrypting it with her private key",
      "--dir alice --in messca_out messca_in")
   assertExists("messca_in")
   assertSameBody("messca", "messca_in")
 
def test_ABCD():
   runHerbrip("Initialize Daniel", "--dir daniel --create")
   
   comment("sending message Alice --> Bob, Cathy, Daniel")
   utility.writeFile("mess4_abcd", tmessages.ABCD)
   assertExists("mess4_abcd")
   
   runHerbrip("Alice prepares her outgoing message", 
      '--dir alice --dest "file:mess4o_%(ix)s" --outd mess4_abcd')
   assertExists("mess4o_1")  
   assertExists("mess4o_2")  
   assertExists("mess4o_3")  
 
   runHerbrip("Bob reads Alice's message", 
      '--dir bob --in mess4o_1 mess4i_1')
   assertExists("mess4i_1")
   assertSameBody("mess4_abcd", "mess4i_1")
   assertSameHeader("mess4_abcd", "mess4i_1", "From", "To", "Subject")
      
   runHerbrip("Cathy reads Alice's message", 
      '--dir cathy --in mess4o_2 mess4i_2')
   assertExists("mess4i_2")
   assertSameBody("mess4_abcd", "mess4i_2")
   assertSameHeader("mess4_abcd", "mess4i_2", "From", "To", "Subject")
   
   runHerbrip("Daniel reads Alice's message", 
      '--dir daniel --in mess4o_3 mess4i_3')
   assertExists("mess4i_3")
   assertSameBody("mess4_abcd", "mess4i_3")
   assertSameHeader("mess4_abcd", "mess4i_3", "From", "To", "Subject")
   
 
def test_DccABC():
   """testing when D CC:s a message to A B and C """
   comment("sending message Daniel -->(CC:) Alice, Bob, Cathy")
   utility.writeFile("mess5_dabc", tmessages.D_ccABC)
   assertNotEnc("mess5_dabc")
   
   runHerbrip("Daniel prepares his outgoing message", 
      '--dir daniel --dest "sendmail:fake_sendmail.py mess5_" --outd mess5_dabc')
   assertEnc("mess5_alice")  
   assertHerbPlain("mess5_bob")  
   assertHerbPlain("mess5_cathy")  
   
   runHerbrip("Alice reads Daniel's message", 
      '--dir alice --in mess5_alice mess5i_da')
   assertExists("mess5i_da")
   assertSameBody("mess5_dabc", "mess5i_da")
   assertSameHeader("mess5_dabc", "mess5i_da", "Cc", "Subject")
   
   runHerbrip("Bob reads Daniel's message", 
      '--dir bob --in mess5_bob mess5i_db')
   assertExists("mess5i_db")
   assertSameBody("mess5_dabc", "mess5i_db")
    
   runHerbrip("Cathy reads Daniel's message", 
      '--dir cathy --in mess5_cathy mess5i_dc')
   assertExists("mess5i_dc")
   assertSameBody("mess5_dabc", "mess5i_dc")
  
def test_fileDest():
   """testing when --dest uses the 'file:' protocol; uses C_A_ccB_bccD"""
   comment("sending message Cathy --> Alice, Cc: Bob, Bcc: Daniel")
   utility.writeFile("mess6_cabd", tmessages.C_A_ccB_bccD)
   assertNotEnc("mess6_cabd")
    
   runHerbrip("Cathy prepares her outgoing message", 
      '--dir cathy --dest "file:mess6o_%(ix)s" --outd mess6_cabd')
   assertEnc("mess6o_1")  
   assertHeaderValue("mess6o_1", "To", "alice@nowhere.org")
   assertHerbPlain("mess6o_2")  
   assertHeaderValue("mess6o_2", "To", "alice@nowhere.org")
   assertHeaderValue("mess6o_2", "Cc", "bob@nowhere.org")
   assertEnc("mess6o_3") 
   assertHeaderValue("mess6o_3", "To", "daniel@nowhere.org")
   
   runHerbrip("Alice reads Cathy's message", 
      '--dir alice --in mess6o_1 mess6i_1')
   assertExists("mess6i_1")
   assertSameBody("mess6_cabd", "mess6i_1")
   # for all recipients, in the decoded message, the To: should
   # be alice, and the Cc: Bob. There shouldn't be a Bcc:, because
   # it is a *blind* carbon copy
   assertHeaderValue("mess6i_1", "To", "alice@nowhere.org")
   assertHeaderValue("mess6i_1", "Cc", "bob@nowhere.org")
   assertHeaderValue("mess6i_1", "Bcc", None)

   runHerbrip("Bob reads Cathy's message", 
      '--dir bob --in mess6o_2 mess6i_2')
   assertExists("mess6i_2")
   assertSameBody("mess6_cabd", "mess6i_2")
   assertHeaderValue("mess6i_2", "To", "alice@nowhere.org")
   assertHeaderValue("mess6i_2", "Cc", "bob@nowhere.org")
   assertHeaderValue("mess6i_2", "Bcc", None)
    
   runHerbrip("Daniel reads Cathy's message", 
      '--dir daniel --in mess6o_3 mess6i_3')
   assertExists("mess6i_3")
   assertSameBody("mess6_cabd", "mess6i_3")
   assertHeaderValue("mess6i_3", "To", "alice@nowhere.org")
   assertHeaderValue("mess6i_3", "Cc", "bob@nowhere.org")
   assertHeaderValue("mess6i_3", "Bcc", None)

def test_csm_pin():
   """ testing the herbrip_csm and herbrip_pin executables """ 
   
   # create output file, ready for processing: 
   utility.writeFile("mess7_ab", tmessages.AB)
   assertExists("mess7_ab")  
   
   runHerbrip_csm('Alice processes msg to Bob for output, using herbrip_csm',
      '--dir alice --dest "file:mess7_ab_o"'
      + ' bob@nowhere.org <mess7_ab')
   assertEnc("mess7_ab_o")  
   assertHeaderValue("mess7_ab_o", "To", "bob@nowhere.org") 
   # processed output file is <mess7_ab_o>
     
   runHerbrip_pin("Bob reads Alice's message, using herbrip_pin",
      "--dir bob <mess7_ab_o >mess7_ab_i")
   assertExists("mess7_ab_i")
   assertSameBody("mess7_ab", "mess7_ab_i")
   assertSameHeader("mess7_ab", "mess7_ab_i", "To", "Subject")
      
   
def testAll():
   test_start()
   test_AB()
   test_BA()
   test_CA()
   test_ABCD()
   test_DccABC()
   test_fileDest()
   test_csm_pin()
   print "***** TESTS COMPLETED SUCCESSFULLY ! *****"
         
#---------------------------------------------------------------------

if __name__ == "__main__":
   testAll()

#end test_herbrip.py
