# destination.py = send outgoing mail to its final destination
#
###############################################################
# Copyright (c) 2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 24-Jan-2002
# History:
# 19-Jan-2002 PH: created
# 24-Jan-2002: added logging of commands to Herbrip log file

"""***

***"""


#***** python standard libraries:
import string
import os
import commands

#***** PH libraries:
import mailheader
import utility

#***** Herbrip modules:
import startup

debug = 0

#---------------------------------------------------------------------

class Destination:
   def __str__(self):
      return self.__repr__()

#---------------------------------------------------------------------

class MifDest(Destination):
   typeId = "mif"
   
   def __init__(self, param):
      self.formatStr = param
        
   def output(self, mess, outgoingState):
      if debug: print "MifDest:output(mess=%s, outgoingState=%s)" \
         % (mess, outgoingState)
      state = outgoingState.copy()
      tmpFilename = os.tempnam("/tmp", "hdst_")
      utility.writeFile(tmpFilename, mess.asString())
      state['fn'] = tmpFilename
      
      cmd = self.formatStr % state
      startup.log.invoking(cmd)
      status,output = commands.getstatusoutput(cmd)
      if status != 0:
         raise IOError, """Command {%s} returned value %d; output was:
%s
""" % (cmd,status,output) 

      
   def __repr__(self):
      return "<MifDest: %s>" % self.formatStr


#---------------------------------------------------------------------

"""***
Note: There is a problem regarding this class.
The output filename created depends on the number of previous
final-messages output for the message being processed. (For
example, a plain-text message Cc:'d to 3 people will produce
3 final output messages). This means that the code in outgoing.py
needs to be stateful; so perhaps it should be a class, not a
series of functions.

output(self, mess, outgoingState)
   mess = an instance of mailheader.Mail
   outgoingState = a dictionary, containing keys:
      "ix": value is an integer giving the index in the output
         messages, 1 of the 1st one, etc.   
      "recip": the recipient's email address   

***"""

class FileDest(Destination):
   typeId = "file"

   def __init__(self, param):
      self.fileFormatStr = param
      
   def output(self, mess, outgoingState):
      filename = self.fileFormatStr % outgoingState
      utility.writeFile(filename, mess.asString())
    
   def __repr__(self):
      return "<FileDest: %s>" % self.fileFormatStr
     

#---------------------------------------------------------------------

class SendmailDest(Destination):
   typeId = "sendmail"
   
   def __init__(self, param):
      self.executable = param
    
   def output(self, mess, outgoingState):
      tmpFilename = os.tempnam("/tmp", "hsm_")
      utility.writeFile(tmpFilename, mess.asString())
      recipAddr = outgoingState['recip']
      cmd = "%s %s <%s" % (self.executable, recipAddr, tmpFilename)
      
      startup.log.invoking(cmd)
      status,output = commands.getstatusoutput(cmd)
      if status != 0:
         raise IOError, """Command {%s} returned value %d; output was:
%s
""" % (cmd,status,output) 
  
   def __repr__(self):
      return "<SendmailDest: %s>" % self.executable 

#---------------------------------------------------------------------

"""***
Create a Destination object for (destStr), and return it.
***"""

concreteDestinationClasses = (
   MifDest,
   FileDest,
   SendmailDest)

def makeDestination(destStr):
   destSplit = string.split(destStr, ":", 1)   
   type = string.strip(destSplit[0])
   param = string.strip(destSplit[1])
   
   destClass = None
   for dc in concreteDestinationClasses:
      if debug: print "makeDestination(), dc=%s" % dc
      if type == dc.typeId:
         destClass = dc
         break
   if destClass == None:
      raise Error, "destination type '%s' is illegal" % type
      
   return destClass(param)      
   

#---------------------------------------------------------------------

#end destination.py
