# incoming.py = deal with incoming email
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 27-Jan-2002
# History:
# 1-Jun-2001 PH: created
# 9-Aug-2001 PH: copy the values from the plaintext-header into
# the real header of the decrypted incoming email.
# 24-Jan-2002: added logging

"""***
Notes here about the incoming.py module...

***"""

#***** python standard libraries:
import sys
import os
import os.path
import string
import copy

#***** PH libraries
import utility
import mailheader

#***** part of Herbrip:
from herb_globals import *
import startup
import encwrap

debug = 0 # debugging this module?
superDebug = 0 # even more debugging?

#---------------------------------------------------------------------
"""***
process incoming email, reading from filename (fnEmailIn) and
writing plaintext to filename (fnEmailPt).
***"""

def processIn(fnEmailIn, fnEmailPt):
   # get a file handle for the incoming mail
   try:
      mailIn = open(fnEmailIn, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting"\
         % fnEmailIn
      sys.exit(1)  
   
   # read the header lines
   inm = mailheader.Mail()
   inm.readFromFileHandle(mailIn)
   startup.log.message("i", inm)
   if superDebug: 
      cmd = "cp %s %s" % (fnEmailIn, 
         os.path.join(startup.getHerbDir(), "mess_in"))
      os.system(cmd)
   
   processedMail, enc = processInMail(inm, fnEmailIn)
   if enc: startup.log.message("ip", processedMail)
 
   # output result
   utility.writeFile(fnEmailPt, processedMail.asString())
   if superDebug:
      utility.writeFile(os.path.join(startup.getHerbDir(), "mess_inp"),
         processedMail.asString())
      
#---------------------------------------------------------------------

"""***
returns a tuple consisting of:
* a mailheader.Mail object containing the result
* a boolean (0 or 1) saying whether the incoming email was 
  herbivore-encrypted
***"""

def processInMail(inm, fnEmailIn):
   enc = 0
   
   xHerbivore = inm.header.getheader(X_HV)
   if xHerbivore == None:
      print "herbrip: Incoming email <%s> has no '%s:' header, exiting" \
         % (fnEmailIn, X_HV)
      sys.exit(0)
   
   xHerbivoreVersion = inm.header.getheader(X_HV_VERSION)
   # for now, ignore version. Later we will check it for backwards
   # compatibility
   
   if debug: print "** xHerbivore: %s" % xHerbivore
   if debug: print "** xHerbivore-Version: %s" % xHerbivoreVersion
   
   xHerbivoreKey = inm.header.getheader(X_HV_KEY)
   if debug: print """--- Herbivore Key ---
%s
--- end Herbivore Key ---""" % xHerbivoreKey

   #get email address, because we need to store that with the key
   fromName,fromAddr = inm.header.getaddr("From")
   if debug: print "** from \"%s\" <%s>" % (fromName, fromAddr)
   
   dictAdd(fromAddr, fromName, xHerbivoreKey)

   result = mailheader.Mail()
   result.header = copy.deepcopy(inm.header) 
   
   # decode it if we need to
   if xHerbivore == X_HV_ENCRYPTED:
      enc = 1
      decryptForInput(inm, result)
   else:
      result.body.readFromString(inm.body.asString())
      
   return (result, enc)  

#---------------------------------------------------------------------

"""***
Decrypt, and process headers-in-decrypted-plaintext.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) the unprocessed message
mResult (a mailheader.Mail) the same message, when processed.

On input (mResult) has a copy of (m)'s header.

***"""

def decryptForInput(m, mResult):
   myPrivateKey = startup.secretKeys['private']
   plainText = encwrap.pkDec(myPrivateKey, m.body.asString())
      
   # extract the "headers" from (plainText):
   ptAsMailMessage = mailheader.Mail()
   ptAsMailMessage.readFromString(plainText)
   pseudoHeader = ptAsMailMessage.header
   
   if debug:
      plainTextArr = string.split(plainText, "\n")
      comment = "start of plaintext is --\n" \
         + string.join(plainTextArr[:12],"\n") + "\n"
      startup.log.comment(comment)
      comment = "encrypted-header when decrypted is --\n" \
         + pseudoHeader.asString()
      startup.log.comment(comment)
         
   # transfer all header fields in the pt-as-header to the real header
   for f in pseudoHeader.getheadernames():
      v = pseudoHeader.getheader(f, "")
      mResult.header.put(f, v)
      if debug: print "*** adding header %s: %s" % (f,v)
     
   # (plainText) is just the message body, so we must add
   # the message headers to it:

   mResult.body = ptAsMailMessage.body

#---------------------------------------------------------------------

"""***
Add the sender's public key to our dictionary of public keys.
***"""

def dictAdd(fromAddr, fromName, xHerbivoreKey):
   if debug and 0:
      print ("** dictAdd(fromAddr='%s', fromName='%s', ...)" 
         % (fromAddr, fromName))

   startup.publicKeys.addRecord(fromAddr, fromName, xHerbivoreKey)
   startup.publicKeys.save()
   

#---------------------------------------------------------------------

#end incoming.py
