# logging.py = herbrip logging (for debugging etc)
#
###############################################################
# Copyright (c) 2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 6-Feb-2002
# History:
# 23-Jan-2002 PH: created
# 29-Jan-2002 PH: extending selectivity of medssage logging,
#   in Log::message()

"""***
This module enables Herbrip logging. Log files are written to
<~/.herbivore/event.log> and are intended to be useful for a user to
debug their Herbivore setup.
***"""

#***** python standard libraries:
import sys
import string
import time

#***** PH library:
import mailheader

#***** part of Herbrip:

debug = 0 # debugging this module?

#---------------------------------------------------------------------

"""***
Log is the base class for logging

Methods
=======

message(type,mess)
   type = a string denoting what type of message it is
   mess = the message itself. Either a string or a mailheader.Mail
  

***"""

logAllItems = ("o", "op", "i", "ip", "cmd", "comment")
logAllMessItems = ("all")

class Log:
   def __init__(self):
      self.logItems = logAllItems
      self.messItems = logAllMessItems
      
   def whatToLog(self, itemsToLog, partsOfMessageToLog):  
      self.logItems = itemsToLog
      self.messItems = partsOfMessageToLog

   def write(self, text): 
      """(implemented by subclass)"""  
      
   def writeln(self, text): 
      self.write(text + "\n")

   def invoked(self, argv):
      """a herbrip program was invoked"""
      tup = time.localtime(time.time())
      timestamp = time.strftime("%a %Y-%b-%d %H:%M:%S", tup)
      cmd = string.join(argv, " ")
      text = "\n!!!!!!!!!! %s, Herbrip was invoked:\n%s\n" % (timestamp, cmd)
      self.write(text)
   
   def message(self, type, mess):
      type = string.lower(type)
      if type not in self.logItems:
         # we aren't logging this type of message
         return
      t = "mail"
      if type == "i": t = "INCOMING mail"
      if type == "ip": t = "INCOMING mail, processed result"
      if type == "o": t = "OUTGOING mail"
      if type == "op": t = "OUTGOING mail, processed result"
      top = "########## " + t + " "
      top2 = top + ("#"*(70 - len(top)))
      text = top2 + "\n"
      
      # use (self.messItems) to determine which parts of the message
      # we are logging
      if "all" in self.messItems:
         # log everything
         text += str(mess)
         
      else: 
         loggedAnyHeaders = 0
      
         # not logging everything, make sure (mess) is a Mail.
         if not isinstance(mess, mailheader.Mail):
            mess = mailheader.makeMailFromString(str(mess))
            
         loggingAllHeader = ("head" in self.messItems)
         if loggingAllHeader:
            loggedAnyHeaders = 1
            text += mess.header.asString()
         else:
            mh = mess.header
            for h in self.messItems:
               hValue = mh.get(h)
               if hValue != None:
                  loggedAnyHeaders = 1
                  text += "%s: %s\n" % (h, hValue)
            #//for h  
            
         if "body" in self.messItems:
            if loggedAnyHeaders: text += "\n"
            text += mess.body.asString()       
   
      
      self.write(text)
   
   def invoking(self, cmd):
      if "cmd" in self.logItems:
         self.writeln("CMD { %s }" % cmd)
   
   def comment(self, text):
      """a comment"""
      if "comment" in self.logItems:
         self.writeln("# " + text)
   

#---------------------------------------------------------------------

class NullLog(Log):
   """ an example of the Null Object design pattern """
   def write(self, text): pass
   
   # these aren't really necessary, but add them too:
   def invoked(self, argv): pass
   def message(self, type, mess): pass
   def invoking(self, cmd): pass
   def comment(self, text): pass


class FileLog(Log):
   def __init__(self, filename):
      self.filename = filename
      
   def write(self, text):
      f = open(self.filename, "a+")
      f.write(text)
      f.close()   
      
class PrintLog(Log):
   def write(self, text):
      print text,
     

#---------------------------------------------------------------------
# testing this module:

msg = """From: philh@comuno.freeserve.co.uk
To: alice@nowhere.org
Subject: hello, world

Hello, world!!

-- 
phil hunt
"""

def testme():
   print "***** testing logging.py *****"
   log = PrintLog()
   log.comment("this is a comment")
   log.invoked(sys.argv)
   log.comment("comment2")
   log.invoking("cp hello goodbye")
   log.message("I", msg)
   msgMail = mailheader.makeMailFromString(msg)
   msgMail.header.put("X-Herbivore-Proc", "yes")
   log.message("OP", msgMail)
   log.comment("comment3")

#---------------------------------------------------------------------

if __name__ == "__main__": 
   testme()

#end logging.py
