# encwrap.py = wrapper around the encryption engine
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 24-Jan-2001
# History:
# 8-Jul-2001 PH: created
#
# 24-Jan-2002: added logging of commands to Herbrip log file

"""***
encwrap.py is a wrapper around the encryption engine.

The engine used is OpenSSL, version 0.9.6a. Herbrip runs the
openssl executable with various parameters, using system()
and equivalent functions.

NOTE 10-Jul-2001: none of the openssl calls includes
generation of randomness. This must be fixed later. Randomness
can be got by calls to ``ps auxw'', ``date'', and ``df'', among 
other things.

***"""


#***** python standard libraries:
import sys, os, string, commands

#***** Phil's libraries:
import utility

#***** part of Herbrip:
from herb_globals import *
import startup

debug = 0  # debugging this module?

#***** constants for openssl encryption

BLOCK_BEGIN_PUBLIC = "-----BEGIN PUBLIC KEY-----"
BLOCK_END_PUBLIC = "-----END PUBLIC KEY-----"

# the name for openssl on the command line
#OPENSSL_CL = "openssl"
OPENSSL_CL = "/usr/local/ssl/bin/openssl"

#---------------------------------------------------------------------
# internal functions.
#
# These are intended for internal use only

#listg of temporary files named by _tmp()
_tempFiles = [ ]

def _tmp(filenameIfDebugging=""):
   """return a temporary filename"""
   global _tempFiles   
   if debug and filenameIfDebugging != "":
      fn = "tmp_" + filenameIfDebugging
   else:
      fn = os.tempnam("/tmp", "henc_")
      _tempFiles.append(fn)
   return fn
   
def _rm(fn):
   """delete file (fn)"""
   os.remove(fn)
   
def _rmAll():
   """delete all temporary files named with _tmp() """
   global _tempFiles
   for fn in _tempFiles:
      _rm(fn)
   _tempFiles = [ ]
   
def _randomFile():
   """return the filename of a temporary file containing data for 
   input to a randomness-consumer"""
   
   fn = _tmp()
   out1 = commands.getoutput("df") + "\n"
   out2 = commands.getoutput("ps auxw") + "\n"
   out3 = commands.getoutput("date") + "\n"
   
   # get some stuff from /dev/urandom
   fh = open("/dev/urandom", "r")
   out4 = fh.read(32) + "\n" # read 32 bytes
   fh.close()   
         
   utility.writeFile(fn, out1+out2+out3+out4) 
   return fn  
 
# a log of command-line commands, for debugging:   
_logCommands = [ ]  
   
def _system(cmd):
   """run (cmd) at the command line"""
   global _logCommands
   if debug: print "CMD { %s }" % cmd
   if debug: _logCommands.append(cmd)
   startup.log.invoking(cmd)
   status,output = commands.getstatusoutput(cmd)
   if status != 0:
      raise IOError, """***** Command {%s} returned value %d; output was:
%s
*****(end)""" % (cmd,status,output)  

def _openssl(paramStr):
   """run the openssl program"""
   _system("%s %s" % (OPENSSL_CL, paramStr))
   
   
#---------------------------------------------------------------------

"""***
Make a public/private keypair.

Returns a tuple (pubkey,privkey) where both values are
Base64-encoded strings.

Both these are multi-line Base64 encoded strings that
are enclosed in header/footer.
***"""

def makeKeypair():
   keypairFn = _tmp("m_keypair") # keypair (Public+Secret keys)
   pubFn = _tmp("m_pub") # public key
   randomnessFn = _randomFile() # file full of randomness
   
   p = "genrsa -out %s -rand %s 1024" % (keypairFn, randomnessFn)
   _openssl(p)
   keypairStr = utility.readFile(keypairFn)
   #keypairStr = stripHF(keypairStr)
   
   p = "rsa -in %s -out %s -pubout" % \
      (keypairFn, pubFn)
   _openssl(p)
   pubStr = utility.readFile(pubFn)
      
   # zap all temporary files:   
   _rmAll()
   
   return (pubStr, keypairStr)


#---------------------------------------------------------------------

"""***
Perform public-key encryption on a message, returning
a base64-encoded string, encased in Herbivore header/footer.
***"""

def pkEnc(publicKey, plainText):
   if debug: print "** pkEnc([%s], [%s])" % (publicKey, plainText)

   # store the public key and plain text as temporary files
   pubKeyFn = _tmp("e_pubKey")
   normalizedPublicKey = normalizePublicKey(publicKey)
   utility.writeFile(pubKeyFn, normalizedPublicKey)
   plainTextFn = _tmp("e_plainText")
   utility.writeFile(plainTextFn, plainText)

   # get a random 128-bit session key, encoded as 
   # a 16-byte file
   sessionKeyFn = _tmp("e_sessionKey")
   randomnessFn = _randomFile() # file full of randomness
   cmd = "rand -rand %s 16 >%s" % (randomnessFn, sessionKeyFn)
   _openssl(cmd)
   
   # encode the session key, using the public key. The
   # result is a binary file, stored in file
   # (sessionKeyEncFn).
   sessionKeyEncFn = _tmp("e_sessionKeyEnc")
   cmd = "rsautl -in %s -out %s -inkey %s -pubin -encrypt"\
      % (sessionKeyFn, sessionKeyEncFn, pubKeyFn)
   _openssl(cmd)
   
   # get a base64 encoded version of the encoded session key
   sessionKeyEnc64Fn = _tmp("e_sessEnc64")
   cmd = "base64 -in %s -out %s"\
      % (sessionKeyEncFn, sessionKeyEnc64Fn)
   _openssl(cmd)
   
   # using the session key, encode the plain text file
   cipherTextFn = _tmp("e_cipherText")
   sessionKeyHex = file2hexStr(sessionKeyFn)
   cmd = "bf -in %s -out %s -e -a -K %s"\
      % (plainTextFn, cipherTextFn, sessionKeyHex)
   _openssl(cmd)
   
   result = ( BEGIN_HERBIVORE + "\n"
              + utility.readFile(sessionKeyEnc64Fn)
              + HERBIVORE_SEP + "\n"
              + utility.readFile(cipherTextFn)
              + END_HERBIVORE + "\n" )
      
   # zap all temporary files:
   _rmAll()
   
   return result


#---------------------------------------------------------------------

"""***
Perform the opposite of pkEnc(); pkDec() decrypts a message
encoded with pkEnc(), returning the same plainText that was the
argument to pkEnc().

cipherText = a string containing the encrypted message, including
   the herbivore starting/trailing blocks.
   
privateKey = the RSA private key, including openSSL start/trail 
   blocks.   

***"""

def pkDec(privateKey, cipherText):

   # split cipherText into encrypted session key, and package   
   encSess, package = splitHerbEnc(cipherText)
   
   # write the encrypted session key to a file
   encSessBase64Fn = _tmp("d_encSessBase64")
   utility.writeFile(encSessBase64Fn, encSess)
   
   # convert to binary (i.e. base64-decode it)
   encSessFn = _tmp("d_encSess")
   cmd = "base64 -in %s -out %s -d"\
      % (encSessBase64Fn, encSessFn)
   _openssl(cmd)
   
   # get the decrypted session key
   privateKeyFn = _tmp("d_privateKey")
   utility.writeFile(privateKeyFn, privateKey)
   ptSessFn = _tmp("d_ptSess") 
   cmd = "rsautl -in %s -out %s -inkey %s -decrypt"\
      % (encSessFn, ptSessFn, privateKeyFn)
   _openssl(cmd)
   
   # (ptSessFn) is a binary file, and must be converted to
   # a hex string:
   sessionKeyHex = file2hexStr(ptSessFn)
   
   # use the session key to decode the package
   packageFn = _tmp("d_package")
   utility.writeFile(packageFn, package)
   plainTextFn = _tmp("d_plainText")
   cmd = "bf -in %s -out %s -d -a -K %s"\
      % (packageFn, plainTextFn, sessionKeyHex)
   _openssl(cmd)
   
   result = utility.readFile(plainTextFn)
   
   # delete temporary files:
   _rmAll()
   
   return result
      

#---------------------------------------------------------------------

"""***
Read in a binary file. Output the data as a hex string
***"""

def file2hexStr(filename, size=-1):
   data = utility.readFile(filename)
   if debug: print "in file2hexStr(), len(data)=%d" % len(data)
   result = ""
   bytesWritten = 0
   for c in data:
      #if bytesWritten == size: break
      result += "%02x" % ord(c)
      bytesWritten += 1
   if debug: print "in file2hexStr(), bytesWritten=%d" % bytesWritten
   return result

#---------------------------------------------------------------------

"""***
Remove header and footer lines from an ascii string.

A header line begins with "-----BEGIN", and a footer line
begins with "-----END".

***"""

def stripHF(str):
   sa = string.split(str, "\n")
   #print sa
   
   #remove header
   if utility.startsWith(sa[0], "-----BEGIN"):
      sa = sa[1:]
      
   #remove footer
   while sa[len(sa)-1] == "":
      sa = sa[:-1]
      
   if utility.startsWith(sa[len(sa)-1], "-----END"):
      sa = sa[:-1]
   
   str2 = string.join(sa, "\n")
   
   return str2

#---------------------------------------------------------------------

"""***
Normalize a public key.

Takes as input a string representing a public key.
Returns a normalized string representing the same public
key. To normalize:

(1) starting/trailing whitespace removed from each line

(2) blank lines removed

(3) if it doesn't already exist, a line "-----BEGIN PUBLIC KEY-----"
is added at the start

(4) if it doesn't already exist, a line "-----END PUBLIC KEY-----"
is added at the end

***"""

def normalizePublicKey(pkStr):
   sa = string.split(pkStr, "\n")
   sa2 = []
   for s in sa:
      s2 = string.strip(s)
      sa2.append(s2)
      
   if len(sa2) >= 1:
      while sa2[0] == "": sa2 = sa2[1:]
   if len(sa2) >= 1:
      while sa2[-1] == "": sa2 = sa2[:-1]   

   if len(sa2) == 0:
      raise Error, "there's no key here!!!"
      
   if sa2[0] != BLOCK_BEGIN_PUBLIC:
      sa2 = [BLOCK_BEGIN_PUBLIC] + sa2
      
   if sa2[-1] != BLOCK_END_PUBLIC:
      sa2 = sa2 + [BLOCK_END_PUBLIC] 
      
   result = string.join(sa2, "\n") + "\n"
   if debug: 
      print "---normalize---\n%s\n---result---\n%s\n---end---"\
         % (pkStr, result)
   return result   
      
      

#---------------------------------------------------------------------

"""***
Split a herbivore-encrypted message into the session key and
package. Note that a Herbivore-encrypted message consists of:

   BEGIN_HERBIVORE 
   + sessionKey
   + HERBIVORE_SEP
   + package
   + END_HERBIVORE

(with '\n' between each part)   
***"""

def splitHerbEnc(message):
   messa = string.split(message, "\n")
   
   locBegin = utility.index(messa, BEGIN_HERBIVORE)
   locSep = utility.indexp(messa, HERBIVORE_SEP, locBegin+1)
   locEnd = utility.indexp(messa, END_HERBIVORE, locSep+1)
   
   sessa = messa[locBegin+1:locSep]
   sessionKey = string.join(sessa, "\n") + "\n"  
   packa = messa[locSep+1:locEnd]
   
   # note that the file that the package goes into nmust end
   # with a newline, so we put one in here
   package = string.join(packa, "\n") + "\n"
   return (sessionKey, package)

#---------------------------------------------------------------------
# testing this module

def testMe():
   pub, priv = makeKeypair()
   print "public: ", pub
   #print "private: ", priv
   
def testPkEnc():
   pub, priv = makeKeypair()
   plainText = "this is a plain text message"
   enc = pkEnc(pub, plainText)
   print "enc: ", enc
   
def testEncDec():
   "test encryption and decryption"
   pub, priv = makeKeypair()
   plainText = "*"*50
   enc = pkEnc(pub, plainText)
   print ">>>>>enc: ", enc
   dec = pkDec(priv, enc)
   print ">>>>>dec: ", dec
   print ">>>>>commands:"
   i = 1
   for cmd in _logCommands: 
      print "(%d) %s" % (i, cmd)
      i += 1

def testRandomFile():
   "test the creation of a file containing data for randomness"
   fn = _randomFile()
   print "Name of random file: %s" % fn
   print "---------------------------------"
   print utility.readFile(fn)  

if __name__ == "__main__":
   testMe()
   #testPkEnc()
   #testEncDec()
   #testRandomFile()
   
   
#---------------------------------------------------------------------


#end encwrap.py
