# mailheader.py = process RFC822 mail (header and body)
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 6-Feb-2002
# History:
# 10-Jun-2001 PH: created
# 7-Aug-2001 PH: added Mail, MailBody classes. Now this module
# processes the whole email, not just the header, so perhaps the name
# is inappropriate!

"""***


***"""

#***** python standard libraries:
import rfc822   # understands mail headers
import sys
import string
import StringIO

#***** PH libraries:
import utility

debug = 0  # debugging this module?


#---------------------------------------------------------------------
"""***
Utility function: return a string containing an email address and real
name, in RFC2822 format, given separate strings containing them.
***"""

def calcNameAddr(realName, mailAddr):
   if realName:
      result = "%s (%s)" % (mailAddr, realName)
   else:
      result = mailAddr  
   return result

#---------------------------------------------------------------------

"""***
The MailHeader class stores headers for a rfc822 email message

Creation:
~~~~~~~~~
readFromFileHandle(fh)
readFromString(str)
   read the header lines from a file or string
 
Accessing:
~~~~~~~~~~
getheader(name, default)
get(name, default)
   These do the same thing: return the value of a header line.
   if no (default) is specified and the line doesn't exist, 
   returns None.
   
getheadernames()
   Return a list of all the header names, e.g.
   ['Subject','To','From','Date'] etc.   
   
removeheader(name)
   Remove a header; if it doesn't exist, do nothing   
   
getaddrlist(name)
   Returns a list of all the email addresses for header line (name).   
   
getaddr(name)
   Returns just one address, as a real-name, emailaddress pair.   
   
put(name, value)
   Sets the value of header-line (name) to (value).

Writing output
~~~~~~~~~~~~~~
__str__()
asString()
   These do the same thing: Output all the header-lines as a 
   string

Instance variables:
~~~~~~~~~~~~~~~~~~~
self.h = dictionary containing header lines. If there are >1
   lines with a particular header, the value will be a list of all
   the lines, in the order they appeared
   
#self.mess = an instance of rfc822.Message, used internally to
#   decode the header messages
***"""


class MailHeader:

   def __init__(self):  
      self.h = {}
   
   def readFromFileHandle(self, fh):
      # get a message object for parsing:
      mess = rfc822.Message(fh)
      headerNames = mess.keys()
      for hn in headerNames:
         hValue = mess.getheader(hn)
         self.putAppend(hn, hValue)
           
   def readFromString(self, str):      
      stringToFileWrapper = StringIO.StringIO(str)
      self.readFromFileHandle(stringToFileWrapper)
      stringToFileWrapper.close()
   
   def getheader(self, name, default =None):
      return self.get(name, default)  
      
   def getheadernames(self):
      return map(utility.upperFirstLetter, self.h.keys())   
      
   def get(self, name, default =None):
      if debug: print "@@@in MailHeader:get(), self.h = %s" % self.h
      
      # Note that rfc822 header names are not case-sensitive,
      # and that we are storing them internally in lower case
      name = string.lower(name) 
      
      try:
         value = self.h[name]
         return self._valueMultiline(value)
      except:
         # couldn't return a value, use default
         return default
         
   def removeheader(self, name):
      name = string.lower(name)
      if self.h.has_key(name): 
         del self.h[name]
                  
   def getaddrlist(self, name):
      headerText = self.getheader(name)
      addressParser = rfc822.AddressList(headerText)
      return  addressParser.addresslist
      
   def getaddr(self, name):
      list = self.getaddrlist(name)
      if len(list) == 0: return ('', '')
      return list[0]   
  
   def put(self, name, value):
      # Note that rfc822 header names are not case-sensitive,
      # and that we are storing them internally in lower case
      name = string.lower(name)        
      self.h[name] = value
      
   def putAppend(self, name, value):
      if self.h.has_key(name):
         v = self.h[name]
         if type(v)==type([]):
            if debug: print "@@@ self.h[%s] was %s @@@" % (name,self.h[name])
            v.append(value)
            if debug: print "@@@ self.h[%s] is %s @@@" % (name,self.h[name])
            # not sure if we need to add anything here???
         else:
            # it's a string, change to a list:
            self.h[name] = [self.h[name], value] 
      else:
         self.h[name] = value
   
   def __str__(self):                               
      return self.asString()

   def asString(self):
      result = ""
      for name, value in self.h.items():
         name = utility.upperFirstLetter(name)
         if type(value)==type([]):
            for v in value:
               result = result + name + ": " \
                  + self._valueMultiline(v) + "\n"
         else:
            result = result + name + ": " \
               + self._valueMultiline(value) + "\n"
      return result
            
   def _valueMultiline(self, v):
      """*** 
      if (v) has more than one line, ensure that lines
      after the first one begin with a space
      ***"""
      sa = string.split(v, '\n')
      for i in range(len(sa)):
         if i == 0: continue
         if not utility.startsWith(sa[i], " "):
            sa[i] = " " + sa[i]
      v = string.join(sa, '\n')    
      return v
   
   
#---------------------------------------------------------------------
   
"""***
The MailBody class stores the body of an RFC822 mail message.


Creation:
~~~~~~~~~
readFromFileHandle(fh)
readFromString(str)
   read the mail message from a file or string
 
Writing output
~~~~~~~~~~~~~~
__str__()
asString()
   These do the same thing: Output all the header-lines as a 
   string
   
Instance Variables
~~~~~~~~~~~~~~~~~~
data = the body (a string)

***"""

class MailBody:

   def __init__(self):
      self.data = ""
  
   def readFromFileHandle(self, fh):
      self.data = fh.read()
           
   def readFromString(self, str):      
      self.data = str

   def __str__(self):                               
      return self.asString()

   def asString(self):
      return self.data
 
#---------------------------------------------------------------------

"""***
The Mail class stores both the header and the body of an RFC822
mail message.

Creation:
~~~~~~~~~
readFromFileHandle(fh)
readFromString(str)
   read the mail message from a file or string
 
Writing output
~~~~~~~~~~~~~~
__str__()
asString()
   These do the same thing: Output all the header-lines as a 
   string

Instance Variables
~~~~~~~~~~~~~~~~~~
header = the header (a MailHeader)
body = the body (a MailBody)

***"""

class Mail:

   def __init__(self):
      self.header = MailHeader()
      self.body = MailBody()

   def readFromFileHandle(self, fh):
      self.header.readFromFileHandle(fh)
      self.body.readFromFileHandle(fh)
           
   def readFromString(self, str):      
      stringToFileWrapper = StringIO.StringIO(str)
      self.readFromFileHandle(stringToFileWrapper)
      stringToFileWrapper.close()

   def __str__(self):                               
      return self.asString()

   def asString(self):
      result = self.header.asString() + "\n" + self.body.asString()
      return result
 
def makeMailFromString(str):
   m = Mail()
   m.readFromString(str)
   return m
   
def makeMailFromFileHandle(fh): 
   m = Mail()
   m.readFromFileHandle(fh)
   return m



#---------------------------------------------------------------------
# test the classes:


test_mailHead = """From: hpilh@sdfsf.sdfsadfasfd.com
To: another@aol.com (A N Other)
To: fred@here.com
To: jim@there.com
Subject: noyb
"""

test_mailBody = """the first line
the second line
the last line with no return"""

test_mail = """Subject: this is a boring test
To: alice@somewhere.com
From: bob@nowhere.com

Hi alice,
how are you?

-- 
Bob"""

def testMailHeader():
   print "Testing MailHeader class..."
   s = test_mailHead
   mh = MailHeader()
   mh.readFromString(s)
   print "--------result: >>>>%s<<<<" % mh.asString()
   print "--------internal:"   
   print mh.h
   
   # add some headers:
   mh.put("X-Herbivore", "plain")
   mh.put("X-Herbivore-Version", "0.1")
   print "--------result: >>>>%s<<<<" % mh.asString()


def testMailBody():
   print "Testing MailBody class..."
   mb = MailBody()
   mb.readFromString(test_mailBody)
   print "--------result: >>>>%s<<<<" % mb.asString()
   print "--------internal:" 
   print mb.data 
   
   
   
def testMail():
   print "Testing Mail class..."
   m = Mail()
   m.readFromString(test_mail)
   print "[----- mail header, m.header.h = \n%s\n-----]" % m.header.h
   print "[----- mail body, m.body.data = \n%s\n-----]" % m.body.data
   print "(m) is ====>>%s<<====" % (m)
   



if __name__ == "__main__":
   #testMailHeader()
   #testMailBody()
   testMail()

#---------------------------------------------------------------------


#end mailheader.py
