# utility.py
# 
# Copyright (c) 1999-2001 Philip Hunt
#
# Various python utility functions

# Last altered: 28-Oct-2001
#
# History:
# 28-Apr-1999 PhilHunt: created
# 11-May-1999 PH: added savePickle(), loadPickle() functions
# 2-Jun-2001: renamed to utility.py
# 8-Jul-2001: added outPrettyPickle(), prettyStrPickle()
# 8-Aug-2001: added upperFirstLetter()
#
# 14-Oct-2001: re-ordered, put into pylib, added forceDir()

import string, os, stat, pprint, sys, re
from cPickle import *

debug = 0 # debugging this module?

#===========================================================
#  String searching / matching functions
#===========================================================

# Return a string like (s) except that the first character in 
# every word is capitalised. A word is defined as a maximal contiguous
# block of alphabetical characters.
   
def upperFirstLetter(s):
   result = ""
   inWord = 0
   for ch in s:
      wordChar = ch in string.letters
      if wordChar and not inWord: ch = string.upper(ch)
      inWord = wordChar
      result += ch
   return result

#-----------------------------------------------------------
# return 1st index of character (c) in string (s)
# or -1 if no match
#
# Note: works for strings, lists and tuples

def strIndex(s, c):
   for i in xrange(0,len(s)):
      if s[i] == c: return i
   # no match:
   return -1
   
# better name of non-string version:
index = strIndex   

#-----------------------------------------------------------
# return 1st index of character (c) in string (s), starting
# at character position (p).
# Returns -1 if no match
#
# Note: works for strings, lists and tuples

def strIndexp(s, c, p):
   for i in xrange(p,len(s)):
      if s[i] == c: return i
   # no match:
   return -1
   
# better name of non-string version:
indexp = strIndexp   

#-----------------------------------------------------------
# return 1st index of substring (ss) in string (s)
# or -1 if no match
#
# Note: works for strings, lists and tuples

def strIndexStr(s, ss):
   return strIndexStrp(s, ss, 0)

#-----------------------------------------------------------
# return 1st index of substring (ss) in string (s), starting
# at position (p) in (s).
# Returns -1 if no match
#
# Note: works for strings, lists and tuples

def strIndexStrp(s, ss, p):
   ls = len(s)
   lss = len(ss)
   if ls == 0 or lss == 0 or lss>ls: return 0
   for i in xrange(p,ls-lss+1):
      for j in xrange(0,lss):
         if s[i+j] != ss[j]: break
      else:
         #got match so return it
         return 
   # no match:
   return -1

#-----------------------------------------------------------
# Does string (s) start with string (sw)

def startsWith(s, sw):
   return (s.find(sw)==0)


#===========================================================
#  Filesystem functions
#===========================================================

# does a directory exist?
#
# returns a value of 1 if the directory exists and is readable
# returns 0 otherwise

def dirExists(dirname):
   dirname = os.path.expanduser(dirname)
   readable = os.access(dirname, os.R_OK)
   # (if it doesn't exist, it can't be readable, so don't bother
   # testing that separately)
   
   if not readable: return 0
   
   # now test if it's a directory
   dr = isDir(dirname)
   return dr

#-----------------------------------------------------------
# is string (s) a directory?

def isDir(s):
   s = os.path.expanduser(s)
   mode = os.stat(s)[stat.ST_MODE]
   return stat.S_ISDIR(mode)
   
#-----------------------------------------------------------
# Does a file exist?
#
# Returns 1 if (fn) is the filename of an existing file and it 
# is readable.

def fileExists(fn):
   fn = os.path.expanduser(fn)
   readable = os.access(fn, os.R_OK)
   # (if it doesn't exist, it can't be readable, so don't bother
   # testing that separately)
   
   if not readable: return 0

   # now test if it's a file
   mode = os.stat(fn)[stat.ST_MODE]
   return stat.S_ISREG(mode)


#-----------------------------------------------------------
# ensure that a directory exists
#
# make the directory (dirname) exist, creating it and intermediate
# directories if necessary.
# Exit with the value 1 if it could be created, 0 if it couldn't.
# Exit with value 1 if there is no directory needing to be created,
# e.g. because dirname is '' or '.'


def forceDir(dirname):
   if debug: print ".. forceDir('%s') (%s)" \
      % (dirname, dirExists(dirname))
   dirname = os.path.expanduser(dirname)
   if debug: print "   dirname='%s'" % dirname
   
   if dirname == None or dirname == '' or dirname == '.': 
      if debug: print "   forceDir() exiting, dirname empty"
      return 1
   if not dirExists(dirname):
      if debug: "   forceDir(): calling os.makedirs('%s')" % dirname
      os.makedirs(dirname)
   return dirExists(dirname)


#-----------------------------------------------------------
# read a file into a string

def readFile(filename):
   f = open(filename, 'r')
   s = f.read()
   f.close()
   return s

#-----------------------------------------------------------
# write a string into a file
#
# (filename) can be a full pathname. if it is, then if any
# directories need to be created to create the file, they 
# are created.

def writeFile(filename, newValue):
   # create directories if they don't exist
   dirname = os.path.dirname(filename)
   forceDir(dirname)
   
   f = open(filename, 'w')
   f.write(newValue)
   f.close()



#===========================================================
#  Pickling and Unpickling
#===========================================================

# save a pickled file
#
# TO DO: if (makeTemp) is 1, and the file (filename) already exists,
# makes a temporary file first, writes into that, then used the
# 'mv' command to replace the original file with the temporary one.

def savePickle(filename, object, makeTemp =0, tempFileExtension =".tmp"):
   f = open(filename, 'w')
   p = Pickler(f)
   p.dump(object)
   f.close()

#-----------------------------------------------------------
# load a pickled file into an object
#
# if couldn't read it, return None

def loadPickle(filename):
   try:
      f = open(filename, 'r')
      u = Unpickler(f)
      object = u.load()
      f.close()
   except:
      object = None

   return object

#-----------------------------------------------------------
# read a pickled file and prettyprint it to stdout

def outPrettyPickle(pickleFilename):
   pretty = prettyStrPickle(pickleFilename)
   print "===== prettyprint of '%s': =====" % pickleFilename
   print pretty
   print "===== end. ====="

#-----------------------------------------------------------
# read a pickled file, outputting a string containing its 
# prettyprinted form.
# If couldn't read it, return a string containing the
# exception.

def prettyStrPickle(pickleFilename):
   result = "(something wierd happened)"
   try:
      f = open(pickleFilename, 'r')
      u = Unpickler(f)
      object = u.load()
      f.close()
      pp = pprint.PrettyPrinter(indent=4,depth=4,width=80)
      result = pp.pformat(object) 
   except:
      result = "(Error: " + sys.exc_info()[0] + ")"
   return result



#===========================================================
#  Module testing
#===========================================================

def testFileDir():
   print "### test file and directory functions ###"
   dn = "sub/t/u"
   b = dirExists(dn)
   print "exists '%s'? %s" % (dn, b)
   forceDir(dn)
   b = dirExists(dn)
   print "exists '%s'? %s" % (dn, b)
  
   dn = "sub/v/w"
   b = dirExists(dn)
   print "exists '%s'? %s" % (dn, b)
   fn = "sub/v/w/newFile"
   writeFile(fn, "this is some data")
   b = dirExists(dn)
   print "exists '%s'? %s" % (dn, b)
   data = readFile(fn)
   print "File '%s' contains '%s'" % (fn, data)
   b = dirExists(dn)
   print "exists '%s'? %s" % (dn, b)
   
   

if __name__ == "__main__":
   testFileDir()


#end 
