# initial.py = initialize Herbrip
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 6-Feb-2002
# History:
# 10-Jun-2001 PH: created
#
# 17-Nov-2001 PH: added code to create .herbivore/config file

"""***
The initial.py module is responsible for  initialising the herbivore
environment, which involves creating a directory to hold keys in,
creating a public/private keypair for the user, and creating several
files to contain  keys and other information
***"""

#***** python standard libraries:
import os
import os.path
import popen2
import commands
from time import *

#***** part of Herbrip:
import utility
from herb_globals import *
import startup
import encwrap

debug = 0  # debugging this module?

#---------------------------------------------------------------------

def isInitialized():   
   # ~/.herbivore must exist and be writeable
   if not os.access(startup.getHerbDir(), os.W_OK): return 0
      
   # secret keyring file must exist and be writeable
   if not os.access(startup.getSecretPn(), os.W_OK): return 0
    
   # public keyring file must exist and be writeable 
   if not os.access(startup.getPublicPn(), os.W_OK): return 0
   
   #...other tests here...
    
   # passed all tests, it's OK
   return 1

#---------------------------------------------------------------------

"""***
Forced initialization of a herbivore system. Any existing system in
the target directory is removed.

this creates:
  directory: ~/.herbivore/
in this directory:
  file: <secret_keys> - public/private keys for user
  file: <public_keys> - other people's public keys
  file: <README> - text file saying what the directory's for
***"""

def forceInit(realName, email):
   # zap the directory if it exists:
   cmd = "rm -rf %s" % startup.getHerbDir()
   os.system(cmd)

   # make the herbivore directory, with all permissions for the user,
   # and no permissions for group and world (0700 is an octal number)
   try:
      if not os.access(startup.getHerbDir(), os.W_OK):
         os.mkdir(startup.getHerbDir(), 0700)
   except OSError,e:
      print "!!! in OSError, e=[%s] !!!" % e
         
   createSecretKey(realName, email)
   createReadme()
   createConfig()

#---------------------------------------------------------------------

"""***
Ensure that the herbivore system is initialized, by checking
if it is already, and if not, initilizing it.
***"""
   
def ensureInitialized():
   if not isInitialized():
      initialize("","")
      
    
#---------------------------------------------------------------------

def createSecretKey(realName, email):
   """ invoke OpenSSL to create a secret key for me """  
   
   pub,priv = encwrap.makeKeypair()
   
   startup.secretKeys = {}
   startup.secretKeys['public'] = pub
   startup.secretKeys['private'] = priv
   
   utility.savePickle(startup.getSecretPn(), startup.secretKeys)
   
#---------------------------------------------------------------------

def createReadme():
   timeNow = strftime("%Y-%b-%d %a %H:%M", localtime(time()))
   readmeData = """README file for %s/

Herbrip version: %s
Herbivore version: %s
This file created at: %s

Herbrip is an implementation of the Herbivore protocol, aimed
at hassle-free, automatic, strong encryption of email.
More information about Herbrip and Herbivore can be found at
<http://www.vision25.demon.co.uk/oss/oss.html>.
""" % (
      startup.getHerbDir(),
      HERBRIP_VERSION,
      HERBIVORE_VERSION,
      timeNow
      )
   readmeFile = os.path.join(startup.getHerbDir(), DEFAULT_README_FN)
   if debug: print "README file is %s" % readmeFile
   utility.writeFile(readmeFile, readmeData)   

#---------------------------------------------------------------------

configData = """# Herbrip config file

# destination for outgoing mail
# In use '%%(ix)s' gets replaced by the index of the final-output message
# for this particular unencrypted output message
%(dest)s

# Logging: do we want to log events that happen within the herbrip system?
# 1=Yes, 0=No. (Later there will be more options, allowing the user to
# adjust the detail of logging).
# The log goes into file <event.log> in the .herbivore directory.
%(log)s

# if (log) is 1, this determines what will be logged. values are:
# "i" = incoming messages
# "ip" = incoming messages, processed results
# "o" = outgoing messages
# "op" = outgoing messages, processed results
# "comment" = comments in the code
# "cmd" = sub-processes invoked by 
# (If logging is switched on, Herbrip commands are always logged.)
%(logItems)s

# for messages that we are logging, this determined what parts of
# the message get logged.
# "head" = all headers
# "body" = all the body
# "all" = everything
# other strings refer to individual headers
%(logMessItems)s


#end   
"""

#---------------------------------------------------------------------

def processDefaults(cdf):
   result = {}
   keys = cdf.keys()
   for k in keys:
      v = cdf[k]
      assignment = "%s = %s" % (k, repr(v))
      entry = "##%s\n%s" % (assignment, assignment)
      result[k] = entry
   return result   
     

def createConfig():
   """create the default config file"""
   
   configFile = os.path.join(startup.getHerbDir(), DEFAULT_CONFIG_FN)
   if debug: print "config file is %s" % configFile
   
   configDefaults = processDefaults(CONFIG_DEFAULT_VALUES)
   
   configFileContents = configData % configDefaults   
   utility.writeFile(configFile, configFileContents)   

#---------------------------------------------------------------------


#end initial.py
