# outgoing.py = deal with outgoing email
#
###############################################################
# Copyright (c) 2001,2002 Philip Hunt.
# You are permitted to use this software under the terms of the 
# GNU General Public License. Details are in the file COPYING, 
# which you should have received with this distribution.
###############################################################

# Last altered: 27-Jan-2002
# History:
# 9-Jun-2001 PH: created
# 9-Aug-2001 PH: puts Subject: and some of the other header lines
# in the unencrypted header into a "plaintext-header" which becomes
# part of the encrypted plaintext; this enhances security because 
# an eavesdropper can no longer see the Subject in clear.
#
# 19-Nov-2001 PH: new function processOutDest() handles output where
# the destination is to another program
#
# 24-Jan-2002: added logging

"""***
This module deals with processing outgoing email.

Outgoing email has two things which need doing to it:

(1) add X-Herbivore...: headers to indicate the outgoing email
system is herbivore-aware, and to broadcast the sender's public
key.

(2) Do a lookup on the public keyring to see if there is
a Herbivore key for the recipient. If there is a public key, 
get the relevant headers that need to be encrypted, put them
in a file together with the message body, encrypt them,
and append the result to the message headers as determined by 
this program.

***"""

#***** python standard libraries:
import sys
import os
import os.path
import commands
from string import *

#***** PH library:
import utility
import mailheader

#***** part of Herbrip:
from herb_globals import *
import pubkeydict
import startup
import encwrap
import destination

debug = 0 # debugging this module?
superDebug = 0 # even more debugging


#---------------------------------------------------------------------

# the --out command (which is deprecated):
# process outgoing email, reading from filename (fnEmailOut) and
# writing herbivorized result to filename (fnEmailHerb)

def processOut(fnEmailOut, fnEmailHerb):
   # get a file handle for the outgoing mail
   try:
      mailOut = open(fnEmailOut, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting" % fnEmailOut
      sys.exit(1)   

   # read the header lines
   m = mailheader.Mail()
   m.readFromFileHandle(mailOut)
   startup.log.message("o", m)

      
   # find out who the message is to  
   # for now, assume one recipient. If more than one,
   # use the first. This is an assumption that **MUST**
   # be changed later
   if debug:
      print "*** m.header: %s ***" % m.header
   recipients = m.header.getaddrlist("To")
   numRecipients = len(recipients)
   if numRecipients != 1:
      print "!! Warning: There are %d recipients; there should be exactly 1 !!"\
         % numRecipients
      print "!!***** They are: %s" % recipients 
      
   # get the email address if the initial recipient     
   recipient = recipients[0]  

   if debug: print "outgoing: recipient = %s" % (recipient,)
   mProcessed = processForRecipient(m, recipient)
   startup.log.message("op", mProcessed)
  
   # write result to (fnEmailHerb)
   fhEmailOut = open(fnEmailHerb, "w")
   fhEmailOut.write(mProcessed.asString())
   fhEmailOut.close()
   

#---------------------------------------------------------------------

# the --outd command:
# process outgoing email, send results to somewhere defined by the
# 'dest' config variable.

def processOutDest(fnEmailOut):
   # get a file handle for the outgoing mail
   try:
      mailOut = open(fnEmailOut, "r")
   except:
      print "herbrip: cannot open email file '%s', aborting" % fnEmailOut
      sys.exit(1)   

   # read the mail into a Mail object
   m = mailheader.Mail()
   m.readFromFileHandle(mailOut)
   if superDebug: 
      cmd = "cp %s %s" % (fnEmailOut, 
         os.path.join(startup.getHerbDir(), "mess_out"))         
      os.system(cmd)

   startup.log.message("o", m)
   processOutDestMail(m)
   

#---------------------------------------------------------------------

"""***
Process an outgoing email for multiple recipients. The outgoing email
is passed as an instance of mailheader.Mail.
***"""

def processOutDestMail(m):
   recipTo = m.header.getaddrlist("To")
   recipCc= m.header.getaddrlist("Cc")
   recipBcc = m.header.getaddrlist("Bcc")
   recipAll = recipTo + recipCc + recipBcc
   if debug: print "||| outgoing: recipAll = %s" % (recipAll,)
   processForMulti(m, recipAll)


#---------------------------------------------------------------------

"""***
Process an email message that may go to multiple recipients

Parameters
~~~~~~~~~~
m (a mailheader.Mail) the mail message before processing

recipAll -- a list of all the recipients ; each member is a list of
   2 items, [0] is the recipient's real name, [1] is the email address.

***"""

def processForMulti(m, recipAll):
   ix = 1   
   finalDestHandler = destination.makeDestination(startup.config['dest'])

   for recipAddress in recipAll:
      processedMail = processForRecipient(m, recipAddress)
      if debug: print ("||| outgoing: recipAddress[1]=%s"
         + " finalDestHandler=%s") % (recipAddress[1], finalDestHandler)
      startup.log.message("op", processedMail)

      context = {'ix': ix, 'recip': recipAddress[1]}
      finalDestHandler.output(processedMail, context) 
      if superDebug:
         utility.writeFile(os.path.join(startup.getHerbDir(), "mess_outp"),
            processedMail.asString())

      ix += 1

#---------------------------------------------------------------------

"""***
Process an email message for one recipient.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) the mail message, before processing

recipient (a list)
   recipient[0] is the recipient's real name
   recipient[1] is the recipient's email address

Return value
~~~~~~~~~~~~
A mailheader.Mail containing the mail message to be output.

***"""

def processForRecipient(m, recipient):

   # do we have a public key for this recipient?
   # If so, we must encrypt this message
   encrypt = startup.publicKeys.has_key(recipient[1])
   if debug: print "outgoing.py processForRecipient(%s) encrypt = %s" \
      % (recipient[1], encrypt)
   encTxt = X_HV_PLAINTEXT # value in X-Herbivore header for non-encrypted
   if encrypt: encTxt = X_HV_ENCRYPTED
   
   # Build up the processed version of this email:
   mProcessed = mailheader.Mail()
   mProcessed.header.readFromString(m.header.asString())
   
   # add Herbivore headers to the processed email:
   mProcessed.header.put(X_HV, encTxt)
   mProcessed.header.put(X_HV_VERSION, HERBIVORE_VERSION)
   
   # broadcast our key:
   myPublicKey = encwrap.stripHF(startup.secretKeys['public'])
   mProcessed.header.put(X_HV_KEY, myPublicKey)
   
   # encrypt if we can
   if encrypt:      
      encryptForOutput(m, mProcessed, recipient[1])
   else:
      mProcessed.body = m.body # don't process the body 
      mProcessed.header.removeheader("Bcc")   
      # for an unencrypted email, we should include the old values 
      # of the To: and Cc: header fields (but not the Bcc:).
      # This is so the recipient can reply to all the people
      # the original post was sent to, if he wants to.
    
   return mProcessed
   
   
#---------------------------------------------------------------------

"""***
Encrypt an email message for output.

This includes putting some of the headers from the message into
a header-in-plaintext text block at the start of the plaintext to
be encrypted.

On exit, the value of (mProcessed) is suitably changed; it follows
that this is passed-by-reference.

Parameters
~~~~~~~~~~
m (a mailheader.Mail) contains the input message (before processing)

mProcessed (a mailheader.Mail) contains the output message being worked
   on. When this function is invoked, (mProcessed.header) contains the 
   headers from (m) with the X-Herbivore etc headers added, and 
   (mProcessed.body) is empty; it will contain the encrypted body.
  
recipient (a string). The email address of the recipient of this
   message.  

Local variables
~~~~~~~~~~~~~~~

headerInPT (a mailheader.MailHeader) various "header" lines that go
   at the start of the plain text.
   
plainText (a string) the plain text, ie. the stuff that gets encrypted.

cipherText (a string) what you get when you encrypt (plainText).   

***"""

def encryptForOutput(m, mProcessed, recipientAddr):
   recipientPubKey = startup.publicKeys[recipientAddr]['key']
   headerInPT = mailheader.MailHeader()
      
   # transfer other fields, removing the field from mProcessed's header   
   for f in XFER_FIELDS:
      v = m.header.get(f)
      if v == None: continue
      headerInPT.put(f, v)
      mProcessed.header.removeheader(f) 
          
   for f in REMOVE_FIELDS: 
      mProcessed.header.removeheader(f) 
        
   mProcessed.header.put("Subject", "(herbivore-encrypted mail)")
   mProcessed.header.put("To", recipientAddr)
         
   if debug:
      comment = "Encrypted-header (before encryption) is\n" \
         + headerInPT.asString()
      startup.log.comment(comment)
   
   # for some wierd reason that I've nbot been able to figure out,
   # sometimres the first 2 characters to get encrypted become garbled,
   # so I'm putting some crap here, so that gets garbled instead of good
   # stuff.
     
   plainText = "Aaaa: dummy\n" + headerInPT.asString() + "\n" + m.body.asString()
   if superDebug:
      utility.writeFile(os.path.join(startup.getHerbDir(), "mess_out_pt"),
         plainText)
   cipherText = encwrap.pkEnc(recipientPubKey, plainText)
   mProcessed.body.readFromString(cipherText)
      

#---------------------------------------------------------------------
#end outgoing.py
