/* pgp4pine keyrings.c
 *
 * See common.c for the version and license.
 * Copyright (C) 1998 by Chris Wiegand
 */

#include "defines.h"
#include "includes.h"
#include "structs.h"
#include "declares.h"
#include "externs.h"

/* ------------------------------------------------------------ */

char *getItem(const char *incoming, const char delimiter, unsigned int fieldNum)
{
	// I support a string as long as an "unsigned long" is big,
	//     and as many fields as an "unsigned int" is big.
	// I return a pointer that must be "free"ed!

	char *tmpString = NULL, *retVal=NULL, *tmpStr = NULL;
	unsigned long l;
	unsigned int thisItem=1;

	tmpString = (char *) myMalloc(strlen(incoming)+2);
	retVal = (char *) myMalloc(strlen(incoming)+2);    // 2 just to be safe.
	strcpy(tmpString,incoming); // STRCPY SAFE
	strcpy(retVal,""); // STRCPY SAFE

	/* I only set startChar and endChar on a delimiter so that if the
	 * requested item is the 1st item, it starts and ends on item #1.
	 */

	for (l=0;l<=strlen(tmpString);l++) {
		if (tmpString[l] == delimiter) {
			thisItem++;
		} else {
			if (thisItem == fieldNum) {
				tmpStr = tmpString+l; // we want to set the "tmpStr" to a particular starting place within tmpString;
				strncat(retVal,tmpStr,1); // now, copy 1 ONE character...
			}
		}
	}

	if (tmpString != NULL) free(tmpString);
	return retVal;
}

void deallocAllKeys(struct pkiKey *killKey)
{
	struct pkiKey *thisKey = NULL;
	struct pkiKey *nextKey2 = NULL;
	thisKey = killKey; // copy the pointer...
	if (thisKey != NULL) {
		do {
			snprintf(debugLine,DEBUG_LINE_LENGTH,"%p, ",thisKey);
			documentStatus(debugLine);
			nextKey2 = thisKey->nextKey;
			myFree(thisKey);
			thisKey = nextKey2;
		} while (thisKey != NULL);
	}
	documentStatus("Done with this keyring.\n");
}

// -----------------------------------------------------------------

struct pkiKey *getKeyring(int getSecretOnly)
{
	// I return a pointer that must be "free"ed!
	struct pkiKey *myNewKey = NULL;
	struct pkiKey *firstKey = NULL;
	struct pkiKey *oldKey = NULL;

	FILE *inFile, *outFile; // DO NOT FREE ME ( I don't think so, that is. )
	char buf[65530], buf2[CONSOLE_IO_LINE_LENGTH], keyType[8], tmpString[CONSOLE_IO_LINE_LENGTH], appString[MAX_COMMAND_LINE_LENGTH];
	char *tmpString2;
	int inSec=0, lineType=0, needSubkey=0;
	
	switch (myUserPrefs->pgpVersion) {
	case 5:
		// PGP 5 only needs one pass, as it gives us ALL the keys...
		sprintf(appString,"%s +language=us -l",myUserPrefs->pgp5pgpk);
			
		pipe_pgp(appString,&inFile,&outFile);
		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; // we're done.....
			documentStatus(buf);
			strncpy(keyType,strtok(buf," "),8);
			lineType=0; // unknown default...
			if (strcmp(keyType,"sec") == 0 || strcmp(keyType,"sec+") == 0) lineType=1; // secret line...
			if (strcmp(keyType,"pub") == 0) lineType=2; // public key
			if (strcmp(keyType,"sub") == 0) lineType=3; // subkey...
			if (strcmp(keyType,"uid") == 0) lineType=4; // userid..
			if (strcmp(keyType,"sig") == 0 || strcmp(keyType,"SIG") == 0 
			                               || strcmp(keyType,"sig?") == 0) lineType=5; // signature...

			if (lineType == 1) inSec = 1;
			if (lineType == 2) inSec = 0;

			switch (lineType) {
			case 1:
			case 2:
				// If we don't want public keys, skip this look...
				if (lineType == 2 && getSecretOnly) break;
				stillAlive();
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;
                                                                                                                                
				// next, key size...
				strncpy(tmpString,strtok('\0'," "),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				// next, keyID
				strncpy(tmpString,strtok('\0'," "),KEY_ID_LENGTH);
				strncpy(myNewKey->keyID,tmpString,KEY_ID_LENGTH);
				
				// next, 2 dates
				strncpy(tmpString,strtok('\0'," "),5);
				strncpy(tmpString,strtok('\0'," "),5);
				// I don't use them...
				
				// next, key type...
				strncpy(tmpString,strtok('\0'," "),KEY_TYPE_LENGTH);
				strncpy(myNewKey->keyType,tmpString,KEY_TYPE_LENGTH);
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					// If we're loading only secret keys, I only need one name
					// If we're loading all keys, I need all names...
					if (getSecretOnly) {
						// If loading only secret keys, and already have a name, break.
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						// If loading all keys, need all names, if last key has a name already, make new key...
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID); // STRCPY SAFE
							strcpy(myNewKey->keySize,oldKey->keySize); // STRCPY SAFE
							strcpy(myNewKey->keyType,oldKey->keyType); // STRCPY SAFE
			                                oldKey = myNewKey;
						}
					}

					strncpy(tmpString,strtok('\0',"\n"),EMAIL_ADDRESS_MAX_LENGTH);
					UnescapeString(tmpString);	// remove escaped Chars
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			}
		}
		fclose(outFile);
		break;
	case 6:
		// PGP 6 requires two passes, once for secret keys, then for public keys...
		if (getSecretOnly) {
			sprintf(appString,"%s +LANGUAGE=en -kv \"*\" ~/.pgp/secring.pgp",myUserPrefs->pgp2pgp);
		} else {
			sprintf(appString,"%s +LANGUAGE=en -kvv", myUserPrefs->pgp2pgp);
		}
		pipe_pgp(appString,&inFile,&outFile);

		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			stillAlive();
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; // we're done.....
			documentStatus(buf);
			strncpy(keyType,buf,3);
			lineType = 0; // unknown yet...
			if ((strcmp(keyType,"DSS") == 0) || (strcmp(keyType,"RSA") == 0))
			  lineType = (getSecretOnly ? 1 : 2); // secret or public key
			if (needSubkey && (strcmp(keyType," DH") == 0))
			  lineType = 6;	// Subkey which carries UID we need
			if (strcmp(keyType,"sig") == 0) lineType = 5; // signature
			if (strcmp(keyType,"   ") == 0) lineType = 4; // name

			switch (lineType) {
			case 1:
			case 2:
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;

				// get key type (still in 'keyType')
			        if (strcmp(keyType,"DSS") == 0) strcpy(myNewKey->keyType,"DSS"); else
			        if (strcmp(keyType,"RSA") == 0) strcpy(myNewKey->keyType,"RSA");
			        else strcpy(myNewKey->keyType,"???");
			
				// eat leading "sec" or "pub"
				strtok(buf, " ");
				
				// next, key size...
				strncpy(tmpString,strtok('\0'," "),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				// next, keyID
				strncpy(myNewKey->keyID,strtok('\0'," "),KEY_ID_LENGTH);
				
				// next, 1 date
				strncpy(tmpString,strtok('\0'," "),5);
				// I don't use it...
				
				// Now either a UID follows, or nothing, or "expires..."
				// In the last two cases, a " DH" subkey with UID follows
				
				needSubkey = 1;
				if ((tmpString2=strtok('\0',"\n")) != NULL) // otherwise nothing follows
				 if (strncmp(tmpString2,"expires",6))
				 {			
				   strncpy(myNewKey->displayName,extractDisplayName(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				   strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				   needSubkey=0;
				 }
				break;
			case 6: // Get UID from Subkey to current 'myNewKey'
				// Skip a number of tokens
				strtok(buf ," "); // " DH"
				strtok('\0'," "); // Size
				strtok('\0'," "); // KeyID
				strtok('\0'," "); // Creation Date
				tmpString2 = strtok('\0',"\n");
				if (tmpString2 != NULL) // It also happens the subkey to be empty
				{
				  strncpy(myNewKey->displayName,extractDisplayName(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				  strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString2),EMAIL_ADDRESS_MAX_LENGTH);
				  needSubkey=0; // We got it !!!
				}
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					// If we're loading only secret keys, I only need one name
					// If we're loading all keys, I need all names...
					if (getSecretOnly) {
						// If loading only secret keys, and already have a name, break.
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						// If loading all keys, need all names, if last key has a name already, make new key...
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID); // STRCPY SAFE
							strcpy(myNewKey->keySize,oldKey->keySize); // STRCPY SAFE
							strcpy(myNewKey->keyType,oldKey->keyType); // STRCPY SAFE
			                                oldKey = myNewKey;
						}
					}

					// We have to use sscanf with a space in the first position
					// of the format string to ignore the leading spaces before the name
					strncpy(buf2,buf,CONSOLE_IO_LINE_LENGTH); // Makes sscanf safe...
					sscanf(buf2," %[^\n]",tmpString); // SSCANF SAFE
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			}
		}
		fclose(outFile);
		break;
	case 2:
		// PGP 2, OTOH, requires two passes, once for secret keys, then for public keys...
		if (getSecretOnly) {
			sprintf(appString,"%s +LANGUAGE=en -kv ~/.pgp/secring.pgp",myUserPrefs->pgp2pgp);
		} else {
			sprintf(appString,"%s +LANGUAGE=en -kvv", myUserPrefs->pgp2pgp);
		}
		pipe_pgp(appString,&inFile,&outFile);

		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			stillAlive();
			fgets(buf,sizeof(buf),outFile);
			if (strlen(buf) < 1) break; // we're done.....
			documentStatus(buf);
			strncpy(keyType,buf,3);
			lineType = 0; // unknown yet...
			if (strcmp(keyType,"sec") == 0) lineType = 1; // secret key
			if (strcmp(keyType,"pub") == 0) lineType = 2; // public key
			if (strcmp(keyType,"sig") == 0) lineType = 5; // signature
			if (strcmp(keyType,"   ") == 0) lineType = 4; // name

			switch (lineType) {
			case 1:
			case 2:
				if (lineType == 2 && getSecretOnly) break;
                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
                                if (firstKey == NULL) firstKey = myNewKey;
                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
                                oldKey = myNewKey;

				// eat leading "sec" or "pub"
				strtok(buf, " ");
				
				// next, key size...
				strncpy(tmpString,strtok('\0',"/"),KEY_SIZE_LENGTH);
				strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);

				// next, keyID
				strncpy(tmpString,strtok('\0'," "),KEY_ID_LENGTH);
				sprintf(myNewKey->keyID,"0x%s",tmpString); // SPRINTF SAFE
				
				// next, 1 date
				strncpy(tmpString,strtok('\0'," "),5);
				// I don't use it...
					
				// next, key type...
				strcpy(myNewKey->keyType,"RSA"); // STRCPY SAFE
				
				strncpy(tmpString,strtok('\0',"\n"),EMAIL_ADDRESS_MAX_LENGTH);
				strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				break;
			case 4:
				if (!getSecretOnly || inSec) {
					// If we're loading only secret keys, I only need one name
					// If we're loading all keys, I need all names...
					if (getSecretOnly) {
						// If loading only secret keys, and already have a name, break.
						if (strlen(myNewKey->displayName) > 2) break;
					} else {
						// If loading all keys, need all names, if last key has a name already, make new key...
						if (strlen(myNewKey->displayName) > 2) {
			                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
			                                if (firstKey == NULL) firstKey = myNewKey;
			                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
							strcpy(myNewKey->keyID,oldKey->keyID); // STRCPY SAFE
							strcpy(myNewKey->keySize,oldKey->keySize); // STRCPY SAFE
							strcpy(myNewKey->keyType,oldKey->keyType); // STRCPY SAFE
			                                oldKey = myNewKey;
						}
					}

					// We have to use sscanf with a space in the first position
					// of the format string to ignore the leading spaces before the name
					strncpy(buf2,buf,CONSOLE_IO_LINE_LENGTH); // Makes sscanf safe...
					sscanf(buf2," %[^\n]",tmpString); // SSCANF SAFE
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
				}
				break;
			}
		}
		fclose(outFile);
		break;
	case 1:
		// GPG also requires two passes...
		if (getSecretOnly) {
			// Because of a bug in GnuPG 0.4x and 0.9x (thru 0.95, at least), we can only tell it ONE keyring file
			// when we're loading keyrings, or it displays it as many times as keyring files we give it.
			sprintf(appString,"%s --list-secret-keys --with-colons --no-greeting 2> /dev/null",myUserPrefs->pgp1gpg);
		} else {
			sprintf(appString,"%s --list-keys --with-colons --no-greeting 2> /dev/null",myUserPrefs->pgp1gpg);
		}

		pipe_pgp(appString,&inFile,&outFile);
		if (outFile == NULL) return NULL;
		while (!feof(outFile)) {
			stillAlive();
			// using fgets() causes problems...
			cpString(outFile,buf,65530);
			if (strlen(buf) < 1) break; // we're done.....
			documentStatus(buf);
			if (strchr(buf,':') != NULL) {
				strncpy(keyType,getItem(buf,':',1),3);
				lineType = 0;
				if (strcmp(keyType,"sec") == 0) lineType = 1; // secret line...
				if (strcmp(keyType,"pub") == 0) lineType = 2; // public key
				if (strcmp(keyType,"uid") == 0) lineType = 4; // user id
	
				if (lineType == 1) inSec = 1;
				if (lineType == 2) inSec = 0;

				switch (lineType) {
				case 1:
				case 2:
					if (lineType == 2 && getSecretOnly) break;
	                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
	                                if (firstKey == NULL) firstKey = myNewKey;
	                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
	                                oldKey = myNewKey;
		
					// next, key size...
					strncpy(tmpString,getItem(buf,':',3),KEY_SIZE_LENGTH);
					strncpy(myNewKey->keySize,tmpString,KEY_SIZE_LENGTH);
	
					// next, GPG key type...
					strncpy(tmpString,getItem(buf,':',4),KEY_TYPE_LENGTH);
					strcpy(myNewKey->keyType,"Unknown"); // STRCPY SAFE
					if (strcmp(tmpString,"1") == 0) strcpy(myNewKey->keyType,"RSA"); // STRCPY SAFE
					if (strcmp(tmpString,"16") == 0) strcpy(myNewKey->keyType,"ElGamal"); // STRCPY SAFE
					if (strcmp(tmpString,"17") == 0) strcpy(myNewKey->keyType,"DSA"); // STRCPY SAFE
					if (strcmp(tmpString,"20") == 0) strcpy(myNewKey->keyType,"ElGamal"); // STRCPY SAFE
				
					// keyID
					strncpy(tmpString,getItem(buf,':',5),KEY_ID_LENGTH);
					//	Any GPG keyid starting w/ a letter
				        //	must be prefixed with '0' to work.
				        //
				        //	As the additional '0' does no harm
				        //	in case of the keyid starting w/ a
				        //	digit, add it anywhere.
				        //
				        //	Keyid len is 64bits, which makes for
				        //	16 digits, size of array is big enough.
				        //
					sprintf(myNewKey->keyID,"0%s",tmpString); // SPRINTF SAFE

					// finally, user name
					strncpy(tmpString,getItem(buf,':',10),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					break;
				case 4:
					if (!getSecretOnly || inSec) {
						// If we're loading only secret keys, I only need one name
						// If we're loading all keys, I need all names...
						if (getSecretOnly) {
							// If loading only secret keys, and already have a name, break.
							if (strlen(myNewKey->displayName) > 2) break;
						} else {
							// If loading all keys, need all names, if last key has a name already, make new key...
							if (strlen(myNewKey->displayName) > 2) {
				                                myNewKey = (struct pkiKey *) myMalloc(sizeof(pkiKeyStruct));
				                                if (firstKey == NULL) firstKey = myNewKey;
				                                if (oldKey != NULL) oldKey->nextKey = myNewKey;
								strcpy(myNewKey->keyID,oldKey->keyID); // STRCPY SAFE
								strcpy(myNewKey->keySize,oldKey->keySize); // STRCPY SAFE
								strcpy(myNewKey->keyType,oldKey->keyType); // STRCPY SAFE
				                                oldKey = myNewKey;
							}
						}

						// finally, user name
						strncpy(tmpString,getItem(buf,':',10),EMAIL_ADDRESS_MAX_LENGTH);
						strncpy(myNewKey->displayName,extractDisplayName(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
						strncpy(myNewKey->emailAddress,extractEmailAddress(tmpString),EMAIL_ADDRESS_MAX_LENGTH);
					}
					break;
				}
			}
		}
		fclose(outFile);
		break;
	}
	return firstKey;
}

void getSecretKeyring()
{
	printf("Loading secret keyring (this may take a couple of seconds)...\n");
	secretKeyring = getKeyring(1);
	printf("\n");
}

void chooseSecretKey()
{
	char retValue[EMAIL_ADDRESS_MAX_LENGTH];
	strcpy(retValue,""); // STRCPY SAFE

	snprintf(debugLine,DEBUG_LINE_LENGTH,"keyrings.c: chooseSecretKey()\n");
	documentStatus(debugLine);
	
	// now, let's ask which one they want to use...
	if (secretKeyring == NULL) {
		// no secret keys found!!!
		printf("\nWarning! It appears that you don't have a secret key defined! Please see\n"
		       "your PGP/GPG documentation for assistance in making a key pair!\n");
	} else {
		if (secretKeyring->nextKey != NULL) {
			// multiple keys...
			int i=0, sel = 0, found=0;
			struct pkiKey *currentKey = NULL;
	
			currentKey = secretKeyring;
			do {
				if (strcmp(myUserPrefs->myAddress,currentKey->emailAddress)==0)
				{
				  found=1;
				  documentStatus("chooseSecretKey(): Using ");
				  documentStatus(myUserPrefs->myAddress);
				}
				else
				  currentKey = currentKey->nextKey;
			} while (currentKey!=NULL && !found);
			
			if (!found)
			{
			// got no default key defined...
			  printf("\nYou have multiple secret keys. Listed below are the keys. Please select one.\n");
			  currentKey = secretKeyring;
			  do {
				i++;
				printKeyInfo((char) (i+'a'-1),currentKey);
				currentKey = currentKey->nextKey;
			  } while (currentKey != NULL);
			  printf("\n");
			  sel = askAlphaRange("Please enter a valid selection: ",(i+'a'-1),0)-'a';
			  currentKey = secretKeyring; // set back to beginning...

			  snprintf(debugLine,DEBUG_LINE_LENGTH,"Selected: %i\n",sel+'a');
			  documentStatus(debugLine);
			  for (i=0;i<sel;i++) currentKey = currentKey->nextKey; // skip to key we want
			  snprintf(debugLine,DEBUG_LINE_LENGTH,"CurrentKey: %s\n",currentKey->keyID);	
			  documentStatus(debugLine);
			}
#ifdef USE_DISPLAY_NAME
			strcpy(retValue,currentKey->displayName);
#else
			strcpy(retValue,currentKey->keyID);
#endif
	
			strcpy(mySecretKey,retValue);
			printf("Please wait ...\n");
		} else {
			// only one secret key...
#ifdef USE_DISPLAY_NAME
			strcpy(retValue,secretKeyring->displayName);
#else
			strcpy(retValue,secretKeyring->keyID);
#endif
			strcpy(mySecretKey,retValue);
		}
	}
	documentStatus("Leaving secret area!\n");
}

void getPublicKeyring()
{
	printf("Loading public keyring (this may take a couple of seconds)...\n");
	publicKeyring = getKeyring(0);
	printf("\n");
}

void regetPublicKeyring()
{
	printf("Okay, getting public keys again ...\n");
	deallocAllKeys(publicKeyring);
	publicKeyring = getKeyring(0);
	printf("\n");
}

void gettheKeys()
{
	if (publicKeyring == NULL)
	{
		getPublicKeyring();
		getSecretKeyring();
		// Is my Adress given in pgp4pine.conf ?
		if ((myUserPrefs->myAddress[0]=='0')
		   && ((myUserPrefs->myAddress[1]=='x')
		   || ((myUserPrefs->myAddress[1]=='X'))))
		  strcpy(mySecretKey,myUserPrefs->myAddress);
		else
		  chooseSecretKey();
	}
}

char *extractDisplayName(const char *stringIn)
{
	// I return a pointer that needs to be "free"ed!
	char tmpString[EMAIL_ADDRESS_MAX_LENGTH], *newStr;
	strcpy(tmpString,stringIn);

	newStr = (char *) myMalloc(EMAIL_ADDRESS_MAX_LENGTH);
	if (tmpString[0] == '<') {
		// this looks like a <email> only line ...
		strcpy(newStr,tmpString); // or what ???
	} else if (strchr(tmpString,'<') > 0 && strchr(tmpString,'>') > 0) {
		// okay, we've got a realName <email> line...
		strcpy(newStr,strtok(tmpString,"<"));
	} else 
		strcpy(newStr,tmpString);
	return newStr;
}

char *extractEmailAddress(const char *stringIn)
{
	// I return a pointer that needs to be "free"ed!
	char tmpString[EMAIL_ADDRESS_MAX_LENGTH], *newStr;
	strcpy(tmpString,stringIn);

	newStr = (char *) myMalloc(EMAIL_ADDRESS_MAX_LENGTH);
	if (tmpString[0] == '<') {
		// this looks like a <email> only line ...
		strcpy(newStr,strtok(tmpString+1,">"));
	} else if (strchr(tmpString,'<') > 0 && strchr(tmpString,'>') > 0) {
		// okay, we've got a realName <email> line...
		strcpy(newStr,strtok(tmpString,"<"));
		strcpy(newStr,strtok('\0',">"));
	} else 
		strcpy(newStr,tmpString);
	return newStr;
}

