/* pgp4pine pki.c
 *
 * See common.c for the version and license.
 * Copyright (C) 1998 by Chris Wiegand
 */

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <time.h>
#include "defines.h"
#include "includes.h"
#include "structs.h"
#include "declares.h"
#include "externs.h"
#include "md5.h"

/* ------------------------------------------------------------ */

int pipe_pgp(const char *cmd, FILE **in, FILE **out)
{
         int pin[2], pout[2], child_pid;

         *in = *out = NULL;

         pipe(pin);
         pipe(pout);
 

	 documentStatus("pki.c:\n\tpipe_pgp (cmd = '");
	 documentStatus(cmd);
	 documentStatus("', FILE **in, FILE **out)\n");

         if(!(child_pid = fork()))
         {
           /* We're the child. */
           close(pin[1]);
           dup2(pin[0], 0);
           close(pin[0]);

           close(pout[0]);
           dup2(pout[1], 1);
           close(pout[1]);

/*            execl("/bin/sh", "sh", "-c", cmd, NULL); */
	  system(cmd);
           _exit(127);
         }

         /* Only get here if we're the parent. */
         close(pout[1]);
         *out = fdopen(pout[0], "r");

         close(pin[0]);
         *in = fdopen(pin[1], "w");

         return(child_pid);
}

/* --------------------------------------------------------------------------------- */

void fileVerify(const char *inFile,const char *outFile)
{
	char sigFile[MAX_COMMAND_LINE_LENGTH];
	char pgpAppString[MAX_COMMAND_LINE_LENGTH];
	char c;
	FILE *sigf, *outf;
	
	documentStatus("pki.c\n\tfileVerify (inFile = '");
	documentStatus(inFile);
	documentStatus("', outFile = '");
	documentStatus(outFile);
	documentStatus("')\n");
	sprintf(sigFile,"%s.sig",outFile); /* Signature file is outfile.sig */
	
	switch (myUserPrefs->pgpVersion) {
	case 5:
		sprintf(pgpAppString,"%s -o %s %s 2>%s",myUserPrefs->pgp5pgpv,outFile,inFile,sigFile);
		break;
	case 2:
	case 6:
		sprintf(pgpAppString,"%s %s -o %s 2>%s",myUserPrefs->pgp2pgp,inFile,outFile,sigFile);
		break;
	case 1:
		sprintf(pgpAppString,"%s --no-greeting --output %s --decrypt %s 2>%s",
		   myUserPrefs->pgp1gpg,outFile,inFile,sigFile);
		break;	
	}
	documentStatus("\t\tpgpAppString = '");
	documentStatus(pgpAppString);
	documentStatus("'\n");
	runProcessFG(pgpAppString);

/* TODO: Signature-ID check: Unknown -> get keyserver */
	
	if (((outf = fopen(outFile,"a")) != NULL) && ((sigf = fopen(sigFile,"r"))!=NULL))
	{
	  fprintf(outf,"------------ Output from %s ------------\n",
	                myUserPrefs->pgpVersion==1?"gpg":"pgp");
          do
	  {
	    c=getc(sigf);
	  } while ((c!=EOF) && (c!='\n')); /* Skip first line */

	  while ((c=getc(sigf))!=EOF)
	    putc(c,outf);
	  fclose(outf); fclose(sigf);
	} else
	   printf("Signature check failed.");
	remove(sigFile);
}

/* --------------------------------------------------------------------------------- */

void fileDecrypt(const char *inFile,const char *outFile)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH];

	documentStatus("pki.c\n\tfileDecrypt (inFile = '");
	documentStatus(inFile);
	documentStatus("', outFile = '");
	documentStatus(outFile);
	documentStatus("')\n");

	switch (myUserPrefs->pgpVersion) {
	case 5:
		sprintf(pgpAppString,"%s -o %s %s",myUserPrefs->pgp5pgpv,outFile,inFile);
		break;
	case 2:
	case 6:
		sprintf(pgpAppString,"%s %s -o %s",myUserPrefs->pgp2pgp,inFile,outFile);
		break;
	case 1:
		sprintf(pgpAppString,"%s --no-greeting --output %s --decrypt %s",myUserPrefs->pgp1gpg,outFile,inFile);
		break;	
	}
	documentStatus("\t\tpgpAppString = '");
	documentStatus(pgpAppString);
	documentStatus("'\n");
	runProcessFG(pgpAppString);
        if (prefsCacheDecrypted>0) /* Cache the new file */
	{
	   struct stat stats;
	   int  in,cache;
	   u_int8_t *inbuf;
	   u_int32_t date;
	   u_int32_t size;
	   MD5_CONTEXT ctx;
	   char tmpString[35];
	   
	   /* If anything goes wrong, just don't cache */
	   if ((in = open(inFile,O_RDONLY))<0) return;
	   if (fstat(in,&stats)<0) return;	/* stat for size */
	   if (stats.st_size==0) return;
	   
	   /* Map file to memory */
	   if ((inbuf = (u_int8_t *)mmap(0,stats.st_size, PROT_READ, MAP_SHARED,in,0))
	       == (u_int8_t *) -1) { close(in); return; }
	   /* Create a md5 sum to identify it later */
	   md5_init(&ctx);
	   md5_write(&ctx, inbuf, stats.st_size);
	   md5_final(&ctx);
	   munmap(inbuf, stats.st_size);
	   close(in); 

	   if ((in = open(outFile,O_RDONLY))<0) return;
	   if (fstat(in,&stats)<0) return;
	   if ((inbuf = (u_int8_t *)mmap(0,stats.st_size, PROT_READ, MAP_PRIVATE,in,0))
	       == (u_int8_t *) -1) { close(in); return; }
	 
	   /* and create the cache entry */
	   documentStatus("Creating cache entry MD5: ");
	   sprintf(tmpString,"%x%x%x%x%x%x%x%x %x%x%x%x%x%x%x%x\n",
 		ctx.buf[0],ctx.buf[1],ctx.buf[2],ctx.buf[3],
		ctx.buf[4],ctx.buf[5],ctx.buf[6],ctx.buf[7],
		ctx.buf[8],ctx.buf[9],ctx.buf[10],ctx.buf[11],
		ctx.buf[12],ctx.buf[13],ctx.buf[14],ctx.buf[15]);
	   documentStatus(tmpString);
	   if ((cache = open(DecryptCacheName,O_WRONLY|O_CREAT|O_APPEND,0600))<0) return;

	   write(cache,"\022\064",2); /* magic 0x1234*/
	   size = stats.st_size+27;
	   size = to_bigend_u32(size);
	   write(cache,&size,4);  /* size */
	   write(cache,"\1",1); /* version */
	   write(cache,ctx.buf,16); /* MD5 */
	   date = (time(0)/3600);    /* hour resolution */
	   date = to_bigend_u32(date);
	   write(cache,&date,4);
	   write(cache, inbuf, stats.st_size);
	   close(cache);
	   munmap(inbuf, stats.st_size);
	   close(in);
	}
}
/* --------------------------------------------------------------------------------- */
/* Try to find a Cleartext version of <inFile> in the cache. 
   Write to <outFile> if successful and return 1, else return 0. */

#define headerstring "----- pgp4pine " VERSION " cached encrypted file -----\n\n"
#define cacheversion 1

void checkCache()
{
	int cache;
	int flush;
	u_int32_t today,date;
	struct stat stats;
	off_t  size,newsize;
	u_int8_t *buf,*bufstart;
	u_int32_t entrysize;
	
	if ((cache=open(DecryptCacheName,O_RDWR))<0) return;
	if (fstat(cache,&stats)<0) return; /* stat for size */
	newsize = size = stats.st_size; /* copy length and use as counter */
	if ((buf = bufstart = (u_int8_t *)mmap(0,stats.st_size, 
	    PROT_READ|PROT_WRITE,MAP_SHARED,cache,0))== (u_int8_t *) -1) return;
	
	do{
	   flush=0;
           if (buf[0] != 0x12) flush=1; 
	   if (buf[1] != 0x34) flush=1;   /* no magic: integrity violated */
	   if (buf[6] >1)   flush=1;         /* unknown version */
	   date  = (buf[23]<<24)|(buf[24]<<16)|(buf[25]<<8)|buf[26];
           today = time(0)/3600;
	   if (today >=  (date + (prefsCacheDecrypted*24)))
	     flush=1;                                      /* too old */
	   entrysize = (buf[2]<<24)|(buf[3]<<16)|(buf[4]<<8)|buf[5];
	   
	   size -= entrysize;
	   if (size>=0) /* if size<0, size integrity was violated */
	   {
	     if (flush==1)  /* do a flush: Copy from next msg until EOF */
	     {
	        if (size>0) 
		   memmove(buf,buf + entrysize, size);
	        newsize -= entrysize;	        
	     }
	     else buf += entrysize;
	   }
	} while(size>0);
	munmap(bufstart, stats.st_size);
	
	if ((newsize < stats.st_size) && newsize>0) 
	  /* If we flushed, truncate file */
	  ftruncate(cache, newsize);
        close(cache);
        if (newsize<=0) remove(DecryptCacheName);
          /* <0 must ne an error. Kill either way .. */
}
int readCache(const char *inFile, const char *outFile)
{
	int i,found;
	int in,out,cache;
	struct stat instats,cachestats;
	u_int8_t *inBuf, *cacheBuf, *cacheBufstart;
	MD5_CONTEXT ctx;
	u_int32_t entrysize;
	
	if (((in=open(inFile,O_RDONLY))<0)
	 || ((cache=open(DecryptCacheName,O_RDONLY))<0)) return (0);
	if (fstat(in,&instats)<0) return(0); /* stat for size */
	if (fstat(cache,&cachestats)<0) return(0); /* stat for size */
	if ((inBuf = (u_int8_t *)mmap(0,instats.st_size, PROT_READ,MAP_SHARED,in,0))
	    == (u_int8_t *) -1) return(0);
	if ((cacheBuf = (u_int8_t *)mmap(0,cachestats.st_size, PROT_READ,MAP_PRIVATE,cache,0))
	    == (u_int8_t *) -1) return(0);
	cacheBufstart = cacheBuf; /* We will count up pointer, so save it */
	    
	md5_init(&ctx);
	md5_write(&ctx, inBuf, instats.st_size);
	md5_final(&ctx);
	munmap(inBuf, instats.st_size);
	close(in);
	
	do {
	   found=1;
	   for (i=0; i<16; i++) 
	      if (ctx.buf[i]!=cacheBuf[i+7]) found=0;
	   entrysize = (cacheBuf[2]<<24)|(cacheBuf[3]<<16)|(cacheBuf[4]<<8)|cacheBuf[5];
	      
	   if (!found) cacheBuf+= entrysize;
	} while (!found && (cacheBuf<cacheBufstart+cachestats.st_size));
	if (found)
	{
	   out=open(outFile,O_CREAT|O_WRONLY|O_TRUNC,0600);
           write(out,headerstring,sizeof(headerstring)-1); /* write without trailing \0 */
	   if (cacheBuf[6] == 1)
	   {
	     write(out,cacheBuf+27,entrysize-27);
           }
	   close(out);
	   munmap(cacheBufstart,cachestats.st_size);
	   close(cache);
	   return(1);
	} else {
	   munmap(cacheBufstart,cachestats.st_size);
	   close(cache);
	  return(0);
	}
}
/* --------------------------------------------------------------------------------- */

int fileSignEncrypt(const char *inFileName,
		     const char *outFileName,
		     const int sign, const int encrypt,
		     char *recipient, 
		     const int asciiArmor, const int universalText,
		     int encryptToSelf,int detachSigs)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH], *commentString;
	char pgpOptions[MAX_COMMAND_LINE_LENGTH], recipList[EMAIL_ADDRESS_MAX_LENGTH];
        int  exitcode;

	documentStatus("pki.c\n\tfileSignEncrypt (inFile = '");
	documentStatus(inFileName);
	documentStatus("', outFile = '");
	documentStatus(outFileName);
	documentStatus("', sign = ");
	if (sign == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", encrypt = ");
	if (encrypt == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", recipient = '");
	documentStatus(recipient);
	documentStatus(", asciiArmor = ");
	if (asciiArmor == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", universalText = ");
	if (universalText == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", encryptToSelf = ");
	if (encryptToSelf == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", detachSigs = ");
	if (detachSigs == 1) documentStatus("1"); else documentStatus("0");
	documentStatus("')\n");

	strcpy(pgpOptions," ");
	strcpy(recipList," ");
	strcat(pgpOptions,myUserPrefs->extraOptions);

	if (encryptToSelf > 0) {
		if (strlen(myUserPrefs->myAddress) > 3) {
			strcat(recipient," ");
			if (addDashR == 1) strcat(recipient,"-r ");
			strcat(recipient,myUserPrefs->myAddress);
			strcat(recipient," ");
		} else {
			if (myUserPrefs->pgpVersion != 5) printf("Because you didn't specify your email address in ~/.pgp4pinerc,\nI can't auto-encrypt to you as well.\n");
			encryptToSelf = -1;
		}
	}

	switch (myUserPrefs->pgpVersion) {
	case 5:
		if (detachSigs > 0) strcat(pgpOptions," -b ");
		/* get recipList... */
		if (encrypt > 0) {
			strcpy(recipList,recipient);
 			if (sign > 0) strcat(pgpOptions," -s ");
		}
		if (sign > 0 && strlen(mySecretKey) > 3) {
 				strcat(pgpOptions," -u \"");
 				strcat(pgpOptions,mySecretKey);
 				strcat(pgpOptions,"\" ");
		}
		if (asciiArmor > 0) strcat(pgpOptions," -a ");
		if (universalText > 0) strcat(pgpOptions," -t ");
		if (prefsAddComment > 0)
		{
			commentString="--comment=\"Made with pgp4pine " VERSION "\"";
		}
		else
			commentString = "";

		if (encrypt > 0)
			sprintf(pgpAppString,"%s %s %s %s -o %s %s",myUserPrefs->pgp5pgpe,pgpOptions,commentString,recipList,outFileName,inFileName);
		else
			sprintf(pgpAppString,"%s %s %s -o %s %s",myUserPrefs->pgp5pgps,pgpOptions,commentString,outFileName,inFileName);
		exitcode = runProcessFG(pgpAppString);
		break;
	case 2:
	case 6:
		strcat(pgpOptions," -"); /* we either have to add a e[s] or just s... */
		if (encrypt > 0) {
			strcat(pgpOptions,"e");
			strcpy(recipList,recipient);
		}
		if (sign > 0) {
			strcat(pgpOptions,"s");
			if (detachSigs > 0) strcat(pgpOptions,"b");
		}
		if (universalText > 0) strcat(pgpOptions,"t");
		if (asciiArmor > 0) strcat(pgpOptions,"a");
		strcat(pgpOptions," ");
		if (sign > 0 && strlen(mySecretKey) > 3) {
			strcat(pgpOptions," -u \"");
			strcat(pgpOptions,mySecretKey);
			strcat(pgpOptions,"\" ");
		}
		if (prefsAddComment > 0) 
			commentString = "+COMMENT=\"Made with pgp4pine\"";
		else
			commentString = "";

		if (encrypt > 0)
			sprintf(pgpAppString,"%s %s %s %s %s",myUserPrefs->pgp2pgp,pgpOptions,commentString,inFileName,recipList);
		else
			sprintf(pgpAppString,"%s %s %s %s",myUserPrefs->pgp2pgp,pgpOptions,commentString,inFileName);
		exitcode = runProcessFG(pgpAppString); 
		
		/* now, we move the <infile>.pgp to <outfile>*/
		if (detachSigs > 0) {
			/* we're detaching a signature, the original file isn't changed.... */
		} else {
			if (asciiArmor > 0)
				sprintf(pgpAppString,"mv %s.asc %s",inFileName,outFileName);
			else
				sprintf(pgpAppString,"mv %s.pgp %s",inFileName,outFileName);
			runProcessFG(pgpAppString);
		}
		break;
	case 1:
		if (myUserPrefs->asciiArmor > 0) strcat(pgpOptions, " -a ");
		if (myUserPrefs->universalText > 0) strcat(pgpOptions, " -t ");

		if (encrypt > 0) {
			strcat(pgpOptions," -e ");
			strcpy(recipList,recipient);
			if (myUserPrefs->encryptToSelf == 1) {
				strcat(recipList," -r ");
				strcat(recipList,myUserPrefs->myAddress);
			}
		}
		if (sign > 0) {
			if (detachSigs > 0) strcat(pgpOptions," -b ");
			strcat(pgpOptions," -s ");
			if (encrypt == 0) strcat(pgpOptions," --clearsign ");
			if (strlen(mySecretKey) > 3) {
				strcat(pgpOptions," -u \"");
				strcat(pgpOptions,mySecretKey);
				strcat(pgpOptions,"\" ");
			}
		}
		/* if we're not encrypting, recipList is just a " ", so we're safe... */
		/* Just moved outfilename to begining of list per email from henrik kg andreasson */
		if (prefsAddComment > 0) 
			commentString = "--comment \"Made with pgp4pine\"";
		else
			commentString = "";
			
		sprintf(pgpAppString,"%s -o %s --no-batch %s --no-greeting %s %s %s",
		    myUserPrefs->pgp1gpg,outFileName,commentString,recipList,pgpOptions,inFileName);
		sprintf(debugLine,"DEBUG: GnuPG Run: \"%s\"\n",pgpAppString);
		documentStatus(debugLine);
		exitcode=runProcessFG(pgpAppString);
		break;
	}
	return exitcode;
}
/* --------------------------------------------------------------------------------- */
/* Get unknown keys from keyserver */
void fileAskKey(char *nokeyrecipients)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH];
	char tmpEmail[EMAIL_ADDRESS_MAX_LENGTH];
	int  i=0;
	
	if (myUserPrefs->pgpVersion == 2) {
	  printf("Who the hell called this routine ? Anyway, PGP 2 cannot get keys from a keyserver.\n");
	} else {
	  documentStatus("Asking Keyserver for unknown keys");
	
	  do {
		if (i==0) strcpy(tmpEmail,strtok(nokeyrecipients," "));
		else strcpy(tmpEmail,strtok('\0'," "));
		if (strcmp(tmpEmail,".")) {
			if (strlen(tmpEmail) > 1)
			{ 
			  if (myUserPrefs->pgpVersion == 5)
			    sprintf(pgpAppString,"%s -a \"%s\"",myUserPrefs->pgp5pgpk,tmpEmail);
			  else
			    sprintf(pgpAppString,"%s --recv-keys \"%s\"",myUserPrefs->pgp1gpg,tmpEmail);

			  documentStatus(pgpAppString);
			  runProcessFG(pgpAppString);
			}
		}
		i++;
	  } while(strcmp(tmpEmail,"."));
	}
}

/* Parse Recipient list *recipients and correct ambiguous keys. 
Return is key-owners list *recipients and no-key list *nokeyrecipients. */

void parseRecipientList(char *recipients, char *nokeyrecipients)
{
	char tmpEmail[EMAIL_ADDRESS_MAX_LENGTH];
	char recipList[EMAIL_ADDRESS_MAX_LENGTH], nokeyList[EMAIL_ADDRESS_MAX_LENGTH];
	int i=0;

	strcpy(recipList,"");
	strcpy(nokeyList,"");
	
	/* recipients is passed back, so it's a pointer */
	strcat(recipients," . "); /* we add a space so that anything at the end is an item */
	/* if you don't, strtok() can segfault because there's no terminating " "... */
	documentStatus("pki.c\n\tparseRecipientList (recipients = '");
	documentStatus(recipients);
	documentStatus("', addDashR = ");
	if (addDashR == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(")\n");

 	do {
		if (i==0) strcpy(tmpEmail,strtok(recipients," "));
		else strcpy(tmpEmail,strtok('\0'," "));
		if (strcmp(tmpEmail,".")) {
			if (strlen(tmpEmail) > 1)
			{
/*			  printf(tmpEmail); */
			  if (checkRecipient(tmpEmail) == 1) 
			  {
			     printf(" - key %s found\n",tmpEmail);
			     if (addDashR == 1) strcat(recipList," -r");
			     strcat(recipList," \"");
			     strcat(recipList,tmpEmail);
			     strcat(recipList,"\" ");
			  } else {/* not found */
			     printf(" - no key found\n");
			     strcat(nokeyList,"\"");
			     strcat(nokeyList,tmpEmail);
			     strcat(nokeyList,"\" ");
			  }
			}
		}
		i++;
	} while(strcmp(tmpEmail,"."));
	strcpy(recipients,recipList);
	strcpy(nokeyrecipients,nokeyList);
	documentStatus("Recipients: with key: ");
	documentStatus(recipients);
	documentStatus(" without key: ");
	documentStatus(nokeyrecipients);
	documentStatus("\n");
	strcat(nokeyrecipients," . "); /* Terminate list */
}

int checkRecipient(char *thisRecip)
{
	/* This just checks a single recipient, when we parse the recipient list, we check each one... */
	struct pkiKey *thisKey = NULL;
	struct pkiKey *lastKey = NULL;
	int found = 0;

	documentStatus("pki.c\n\tcheckRecipient (thisRecip = '");
	documentStatus(thisRecip);
	documentStatus("')\n");
	printf("%s... ",thisRecip);

	found = 0;
	if (publicKeyring != NULL) {
		thisKey = publicKeyring;
		lastKey = thisKey;
		while (thisKey != NULL) {
			if (strcasecmp(thisRecip,thisKey->emailAddress) == 0) {
				found++;
				lastKey = thisKey;
			}
			thisKey = thisKey->nextKey;
		}
	}

	if (found < 1) {   /* No key found */
		documentStatus("\tNo key found for ");documentStatus(thisRecip);
		documentStatus("\n");
	} else if (found > 1) {
		int i=0,j=0,k=1;
		/* wow..two keys... */
		printf("\n\nThe following recipient has multiple keys. Please select one.\n");
		thisKey = publicKeyring; /* set back to beginning */
		do {
			if (strcasecmp(thisKey->emailAddress,thisRecip) == 0) {
				i++; /* number of able and ready keys... */
				printKeyInfo((char) ('a'+i-1),thisKey);
			}
			thisKey = thisKey->nextKey;
		} while (thisKey != NULL);
		j = askAlphaRange("Please select the recipient key to use.",'a',(i+'a'-1))-'a';
		thisKey = publicKeyring;
		k=0;
		while (k<=j) {
			documentStatus("\t\tk vs. j loop:\n\t\t\tthisKey->emailAddress = ");
			documentStatus(thisKey->emailAddress);
			documentStatus("\n\t\t\tthisKey->keyID = ");
			documentStatus(thisKey->keyID);
			documentStatus("\n");
			if (strcasecmp(thisKey->emailAddress,thisRecip) == 0) {
				k++;
				documentStatus("\t\t\tKey Matched!\n");
			}
			if (k <= j) thisKey = thisKey->nextKey;
		}
#ifdef USE_DISPLAY_NAME
		strcpy(thisRecip,thisKey->emailAddress);
#else
		strcpy(thisRecip,thisKey->keyID);
#endif
		documentStatus("\t\tThisRecip: ");
		documentStatus(thisRecip);
		documentStatus("\n");
		found = 1;	/* One key found */
	} else {
/*		printf("found (%s)\n",lastKey->keyID); */
		/* only one key for this recipient */
#ifdef USE_DISPLAY_NAME
		strcpy(thisRecip,lastKey->emailAddress);
#else
		strcpy(thisRecip,lastKey->keyID);
#endif
	}
	return found;
}

void parseList(char *recipients, char *nokeyrecipients)
{
	printf("\nChecking recipients ...\n\n");
	parseRecipientList(recipients,nokeyrecipients);
	/* addDashR: PGP wants -r before recipients, GPG not */
	printf("\n\n");
}

void printKeyInfo(char c,struct pkiKey *thisKey)
{
	printf("\n%c: %s <%s>\n\tType: %s\tBits: %s\tID: %s\n",c,thisKey->displayName,thisKey->emailAddress,thisKey->keyType,thisKey->keySize,thisKey->keyID);
	return;
}
