/* pgp4pine pki.c
 *
 * See common.c for the version and license.
 * Copyright (C) 1998 by Chris Wiegand
 */

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <time.h>
#include "defines.h"
#include "includes.h"
#include "structs.h"
#include "declares.h"
#include "externs.h"
#include "md5.h"

/* ---------------------------------------------------------------- */
/* Start a command in backgroup and connect STDIN/STDOUT to FILE ** */
/* ---------------------------------------------------------------- */

int pipe_pgp(const char *cmd, FILE **in, FILE **out)
{
         int pin[2], pout[2], child_pid;

         *in = *out = NULL;

         pipe(pin);
         pipe(pout);

	 documentStatus("pki.c:\n\tpipe_pgp (cmd = '");
	 documentStatus(cmd);
	 documentStatus("', FILE **in, FILE **out)\n");

         if(!(child_pid = fork()))
         {
           /* We're the child. */
           close(pin[1]);
           dup2(pin[0], 0);
           close(pin[0]);

           close(pout[0]);
           dup2(pout[1], 1);
           close(pout[1]);

	         system(cmd);
           _exit(127);
         }

         /* Only get here if we're the parent. */
         close(pout[1]);
         *out = fdopen(pout[0], "r");

         close(pin[0]);
         *in = fdopen(pin[1], "w");

         return(child_pid);
}

/* ------------------------------------------------------------ */
/* Another Start-In-Backgroup. Connects STDOUT and STDERR.      */
/* ------------------------------------------------------------ */

int pipe_pgp2(const char *cmd, FILE **out, FILE **err)
{
         int pout[2], perr[2], child_pid;

         *out = *err = NULL;

         pipe(pout);
         pipe(perr);

	 documentStatus("pki.c:\n\tpipe_pgp2 (cmd = '");
	 documentStatus(cmd);
	 documentStatus("', FILE **out, **err)\n");

         if(!(child_pid = fork()))
         {
           /* We're the child. */

           close(pout[0]);
           dup2(pout[1], 1);
           close(pout[1]);

           close(perr[0]);
           dup2(perr[1], 2);
           close(perr[1]);

	         system(cmd);
           _exit(127);
         }

         /* Only get here if we're the parent. */
         close(pout[1]);
         *out = fdopen(pout[0],"r");

         close(perr[1]);
         *err = fdopen(perr[0],"r");

         return(child_pid);
}

/* ------------------------------------------------------------ */
/* Do a signature check                                         */
/* ------------------------------------------------------------ */

int fileVerify(const char *inFile,const char *outFile)
{
  char buf[500];
  int  sigthere=0,fertig,dontprint;
  long endthere=0;
  char pgpAppString[MAX_COMMAND_LINE_LENGTH];
  int  c;
  FILE *inf, *outf, *pipeout, *pipeerr;

  /* Sometimes people manage to quote (part of) a signed mail WITHOUT using a quote sign,
     and cause us to be called here.
     My experiments show the following:
     (BEGIN always present, otherwise we aren't called):

     BEGIN present  SIGNATURE present  END present  action
  	   *		 *		     *       Strip leading and trailing text
  	   *		 *		     -       normal operation. Strip lead+trail.
  	   *		 -		     *       Error. No output.
  	   *		 -		     -       Error. No output.

     Leading+trailing text is probably vital part of message text. */
  
  documentStatus("pki.c\n\tfileVerify (inFile = '");
  documentStatus(inFile);
  documentStatus("', outFile = '");
  documentStatus(outFile);
  documentStatus("')\n");

  if ((inf = fopen(inFile,"r")) == NULL)
  { fprintf(stderr,"File error while signature check.\n"); return (-1); }

  /* Search for SIGNATURE line. If none present, do not call PGP. */
  while (!feof(inf) && !(sigthere && endthere)) {
    fgets(buf,sizeof(buf),inf);
    if (strncmp(buf,"-----BEGIN PGP SIGNATURE",24)==0) sigthere=ftell(inf);
    if (strncmp(buf,"-----END PGP SIGNATURE"  ,22)==0) endthere=ftell(inf);
  }

  if (!sigthere) {	  /* No signature found. Do nothing, but exit pgp4pine cleanly. */
    return (0);   
  }

  /* Found signature: Start PGP and puzzle output together */

  switch (myUserPrefs->pgpVersion) {
  case 5:
  	  sprintf(pgpAppString,"%s <%s", myUserPrefs->pgp5pgpv, inFile);
  	  break;
  case 2:
  case 6:
  	  sprintf(pgpAppString,"%s -f <%s", myUserPrefs->pgp2pgp, inFile);
  	  break;
  case 1:
  	  sprintf(pgpAppString,"%s --no-greeting --decrypt %s", myUserPrefs->pgp1gpg,inFile);
  	  break;  
  }
  documentStatus("\tpgpAppString = '");
  documentStatus(pgpAppString);
  documentStatus("'\n");
  pipe_pgp2(pgpAppString,&pipeout,&pipeerr);
  if ((pipeout==NULL)||(pipeerr==NULL))
  { printf("Error starting PGP while signature check."); return (-1); }

/* TODO: Signature-ID check: Unknown -> get keyserver */
	
/* Now puzzle: Leading part from Infile, De-signatured part from stdout-Pipe,
   trailing part from Infile, Signature check part from stderr-Pipe. */

  rewind(inf);
  if ((outf = fopen(outFile,"w+")) == NULL)
  { printf("File error while signature check."); return (-1); }

  fertig=0;
  do {
    if (fgets(buf,sizeof(buf),inf)) {
  	  if (strncmp(buf,"-----BEGIN PGP SIGNED",21))  fputs(buf,outf);
  	  else fertig=1;
    }
  } while (!fertig && !feof(inf));  /* EOF should not happen. But be sure. */

  do {
    if (fgets(buf,sizeof(buf),pipeout))
      fputs(buf,outf);
  } while (!feof(pipeout));

  if (endthere) fseek(inf,endthere,SEEK_SET); /* Found END SIGNATURE mark */
  else          fseek(inf,sigthere,SEEK_SET); /* else go to begin of signature */

  do {
    if (fgets(buf,sizeof(buf),inf))
       fputs(buf,outf);
  } while (!feof(inf));
  /* If the last character written was not \n, append one. */
  if (buf[strlen(buf)-1] != '\n') putc('\n',outf);

  fprintf(outf,"------------ Output from %s ------------\n",
	           myUserPrefs->pgpVersion==1?"gpg":"pgp");
  /* Strip the output from unnecessary parts while copying */
  do {
    if (fgets(buf,sizeof(buf),pipeerr)) {
      dontprint=0;
      /* Suppress some sort of uninteresting lines */
      if (buf[0]=='\n' || /* empty line */
        !strncmp(buf,"No files specified.",19) || /* PGP5 */
        !strncmp(buf,"Opening file",       12) || /* PGP5 */
        !strncmp(buf,"gpg: Warn",           9))   /* GPG 'Warning' or 'Warnung' */
       dontprint=1;
      if (!dontprint) fputs(buf,outf);
    }
  } while (!feof(pipeerr));

  fclose(outf); fclose(inf); fclose(pipeout); fclose(pipeerr);
}

/* ------------------------------------------------------------- */
/* Decrypt an encrypted mail. Then put it in the cache, if we do */
/* ------------------------------------------------------------- */

void fileDecrypt(const char *inFile,const char *outFile)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH];
	int  ret;

	documentStatus("pki.c\n\tfileDecrypt (inFile = '");
	documentStatus(inFile);
	documentStatus("', outFile = '");
	documentStatus(outFile);
	documentStatus("')\n");

	switch (myUserPrefs->pgpVersion) {
	case 5:
		sprintf(pgpAppString,"%s -o %s %s",myUserPrefs->pgp5pgpv,outFile,inFile);
		break;
	case 2:
	case 6:
		sprintf(pgpAppString,"%s %s -o %s",myUserPrefs->pgp2pgp,inFile,outFile);
		break;
	case 1:
		sprintf(pgpAppString,"%s --no-greeting --output %s --decrypt %s",myUserPrefs->pgp1gpg,outFile,inFile);
		break;	
	}
	documentStatus("\t\tpgpAppString = '");
	documentStatus(pgpAppString);
	documentStatus("'\n");
	ret = runProcessFG(pgpAppString);
  if ((ret==0) && (prefsCacheDecrypted>0)) /* Cache the new file if decoded successful */
	{
	   struct stat stats;
	   int  in,cache;
	   u_int8_t *inbuf,b;
	   u_int32_t date;
	   u_int32_t size;
	   MD5_CONTEXT ctx;
	   char tmpString[35];
	   
	   /* If anything goes wrong, just don't cache */
	   if ((in = open(inFile,O_RDONLY))<0) return;
	   if (fstat(in,&stats)<0) return;	/* stat for size */
	   if (stats.st_size==0) return;
	   
	   /* Map file to memory */
	   if ((inbuf = (u_int8_t *)mmap(0,stats.st_size, PROT_READ, MAP_SHARED,in,0))
	       == (u_int8_t *) -1) { close(in); return; }
	   /* Create a md5 sum to identify it later */
	   md5_init(&ctx);
	   md5_write(&ctx, inbuf, stats.st_size);
	   md5_final(&ctx);
	   munmap(inbuf, stats.st_size);
	   close(in); 

	   if ((in = open(outFile,O_RDONLY))<0) return;
	   if (fstat(in,&stats)<0) return;
	   if ((inbuf = (u_int8_t *)mmap(0,stats.st_size, PROT_READ, MAP_PRIVATE,in,0))
	       == (u_int8_t *) -1) { close(in); return; }
	 
	   /* and create the cache entry */
	   documentStatus("Creating cache entry MD5: ");
	   sprintf(tmpString,"%x%x%x%x%x%x%x%x %x%x%x%x%x%x%x%x\n",
 		ctx.buf[0],ctx.buf[1],ctx.buf[2],ctx.buf[3],
		ctx.buf[4],ctx.buf[5],ctx.buf[6],ctx.buf[7],
		ctx.buf[8],ctx.buf[9],ctx.buf[10],ctx.buf[11],
		ctx.buf[12],ctx.buf[13],ctx.buf[14],ctx.buf[15]);
	   documentStatus(tmpString);
	   if ((cache = open(DecryptCacheName,O_WRONLY|O_CREAT|O_APPEND,0600))<0) return;

	   write(cache,"\x12\x34",2); /* magic 0x1234 */
	   size = stats.st_size+cacheHeaderSize;
	   size = to_bigend_u32(size);
	   write(cache,&size,4);                /* size */
	   b=CACHE_VERSION; write(cache,&b,1);  /* version */
	   b=0;             write(cache,&b,1);  /* Characterize as message entry */
	   write(cache,ctx.buf,16);             /* MD5 */
	   date = (time(0)/60);                 /* minute resolution */
	   date = to_bigend_u32(date);
	   write(cache,&date,4);
	   write(cache,"  ",2);			/* Two dummy bytes (unused) */
	   write(cache, inbuf, stats.st_size);
	   close(cache);
	   munmap(inbuf, stats.st_size);
	   close(in);
	}
}

/* ------------------------------------------ */
/* Throw out old messages in cache.           */
/* This is called every time pgp4pine starts. */
/* ------------------------------------------ */

#define headerstring "----- pgp4pine " VERSION " cached encrypted file -----\n\n"
#define cacheversion 1

void checkCache()
{
	int cache;		/* Parse the cache file. */
	int flush;		/* Entry format see structs.h */ 
	u_int32_t today,date;
	struct stat stats;
	off_t  size,newsize;
	u_int8_t *buf,*bufstart;
	u_int32_t entrysize;
	
	if ((cache=open(DecryptCacheName,O_RDWR))<0) return;
	if (fstat(cache,&stats)<0) return;   /* stat for size */
	newsize = size = stats.st_size;      /* copy length and use as counter */
	if ((buf = bufstart = (u_int8_t *)mmap(0,stats.st_size, 
	    PROT_READ|PROT_WRITE,MAP_SHARED,cache,0))== (u_int8_t *) -1) return;
	
	while(size>0) { 
	   flush=0;
	   if ((size<cacheHeaderSize)                 /* size smaller than header size => corrupt */
              || (buf[0] != 0x12) || (buf[1] != 0x34) /* no magic: integrity violated or */
	      || (buf[6] != CACHE_VERSION))           /* another version => delete the file */
	   {
	     close(cache);
	     remove(DecryptCacheName);
	     return;
	   }
	   date  = (buf[24]<<24)|(buf[25]<<16)|(buf[26]<<8)|buf[27];
           today = time(0)/60;
	   if ((buf[7] == 0)        /* A message entry */
	      && (today >=  (date + prefsCacheDecrypted)))
	        flush=1;                                     /* too old */
	   else if (((buf[7] == 1) || (buf[7] == 2))
	      && (today >= (date + prefsCacheKeyrings)))
	        flush=1;
	   entrysize = (buf[2]<<24)|(buf[3]<<16)|(buf[4]<<8)|buf[5];
	   
	   size -= entrysize;
	   if (size>=0) /* if size<0, size integrity was violated */
	   {
	     if (flush==1)  /* do a flush: Copy from next msg until EOF */
	     {
	        if (size>0) {
		   memmove(buf,buf + entrysize, size);
		   memset (buf+size,0,entrysize);  /* Zero out rest of file */
                }
	        newsize -= entrysize;	        
	     }
	     else buf += entrysize;
	   }
	} 
	munmap(bufstart, stats.st_size);
	
	if ((newsize < stats.st_size) && newsize>0) 
	  /* If we flushed, truncate file */
	  ftruncate(cache, newsize);
        close(cache);
        if (newsize<=0) remove(DecryptCacheName);
          /* <0 must be an error. Kill either way .. */
}

/* ------------------------------------------------------------- */
/* Try to find a Cleartext version of <inFile> in the cache.     */
/* Write to <outFile> if successful and return 1, else return 0. */
/* ------------------------------------------------------------- */

int readCache(const char *inFile, const char *outFile)
{
	int i,found;
	int in,out,cache;
	struct stat instats,cachestats;
	u_int8_t *inBuf, *cacheBuf, *cacheBufstart;
	MD5_CONTEXT ctx;
	u_int32_t entrysize;
	
	if (((in=open(inFile,O_RDONLY))<0)
	 || ((cache=open(DecryptCacheName,O_RDONLY))<0)) return (0);
	if (fstat(in,&instats)<0) return(0); /* stat for size */
	if (fstat(cache,&cachestats)<0) return(0); /* stat for size */
	if ((inBuf = (u_int8_t *)mmap(0,instats.st_size, PROT_READ,MAP_SHARED,in,0))
	    == (u_int8_t *) -1) return(0);
	if ((cacheBuf = (u_int8_t *)mmap(0,cachestats.st_size, PROT_READ,MAP_PRIVATE,cache,0))
	    == (u_int8_t *) -1) return(0);
	cacheBufstart = cacheBuf; /* Will count up pointer, so save it */
	    
	md5_init(&ctx);
	md5_write(&ctx, inBuf, instats.st_size);
	md5_final(&ctx);
	munmap(inBuf, instats.st_size);
	close(in);
	
	do {
	   found=1;
	   for (i=0; found && (i<16); i++) 
	      if (ctx.buf[i]!=cacheBuf[i+8]) found=0;
	   entrysize = (cacheBuf[2]<<24)|(cacheBuf[3]<<16)|(cacheBuf[4]<<8)|cacheBuf[5];
	      
	   if (!found) cacheBuf+= entrysize;
	} while (!found && (cacheBuf<cacheBufstart+cachestats.st_size));
	if (found)
	{
	   if ((out=open(outFile,O_CREAT|O_WRONLY|O_TRUNC,0600))<0)
	     return 0; /* Error while opening */
           write(out,headerstring,sizeof(headerstring)-1); /* write without trailing \0 */
	   write(out,cacheBuf+cacheHeaderSize,entrysize-cacheHeaderSize);
           
	   close(out);
	   munmap(cacheBufstart,cachestats.st_size);
	   close(cache);
	   return(1);
	} else {
	   munmap(cacheBufstart,cachestats.st_size);
	   close(cache);
	  return(0);
	}
}

/* ------------------------------------------------------------- */
/* Sign or encrypt a message from inFileName to outFileName      */
/* ------------------------------------------------------------- */

int fileSignEncrypt(const char *inFileName,
		     const char *outFileName,
		     int sign, int encrypt,
		     char *recipient, 
		     int asciiArmor,   int universalText,
		     int encryptToSelf,int detachSigs)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH], *commentString;
	char pgpOptions[MAX_COMMAND_LINE_LENGTH], recipList[EMAIL_ADDRESS_MAX_LENGTH];
        int  exitcode;
	
	documentStatus("pki.c\n\tfileSignEncrypt (inFile = '");
	documentStatus(inFileName);
	documentStatus("', outFile = '");
	documentStatus(outFileName);
	documentStatus("', sign = ");
	if (sign == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", encrypt = ");
	if (encrypt == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", recipient = '");
	documentStatus(recipient);
	documentStatus(", asciiArmor = ");
	if (asciiArmor == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", universalText = ");
	if (universalText == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", encryptToSelf = ");
	if (encryptToSelf == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(", detachSigs = ");
	if (detachSigs == 1) documentStatus("1"); else documentStatus("0");
	documentStatus("')\n");

	strcpy(pgpOptions," ");
	strcpy(recipList," ");
	strcat(pgpOptions,myUserPrefs->extraOptions);

	if (encryptToSelf > 0) {
		if (strlen(myUserPrefs->myAddress) > 3) {
			strcat(recipient," ");
			if (addDashR == 1) strcat(recipient,"-r ");
			strcat(recipient,myUserPrefs->myAddress);
			strcat(recipient," ");
		} else {
			if (myUserPrefs->pgpVersion != 5) printf("Because you didn't specify your email address in ~/.pgp4pinerc,\nI can't auto-encrypt to you as well.\n");
			encryptToSelf = -1;
		}
	}

	switch (myUserPrefs->pgpVersion) {
	case 5:
		if (detachSigs > 0) strcat(pgpOptions," -b ");
		/* get recipList... */
		if (encrypt > 0) {
			strcpy(recipList,recipient);
 			if (sign > 0) strcat(pgpOptions," -s ");
		}
		if (sign > 0 && strlen(mySecretKey) > 3) {
 				strcat(pgpOptions," -u \"");
 				strcat(pgpOptions,mySecretKey);
 				strcat(pgpOptions,"\" ");
		}
		if (asciiArmor > 0) strcat(pgpOptions," -a ");
		if (universalText > 0) strcat(pgpOptions," -t ");
		if (prefsAddComment > 0)
			commentString="--comment=\"Made with pgp4pine " VERSION "\"";
		else
			commentString = "";

		if (encrypt > 0)
			sprintf(pgpAppString,"%s %s %s %s -o %s %s",myUserPrefs->pgp5pgpe,pgpOptions,commentString,recipList,outFileName,inFileName);
		else
			sprintf(pgpAppString,"%s %s %s -o %s %s",myUserPrefs->pgp5pgps,pgpOptions,commentString,outFileName,inFileName);
		exitcode = runProcessFG(pgpAppString);
		break;
	case 2:
	case 6:
		strcat(pgpOptions," -"); /* we either have to add a e[s] or just s... */
		if (encrypt > 0) {
			strcat(pgpOptions,"e");
			strcpy(recipList,recipient);
		}
		if (sign > 0) {
			strcat(pgpOptions,"s");
			if (detachSigs > 0) strcat(pgpOptions,"b");
		}
		if (universalText > 0) strcat(pgpOptions,"t");
		if (asciiArmor > 0) strcat(pgpOptions,"a");
		strcat(pgpOptions," ");
		if (sign > 0 && strlen(mySecretKey) > 3) {
			strcat(pgpOptions," -u \"");
			strcat(pgpOptions,mySecretKey);
			strcat(pgpOptions,"\" ");
		}
		if (prefsAddComment > 0) 
			commentString = "+COMMENT=\"Made with pgp4pine " VERSION "\"";
		else
			commentString = "";

		if (encrypt > 0)
			sprintf(pgpAppString,"%s %s %s %s %s",myUserPrefs->pgp2pgp,pgpOptions,commentString,inFileName,recipList);
		else
			sprintf(pgpAppString,"%s %s %s %s",myUserPrefs->pgp2pgp,pgpOptions,commentString,inFileName);
		exitcode = runProcessFG(pgpAppString); 
		
		/* now, we move the <infile>.pgp to <outfile>*/
		if (detachSigs > 0) {
			/* we're detaching a signature, the original file isn't changed.... */
		} else {
			if (asciiArmor > 0)
				sprintf(pgpAppString,"mv %s.asc %s",inFileName,outFileName);
			else
				sprintf(pgpAppString,"mv %s.pgp %s",inFileName,outFileName);
			runProcessFG(pgpAppString);
		}
		break;
	case 1:
		if (myUserPrefs->asciiArmor > 0) strcat(pgpOptions, " -a ");
		if (myUserPrefs->universalText > 0) strcat(pgpOptions, " -t ");

		if (encrypt > 0) {
			strcat(pgpOptions," -e ");
			strcpy(recipList,recipient);
		}
		if (sign > 0) {
			if (detachSigs > 0) strcat(pgpOptions," -b ");
			strcat(pgpOptions," -s ");
			if (encrypt == 0) strcat(pgpOptions," --clearsign ");
			if (strlen(mySecretKey) > 3) {
				strcat(pgpOptions," -u \"");
				strcat(pgpOptions,mySecretKey);
				strcat(pgpOptions,"\" ");
			}
		}
		/* if we're not encrypting, recipList is just a " ", so we're safe... */
		/* Just moved outfilename to begining of list per email from henrik kg andreasson */
		if (prefsAddComment > 0) 
			commentString = "--comment \"Made with pgp4pine " VERSION "\"";
		else
			commentString = "";
			
		sprintf(pgpAppString,"%s -o %s --no-batch %s --no-greeting %s %s %s",
		    myUserPrefs->pgp1gpg,outFileName,commentString,recipList,pgpOptions,inFileName);
		sprintf(debugLine,"DEBUG: GnuPG Run: \"%s\"\n",pgpAppString);
		documentStatus(debugLine);
		exitcode=runProcessFG(pgpAppString);
		break;
	}
	/* if not yet broken, and clearsign wanted, check if it really worked */
	if ((exitcode>=0) && (sign>0) && (encrypt<0) && (asciiArmor>0)) { 
		char buf[500];
		FILE *outf;
		int  ok=0;
		
  		if ((outf = fopen(outFileName,"r")) == NULL)
		{ fprintf(stderr,"File error while signing.\n"); return(-1); }
		while (!feof(outf) && (ok==0)) {
		   fgets(buf,sizeof(buf),outf);
		   if      (strncmp(buf,"-----BEGIN PGP SIGNED",21)==0)  ok= 1; /* Everything OK */
		   else if (strncmp(buf,"-----BEGIN PGP MESSAGE",22)==0) ok=-1; /* Did not work !! */
		}
		fclose(outf);
		if (ok==-1) {
		   printf("ATTENTION: You wanted a clearsigned message. PGP generated an armored message,\n"
		          "which needs your recipient(s) to have PGP installed to read it.\n"
			  "This usually happens when the message contains 'special' characters.\n\n"
			  "a) Then send it unsigned\nb) Send anyway\nq) Return to Pine\n");
		   switch (askAlphaRange("What do you want to do ",'b','a','q'))
		   {
		     case 'a': return 1;   /* Signals unmodified */
		     case 'b': return 0;
		     case 'q': return 2;   /* Signals error, but don't throw out 'error while encrypting' */
		   }
		}
	}
	return exitcode;
}

/* ------------------------------------- */
/* Get unknown keys from keyserver       */
/* ------------------------------------- */

void fileAskKey(char *nokeyrecipients)
{
	char pgpAppString[MAX_COMMAND_LINE_LENGTH];
	char tmpEmail[EMAIL_ADDRESS_MAX_LENGTH];
	int  i=0;
	
	if (myUserPrefs->pgpVersion == 2) {
	  printf("Who the hell called this routine ? Anyway, PGP 2 cannot get keys from a keyserver.\n");
	} else {
	  documentStatus("Asking Keyserver for unknown keys");
	
	  do {
		if (i==0) strcpy(tmpEmail,strtok(nokeyrecipients," "));
		else strcpy(tmpEmail,strtok('\0'," "));
		if (strcmp(tmpEmail,".")) {
			if (strlen(tmpEmail) > 1)
			{ 
			  if (myUserPrefs->pgpVersion == 5)
			    sprintf(pgpAppString,"%s -a \"%s\"",myUserPrefs->pgp5pgpk,tmpEmail);
			  else
			    sprintf(pgpAppString,"%s --recv-keys \"%s\"",myUserPrefs->pgp1gpg,tmpEmail);

			  documentStatus(pgpAppString);
			  runProcessFG(pgpAppString);
			}
		}
		i++;
	  } while(strcmp(tmpEmail,"."));
	}
}

/* -------------------------------------------------------------------------- */
/* Parse Recipient list *recipients and correct ambiguous keys.               */
/* Return is have-key list *recipients and have-no-key list *nokeyrecipients. */
/* -------------------------------------------------------------------------- */

void parseRecipientList(char *recipients, char *nokeyrecipients)
{
	char tmpEmail[EMAIL_ADDRESS_MAX_LENGTH];
	char recipList[EMAIL_ADDRESS_MAX_LENGTH], nokeyList[EMAIL_ADDRESS_MAX_LENGTH];
	int i=0;

	strcpy(recipList,"");
	strcpy(nokeyList,"");
	
	/* recipients is passed back, so it's a pointer */
	strcat(recipients," . "); /* we add a space so that anything at the end is an item */
	/* if you don't, strtok() can segfault because there's no terminating " "... */
	documentStatus("pki.c\n\tparseRecipientList (recipients = '");
	documentStatus(recipients);
	documentStatus("', addDashR = ");
	if (addDashR == 1) documentStatus("1"); else documentStatus("0");
	documentStatus(")\n");

 	do {
		if (i==0) strcpy(tmpEmail,strtok(recipients," "));
		else strcpy(tmpEmail,strtok('\0'," "));
		if (strcmp(tmpEmail,".")) {
			if (strlen(tmpEmail) > 1)
			{
/*			  printf(tmpEmail); */
			  if (checkRecipient(tmpEmail) == 1) 
			  {
			     printf(" - key %s found\n",tmpEmail);
			     if (addDashR == 1) strcat(recipList," -r");
			     strcat(recipList," \"");
			     strcat(recipList,tmpEmail);
			     strcat(recipList,"\" ");
			  } else {/* not found */
			     printf(" - no key found\n");
			     strcat(nokeyList,"\"");
			     strcat(nokeyList,tmpEmail);
			     strcat(nokeyList,"\" ");
			  }
			}
		}
		i++;
	} while(strcmp(tmpEmail,"."));
	strcpy(recipients,recipList);
	strcpy(nokeyrecipients,nokeyList);
	documentStatus("Recipients: with key: ");
	documentStatus(recipients);
	documentStatus(" without key: ");
	documentStatus(nokeyrecipients);
	documentStatus("\n");
	strcat(nokeyrecipients," . "); /* Terminate list */
}

/* This just checks a single recipient, when we parse the recipient list, we check each one */

int checkRecipient(char *thisRecip)
{
	struct pkiKey *thisKey = NULL;
	struct pkiKey *lastKey = NULL;
	int found = 0;

	documentStatus("pki.c\n\tcheckRecipient (thisRecip = '");
	documentStatus(thisRecip);
	documentStatus("')\n");
	printf("%s... ",thisRecip);

	found = 0;
	if (publicKeyring != NULL) {
		thisKey = publicKeyring;
		lastKey = thisKey;
		while (thisKey != NULL) {
			if (strcasecmp(thisRecip,thisKey->emailAddress) == 0) {
				found++;
				lastKey = thisKey;
			}
			thisKey = thisKey->nextKey;
		}
	}

	if (found < 1) {   /* No key found */
		documentStatus("\tNo key found for ");documentStatus(thisRecip);
		documentStatus("\n");
	} else if (found > 1) {
		int i=0,j=0,k=1;
		/* wow..two keys... */
		printf("\n\nThe following recipient has multiple keys. Please select one.\n");
		thisKey = publicKeyring; /* set back to beginning */
		do {
			if (strcasecmp(thisKey->emailAddress,thisRecip) == 0) {
				i++; /* number of able and ready keys... */
				printKeyInfo((char) ('a'+i-1),thisKey);
			}
			thisKey = thisKey->nextKey;
		} while (thisKey != NULL);
		j = askAlphaRange("Please select the recipient key to use ",(i+'a'-1),'a',0)-'a';
		thisKey = publicKeyring;
		k=0;
		while (k<=j) {
			documentStatus("\t\tk vs. j loop:\n\t\t\tthisKey->emailAddress = ");
			documentStatus(thisKey->emailAddress);
			documentStatus("\n\t\t\tthisKey->keyID = ");
			documentStatus(thisKey->keyID);
			documentStatus("\n");
			if (strcasecmp(thisKey->emailAddress,thisRecip) == 0) {
				k++;
				documentStatus("\t\t\tKey Matched!\n");
			}
			if (k <= j) thisKey = thisKey->nextKey;
		}
#ifdef USE_DISPLAY_NAME
		strcpy(thisRecip,thisKey->emailAddress);
#else
		strcpy(thisRecip,thisKey->keyID);
#endif
		documentStatus("\t\tThisRecip: ");
		documentStatus(thisRecip);
		documentStatus("\n");
		found = 1;	/* One key found */
	} else {
/*		printf("found (%s)\n",lastKey->keyID); */
		/* only one key for this recipient */
#ifdef USE_DISPLAY_NAME
		strcpy(thisRecip,lastKey->emailAddress);
#else
		strcpy(thisRecip,lastKey->keyID);
#endif
	}
	return found;
}

void parseList(char *recipients, char *nokeyrecipients)
{
	printf("\nChecking recipients ...\n\n");
	parseRecipientList(recipients,nokeyrecipients);
	/* addDashR: PGP wants -r before recipients, GPG not */
	printf("\n\n");
}


void printKeyInfo(char c,struct pkiKey *thisKey)
{
	printf("\n%c: %s <%s>\n\tType: %s\tBits: %s\tID: %s\n",c,
	   thisKey->displayName, thisKey->emailAddress,
	   thisKey->keyType,thisKey->keySize,thisKey->keyID);
	return;
}
