/*  pgpdaemon.c

    Source file for  PGPdaemon  (automatic signer/encryptor for PGP).

    Copyright (C) 1994-1999  Richard Gooch

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Richard Gooch may be reached by email at  rgooch@atnf.csiro.au
    The postal address is:
      Richard Gooch, c/o ATNF, P. O. Box 76, Epping, N.S.W., 2121, Australia.
*/

/*  This programme intercepts messages sent by user mail agents to the
    sendmail daemon and checks to see if messages can be encrypted using the
    recipient's PGP public keys.


    Written by      Richard Gooch   4-JUN-1994

    Updated by      Richard Gooch   5-JUN-1994

    Updated by      Richard Gooch   10-JUN-1994: Changed from using  stricmp
  to  st_icmp  ,since some platforms don't have  stricmp  .

    Updated by      Richard Gooch   20-JUN-1994: Added mailbox decrypting
  functionality.

    Updated by      Richard Gooch   21-JUN-1994: Added  #include <sys/types.h>

    Updated by      Richard Gooch   22-JUN-1994: Added IN_SPOOL_DIR  and
  OUT_SPOOL_FILE  keywords for config file.

    Updated by      Richard Gooch   23-JUN-1994: Added MAIL_CHECK_INTERVAL
  keyword for config file.

    Updated by      Richard Gooch   23-JUN-1994: Added  #ifdef _AIX  and made
  set_lockfile  a little more verbose if lockfiles can't be created.

    Updated by      Richard Gooch   27-JUN-1994: Moved  set_env  to
  spawn.c  and added  -detach  and  -pgppath  options.

    Updated by      Richard Gooch   28-JUN-1994: Improved logging information
  when processing mail.

    Updated by      Richard Gooch   1-JUL-1994: Used [-u userID] option to PGP
  when signing messages.

    Updated by      Richard Gooch   2-JUL-1994: Trap unterminated PGP messages.

    Updated by      Richard Gooch   2-JUL-1994: Truncate input mail spool
  rather than removing, since  sendmail  under OSF/1 will create the spoolfile
  even if the lockfiles exist, it just waits for the lockfiles to go before it
  starts writing to the spoolfile. This caused some messages to be lost, since
  the file  sendmail  would eventually write to was unlinked (ie. only
  sendmail  had a handle to the file it was writing to).

    Updated by      Richard Gooch   3-JUL-1994: Added some imformative
  messages when decryption of mail fails.

    Updated by      Richard Gooch   3-JUL-1994: Recover from unterminated PGP
  messages so that subsequent messages can be decrypted.

    Updated by      Richard Gooch   3-JUL-1994: Trap when PGP stops when
  decrypting mail. Usually because message encrypted "For-your-eyes-only".

    Updated by      Richard Gooch   4-JUL-1994: Added  #ifdef O_SYNC  and
  #ifdef __bsdi__  for MAIL_SPOOL_DIR  .

    Updated by      Richard Gooch   5-JUL-1994: Changed to use of  m_clear  .

    Updated by      Richard Gooch   11-JUL-1994: Added check for writability of
  incoming spool directory

    Uupdated by     Richard Gooch   11-JUL-1994: Use MAIL environment variable
  in preference to hard-coded default if  IN_SPOOL_DIR  option not defined.

    Updated by      Richard Gooch   12-JUL-1994: Added support for leading '~'
  in filenames in config file.

    Updated by      Richard Gooch   13-JUL-1994: Limited core dumps to 0 bytes.

    Updated by      Richard Gooch   14-JUL-1994: Added flag response with
  connecting.

    Updated by      Richard Gooch   14-JUL-1994: Added calls to  fsync(2)
  after creating lockfiles.

    Updated by      Richard Gooch   16-JUL-1994: Added hostname to socket
  and log filenames and added secret keyring creation/deletion.

    Uupdated by     Richard Gooch   28-JUL-1994: Dump secret keyring when
  SIGUSR1 is caught.

    Updated by      Richard Gooch   3-AUG-1994: Added  -nomail  option.

    Updated by      Richard Gooch   25-AUG-1994: Fixed descriptor leak in
  write_secring  .

    Updated by      Richard Gooch   26-AUG-1994: Pass  check_mail  flag to
  read_config  so that writability of spool file is not checked when the
  -nomail  option was given.

    Updated by      Richard Gooch   11-NOV-1994: Changed from aborting if
  file locking fails to waiting until file locks released. Added
  -no_filelocking  option.

    Updated by      Richard Gooch   17-NOV-1994: Removed extraneous file
  unlocking in  check_and_process_mail  .

    Updated by      Richard Gooch   3-DEC-1994: Changed to pair of named pipes
  and PID file rather than listening on a Unix domain socket.

    Updated by      Richard Gooch   4-DEC-1994: Added  +batchmode  option to
  PGP when decrypting.

    Updated by      Richard Gooch   9-DEC-1994: Create pipes if they don't
  exist.

    Updated by      Richard Gooch   9-DEC-1994: Changed from  rand48(3)  to
  random(3)

    Updated by      Richard Gooch   4-JAN-1995: Changed from  random(3)  to
  rand(3)  because Slowass 2 doesn't always have  random(3)  .

    Updated by      Richard Gooch   6-MAY-1995: Closed stdin to prevent
  capturing of tty (problem occurred under Linux 1.2.7, libc 5.0.5).

    Updated by      Richard Gooch   1-SEP-1995: Added #ifdef _AIX for
  #include <sys/select.h>

    Updated by      Richard Gooch   25-SEP-1997: Used new ERRSTRING macro.
  Improved child process cleanup.

    Updated by      Richard Gooch   30-SEP-1997: Fixed ASCII protocol: expect
  trailing '\0' character after ASCII message. Bug fix in child process
  cleanup: sometimes PGPdaemon killed itself! Added "BINARY_DECRYPT" protocol.

    Updated by      Richard Gooch   2-OCT-1997: Use paths.h instead of my own
  hardcoded values.

    Updated by      Richard Gooch   3-OCT-1997: Reset timing interval when
  SIGUSR2 is caught or when mail is read to be processed.

    Updated by      Richard Gooch   8-OCT-1997: Supported $env as USERID
  configuration option.

    Updated by      Richard Gooch   9-OCT-1997: Made use of new function
  <st_expr_expand>.

    Updated by      Richard Gooch   21-OCT-1997: Conditionally include paths.h

    Updated by      Richard Gooch   27-OCT-1997: Fixed bug if no logfile
  specified in configuration file.

    Updated by      Richard Gooch   10-JUL-1998: Removed definitions of system
  errlist array.

    Last updated by Richard Gooch   13-FEB-1999: Added regular inversion of
  passphrase


*/
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/mman.h>
#include <signal.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#ifdef _AIX
#  include <sys/select.h>
#endif
#include <string.h>
#include <errno.h>
#ifndef NO_PATH_H
#  include <paths.h>
#endif
#include <pgpsendmail.h>
#include <version.h>


#define MIN_INTERVAL 1
#define DEFAULT_INTERVAL 15
#define MAX_INTERVAL 60

STATIC_FUNCTION (flag service_connect,
		 (CONST char *inpipe_filename, CONST char *outpipe_filename) );
STATIC_FUNCTION (void sig_handler, () );
STATIC_FUNCTION (flag process_message,
		 (int in_fd, int out_fd, flag sign, flag ascii) );
STATIC_FUNCTION (void myexit, (int code) );
STATIC_FUNCTION (flag set_lockfile, (char *filename, flag lock) );
STATIC_FUNCTION (flag set_fd_lock, (int fd, flag lock, flag no_filelocking) );
STATIC_FUNCTION (flag check_and_process_mail, () );
STATIC_FUNCTION (void read_config, (char *pgppath, flag *mailwait,
				    long *interval, flag check_mail) );
STATIC_FUNCTION (flag write_secring, () );
STATIC_FUNCTION (flag scrub_secring, () );
STATIC_FUNCTION (int get_rand, () );
STATIC_FUNCTION (void invert_passphrase, () );


/*  Private data  */
static char *pgppath = NULL;
static char passphrase[STRING_LENGTH];
static SECRING_SIZE_TYPE secring_bytes = 0;
static char *secring_buf = NULL;
static flag keep_going = TRUE;
static flag restart = FALSE;
static flag incoming_locked = FALSE;
static flag insecure = TRUE;
static flag dump_secring = FALSE;
static flag caught_sigusr2 = FALSE;
static flag check_mail = TRUE;
static flag no_filelocking = FALSE;
static flag caught_sigio = FALSE;
static char incoming_spool[STRING_LENGTH];
static char outgoing_spool[STRING_LENGTH];
static char my_userid[STRING_LENGTH];
static char logfile[STRING_LENGTH];
static char hostname[STRING_LENGTH];
static char xor_char = 0x1a;


/*  Code for  myexit  must come first.  */

static void myexit (code)
/*  This routine will first scrub sensitive information prior to calling the
    exit(3) function.
    The exit code must be given by  code  .
*/
int code;
{
    extern flag insecure;
    extern SECRING_SIZE_TYPE secring_bytes;
    extern char passphrase[STRING_LENGTH];
    extern char *secring_buf;

    if (insecure) exit (code);
    fprintf (stderr, "Erasing sensitive information...\n");
    m_clear (passphrase, STRING_LENGTH);
    write_secring ();
    if (secring_bytes > 0) m_clear (secring_buf, secring_bytes);
    sync ();
    exit (code);
#undef exit
#define exit ___illegal_use_of_exit___
}   /*  End Function myexit  */


/*  Now everything else may follow.  */

void main (argc, argv, envp)
int argc;
char *argv[];
char *envp[];
{
    FILE *fp;
    flag mailwait = FALSE;
    flag login_session = TRUE;
    int ppid;
    int sock, fd;
    int tmp, count;
    long icount = 0;
    long max_interval = DEFAULT_INTERVAL;
    long current_interval = 1;
    fd_set input_fds, output_fds, exception_fds;
    struct timeval timeout;
    struct sigaction new_action;
    struct rlimit rlp;
    struct stat statbuf;
    char pid_filename[STRING_LENGTH];
    char inpipe_filename[STRING_LENGTH];
    char outpipe_filename[STRING_LENGTH];
    char txt[LINE_LENGTH];
    extern flag keep_going;
    extern flag restart;
    extern flag incoming_locked;
    extern flag insecure;
    extern flag dump_secring;
    extern flag check_mail;
    extern flag no_filelocking;
    extern flag caught_sigio;
    extern SECRING_SIZE_TYPE secring_bytes;
    extern char *pgppath;
    extern char passphrase[STRING_LENGTH];
    extern char my_userid[STRING_LENGTH];
    extern char incoming_spool[STRING_LENGTH];
    extern char outgoing_spool[STRING_LENGTH];
    extern char hostname[STRING_LENGTH];
    static char usage_string[] =
    "Usage:\tpgpdaemon [-mailwait] [-detach] [-pgppath <path>]";

    if (close (0) != 0)
    {
	fprintf (stderr, "Error closing standard input\t%s\n", ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    /*  Read arguments  */
    for (count = 1; count < argc; ++count)
    {
	if (strcmp (argv[count], "-mailwait") == 0) mailwait = TRUE;
	else if (strcmp (argv[count], "-detach") == 0) login_session = FALSE;
	else if (strcmp (argv[count], "-pgppath") == 0)
	{
	    if (++count >= argc)
	    {
		fprintf (stderr, "%s\n", usage_string);
		myexit (RV_BAD_PARAM);
	    }
	    if (set_env ("PGPPATH", argv[count]) != 0)
	    {
		fprintf (stderr,
			 "Error setting PGPPATH environment variable\n");
		myexit (RV_UNDEF_ERROR);
	    }
	}
	else if (strcmp (argv[count], "-nomail") == 0) check_mail = FALSE;
	else if (strcmp (argv[count],
			 "-no_filelocking") == 0) no_filelocking = TRUE;
	else if (strcmp (argv[count], "-uid") == 0)
	{
	    uid_t new_uid;

	    if (++count >= argc)
	    {
		fprintf (stderr, "%s\n", usage_string);
		myexit (RV_BAD_PARAM);
	    }
	    new_uid = atoi (argv[count]);
	    if (setuid (new_uid) == -1)
	    {
		fprintf (stderr,
			 "Error setting UID to: %d\t%s\n", new_uid, ERRSTRING);
		myexit (RV_SYS_ERROR);
	    }
	}
	else
	{
	    fprintf (stderr, "%s\n", usage_string);
	    myexit (RV_BAD_PARAM);
	}
    }
    if ( ( ppid = getppid () ) < 0 )
    {
	fprintf (stderr, "Error getting parent process ID\t%s\n", ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    if (ppid == 1)
    {
	/*  Init was parent!  */
	fprintf (stderr, "Cannot operate with  init  as parent process\n");
	myexit (RV_UNDEF_ERROR);
    }
    if ( ( pgppath = getenv ("PGPPATH") ) == NULL )
    {
	fprintf (stderr, "No PGPPATH environment variable\n");
	myexit (RV_UNDEF_ERROR);
    }
    umask ( ~(S_IRUSR | S_IWUSR) );
    m_clear (passphrase, STRING_LENGTH);
    new_action.sa_handler = sig_handler;
    sigemptyset (&new_action.sa_mask);
    new_action.sa_flags = 0;
#ifdef SA_RESTART
    new_action.sa_flags |= SA_RESTART;
#endif
    /*  Signals to be restarted  */
    sigaction (SIGINT, &new_action, (struct sigaction *) NULL);
    sigaction (SIGTERM, &new_action, (struct sigaction *) NULL);
    sigaction (SIGHUP, &new_action, (struct sigaction *) NULL);
    sigaction (SIGPIPE, &new_action, (struct sigaction *) NULL);
    sigaction (SIGTTIN, &new_action, (struct sigaction *) NULL);
    sigaction (SIGTTOU, &new_action, (struct sigaction *) NULL);
    sigaction (SIGUSR1, &new_action, (struct sigaction *) NULL);
    sigaction (SIGUSR2, &new_action, (struct sigaction *) NULL);
    sigaction (SIGIO, &new_action, (struct sigaction *) NULL);
    /*  Signals to be restored to the default once caught  */
#ifdef SA_ONESHOT
    new_action.sa_flags |= SA_ONESHOT;
#endif
#ifdef SA_RESETHAND
    new_action.sa_flags |= SA_RESETHAND;
#endif
    sigaction (SIGSEGV, &new_action, (struct sigaction *) NULL);
#ifdef RLIMIT_CORE
    /*  Prevent core dumps  */
    rlp.rlim_cur = 0;
    rlp.rlim_max = 0;
    if (setrlimit (RLIMIT_CORE, &rlp) != 0)
    {
	fprintf (stderr, "Error limiting core dump size\t%s\n", ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
#endif
    if (set_env ("PGPPASSFD", "0") != 0)
    {
	fprintf (stderr, "Error setting PGPPASSFD environment variable\n");
	myexit (RV_UNDEF_ERROR);
    }
    /*  Sanity checks for files the user should have  */
    sprintf (txt, "%s/randseed.bin", pgppath);
    if (access (txt, R_OK) != 0)
    {
	fprintf (stderr, "Could not find file: \"%s\"\t%s\n", txt, ERRSTRING);
	myexit (RV_UNDEF_ERROR);
    }
    if (gethostname (hostname, STRING_LENGTH - 1) != 0)
    {
	fprintf (stderr, "Error getting hostname\t%s\n", ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    hostname[STRING_LENGTH - 1] = '\0';
    read_config (pgppath, &mailwait, &max_interval, check_mail);
    /*  Worry about communication files  */
    sprintf (pid_filename, "%s/.pgpd.PID.%s", pgppath, hostname);
    sprintf (inpipe_filename, "%s/.pgpd.input.%s", pgppath, hostname);
    sprintf (outpipe_filename, "%s/.pgpd.output.%s", pgppath, hostname);
    if (access (pid_filename, F_OK) == 0)
    {
	fprintf (stderr, "File: \"%s\" exists:\n", pid_filename);
	fprintf (stderr, "PGPdaemon did not clean up last time\n");
	myexit (RV_UNDEF_ERROR);
    }
    if (errno != ENOENT)
    {
	fprintf (stderr, "Error testing for PID file: \"%s\"\t%s\n",
		 pid_filename, ERRSTRING);
	myexit (RV_UNDEF_ERROR);
    }
    if (stat (inpipe_filename, &statbuf) == 0)
    {
	if ( !S_ISFIFO (statbuf.st_mode) )
	{
	    fprintf (stderr, "File: \"%s\" is not a named pipe\n",
		     inpipe_filename);
	    myexit (RV_SYS_ERROR);
	}
	chmod (inpipe_filename, S_IRUSR | S_IWUSR);
    }
    else
    {
	if (errno != ENOENT)
	{
	    fprintf (stderr, "Error stat'ing file: \"%s\"\t%s\n",
		     inpipe_filename, ERRSTRING);
	    myexit (RV_SYS_ERROR);
	}
	if (mkfifo (inpipe_filename, S_IRUSR | S_IWUSR) != 0)
	{
	    fprintf (stderr, "Error creating pipe: \"%s\"\t%s\n",
		     inpipe_filename, ERRSTRING);
	    myexit (RV_SYS_ERROR);
	}
    }
    if (stat (outpipe_filename, &statbuf) == 0)
    {
	if ( !S_ISFIFO (statbuf.st_mode) )
	{
	    fprintf (stderr, "File: \"%s\" is not a named pipe\n",
		     outpipe_filename);
	    myexit (RV_SYS_ERROR);
	}
	chmod (outpipe_filename, S_IRUSR | S_IWUSR);
    }
    else
    {
	if (errno != ENOENT)
	{
	    fprintf (stderr, "Error stat'ing file: \"%s\"\t%s\n",
		     outpipe_filename, ERRSTRING);
	    myexit (RV_SYS_ERROR);
	}
	if (mkfifo (outpipe_filename, S_IRUSR | S_IWUSR) != 0)
	{
	    fprintf (stderr, "Error creating pipe: \"%s\"\t%s\n",
		     outpipe_filename, ERRSTRING);
	    myexit (RV_SYS_ERROR);
	}
    }
    /*  Write PID  */
    if ( ( fp = fopen (pid_filename, "w") ) == NULL )
    {
	fprintf (stderr, "Error writing PID file: \"%s\"\t%s\n",
		 pid_filename, ERRSTRING);
	myexit (RV_CANNOT_OPEN);
    }
    fprintf ( fp, "%d\n", getpid () );
    fclose (fp);
    /*  Open logfile  */
    if ( ( fd = open (logfile,
		      O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR) )
	< 0 )
    {
	fprintf (stderr, "Error opening logfile: \"%s\"\t%s\n", logfile,
		 ERRSTRING);
	myexit (RV_UNDEF_ERROR);
    }
    if (dup2 (fd, 1) == -1)
    {
	fprintf (stderr, "Error dup2'ing\t%s\n", ERRSTRING);
	myexit (RV_UNDEF_ERROR);
    }
    if (dup2 (fd, 2) == -1)
    {
	fprintf (stderr, "Error dup2'ing\t%s\n", ERRSTRING);
	myexit (RV_UNDEF_ERROR);
    }
    fprintf (stderr,"-----------------------------------------------------\n");
    fprintf (stderr, "PGPdaemon %s: appending to logfile\n", VERSION);
    fprintf (stderr, "USERID is: \"%s\"\n", my_userid);
    fprintf (stderr, "Incoming spool file is: \"%s\"%s\n",
	     incoming_spool, check_mail ? "" : "\tIGNORED");
    fprintf (stderr, "Outgoing spool file is: \"%s\"%s\n",
	     outgoing_spool, check_mail ? "" : "\tIGNORED");
    fprintf (stderr, "Mail check interval: %ld seconds\n", max_interval);
    if (mlock (passphrase, STRING_LENGTH) == 0)
    {
	fputs ("Passphrase memory sucessfully locked\n", stderr);
    }
    while (keep_going)
    {
	invert_passphrase ();
	sleep (1);
	if ( login_session && (getppid () != ppid) )
	{
	    keep_going = FALSE;
	    continue;
	}
	if (caught_sigusr2)
	{
	    caught_sigusr2 = FALSE;
	    icount = 0;
	    current_interval = 1;
	}
	if ( check_mail && ( !mailwait || (secring_bytes > 0) ) )
	{
	    if (++icount >= current_interval)
	    {
		icount = 0;
		if (current_interval < max_interval) ++current_interval;
		/*  Check and process incoming mail spool file  */
		if ( check_and_process_mail () )
		{
		    icount = 0;
		    current_interval = 1;
		}
	    }
	}
	if (dump_secring)
	{
	    fprintf (stderr, "Caught SIGUSR1: dumping secret keyring\n");
	    write_secring ();
	    dump_secring = FALSE;
	}
	/*  Check for connections  */
	if (caught_sigio)
	{
	    service_connect (inpipe_filename, outpipe_filename);
	    caught_sigio = FALSE;
	}
    }
    fprintf (stderr, "Removing PID file\n");
    unlink (pid_filename);
    if (restart)
    {
	if (!insecure)
	{
	    fprintf (stderr, "Erasing sensitive information...\n");
	    m_clear (passphrase, STRING_LENGTH);
	    write_secring ();
	    if (secring_bytes > 0) m_clear (secring_buf, secring_bytes);
	    sync ();
	}
	fprintf (stderr, "Restarting\n");
	execvp (argv[0], argv);
	fprintf (stderr, "Error calling  execvp(3)\t%s\n", ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    myexit (RV_OK);
}   /*  End Function main  */

static flag service_connect (CONST char *inpipe_filename,
			     CONST char *outpipe_filename)
/*  This routine will service a connection.
    The name of the input named pipe must be pointed to by  inpipe_filename  .
    The name of the output named pipe must be pointed to by  outpipe_filename
    The routine returns TRUE on succes, else it returns FALSE.
*/
{
    flag bool_val;
    int in_fd, out_fd = -1;
    int count, randval, readval, len;
    char command[STRING_LENGTH];
    extern flag insecure;
    extern SECRING_SIZE_TYPE secring_bytes;
    extern char passphrase[STRING_LENGTH];
    extern char *secring_buf;

    /*  Connect to output pipe  */
    for (count = 0; (count < 2) && (out_fd < 0); ++count)
    {
	if ( ( out_fd = open (outpipe_filename, O_WRONLY | O_NDELAY) ) < 0 )
	{
	    if (errno == ENXIO)
	    {
		sleep (1);
		continue;
	    }
	    fprintf (stderr, "Error opening named pipe\t%s\n", ERRSTRING);
	    return (FALSE);
	}
    }
    if (out_fd < 0)
    {
	fprintf (stderr, "Timeout opening named pipe\n");
	return (FALSE);
    }
    if (fcntl (out_fd, F_SETFL, 0) == -1)
    {
	fprintf (stderr, "Error controlling named pipe\t%s\n", ERRSTRING);
	close (out_fd);
	return (FALSE);
    }
    randval = get_rand ();
    if (write (out_fd, (char *) &randval, sizeof randval) != sizeof randval)
    {
	fprintf (stderr, "Error writing random value\t%s\n", ERRSTRING);
	close (out_fd);
	return (FALSE);
    }
    if ( ( in_fd = open (inpipe_filename, O_RDONLY) ) < 0 )
    {
	fprintf (stderr, "Error opening named pipe\t%s\n", ERRSTRING);
	close (out_fd);
	return (FALSE);
    }
    if (read (in_fd, (char *) &readval, sizeof readval) != sizeof readval)
    {
	fprintf (stderr, "Error reading confirmation value\t%s\n", ERRSTRING);
	close (out_fd);
	close (in_fd);
	return (FALSE);
    }
    if (readval != randval)
    {
	fprintf (stderr, "Confirmation value incorrect!\n");
	close (out_fd);
	close (in_fd);
	return (FALSE);
    }
    if ( !read_line_fd (in_fd, command, STRING_LENGTH, TRUE) )
    {
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    fprintf (stderr, "Received command: \"%s\"\n", command);
    /*  Process command  */
    if (st_icmp (command, "passphrase") == 0)
    {
	insecure = FALSE;
	if ( !read_line_fd (in_fd, passphrase, STRING_LENGTH - 1, TRUE) )
	{
	    fprintf (stderr, "Error reading passphrase\t%s\n", ERRSTRING);
	    close (out_fd);
	    close (in_fd);
	    return (FALSE);
	}
	len = strlen (passphrase);
	passphrase[len++] = '\n';
	passphrase[len] = '\0';
	/*  Simple encryption of passphrase  */
	for (count = 0; count < len; ++count) passphrase[count] ^= xor_char;
	if (secring_bytes > 0)
	{
	    m_clear (secring_buf, secring_bytes);
	    m_free (secring_buf);
	}
	secring_bytes = 0;
	secring_buf = NULL;
	if (read (in_fd, (char *) &secring_bytes, sizeof secring_bytes) <
	    sizeof secring_bytes)
	{
	    fprintf (stderr, "Error reading secret keyring length\t%s\n",
		     ERRSTRING);
	    close (out_fd);
	    close (in_fd);
	    myexit (RV_WRITE_ERROR);
	}
	if ( ( secring_buf = m_alloc (secring_bytes) ) == NULL )
	{
	    fprintf (stderr, "Error allocating: %d bytes\n", secring_bytes);
	    close (out_fd);
	    close (in_fd);
	    myexit (RV_MEM_ERROR);
	}
	if (read (in_fd, secring_buf, secring_bytes) < secring_bytes)
	{
	    fprintf (stderr, "Error reading secret keyring\t%s\n", ERRSTRING);
	    close (out_fd);
	    close (in_fd);
	    myexit (RV_READ_ERROR);
	}
	fprintf (stderr, "Passphrase and secret keyring (%d bytes) received\n",
		 secring_bytes);
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    /*  At this point, all requests are for signing or decryption  */
    bool_val = (secring_bytes > 0) ? TRUE : FALSE;
    /*  Return success/failure message  */
    if (write (out_fd, &bool_val, sizeof bool_val) < sizeof bool_val)
    {
	fprintf (stderr, "Error writing boolean\t%s\n", ERRSTRING);
	close (out_fd);
	close (in_fd);
	return (FALSE);
    }
    if (!bool_val)
    {
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    if (st_icmp (command, "SIGN") == 0)
    {
	if ( !process_message (in_fd, out_fd, TRUE, TRUE) )
	{
	    close (out_fd);
	    close (in_fd);
	    return (FALSE);
	}
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    if (st_icmp (command, "DECRYPT") == 0)
    {
	if ( !process_message (in_fd, out_fd, FALSE, TRUE) )
	{
	    close (out_fd);
	    close (in_fd);
	    return (FALSE);
	}
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    if (st_icmp (command, "BINARY_DECRYPT") == 0)
    {
	if ( !process_message (in_fd, out_fd, FALSE, FALSE) )
	{
	    close (out_fd);
	    close (in_fd);
	    return (FALSE);
	}
	close (out_fd);
	close (in_fd);
	return (TRUE);
    }
    fprintf (stderr, "Illegal command: \"%s\"\n", command);
    close (out_fd);
    close (in_fd);
    return (FALSE);
}   /*  End Function service_connect  */

static void sig_handler (sig, code, scp, addr)
/*  This routine will handle signals. It sets the global flag  keep_going  to
    FALSE.
    The signal number will be given by  sig  .
    The routine returns nothing.
*/
int sig;
int code;
struct sigcontext *scp;
char *addr;
{
    struct sigaction new_action;
    extern flag keep_going;
    extern flag restart;
    extern flag dump_secring;
    extern flag caught_sigusr2;
    extern flag caught_sigio;

    switch (sig)
    {
      case SIGINT:
      case SIGTERM:
      case SIGPIPE:
	keep_going = FALSE;
	break;
      case SIGHUP:
	keep_going = FALSE;
	restart = TRUE;
	break;
      case SIGTTIN:
      case SIGTTOU:
	break;
      case SIGUSR1:
	dump_secring = TRUE;
	break;
      case SIGUSR2:
	caught_sigusr2 = TRUE;
	break;
      case SIGIO:
	caught_sigio = TRUE;
	break;
      case SIGSEGV:
	/*  Restore signal action to default in case libc is fouled up  */
	new_action.sa_handler = SIG_DFL;
	sigemptyset (&new_action.sa_mask);
	new_action.sa_flags = 0;
	sigaction (SIGSEGV, &new_action, (struct sigaction *) NULL);
	fputs ("CAUGHT SEGMENTATION VIOLATION: ATTEMPTING CLEANUP!\n", stderr);
	myexit (RV_UNDEF_ERROR);
	break;
      default:
	fprintf (stderr, "Illegal signal: %d\n", sig);
	myexit (RV_UNDEF_ERROR);
	break;
    }
}   /*  End Function sig_handler  */

static flag process_message (int in_fd, int out_fd, flag sign, flag ascii)
/*  [SUMMARY] Process a message, either signing or decrypting it.
    <in_fd> The file descriptor from which the message should be read.
    <sign> If TRUE, the message is signed, else it is decrypted.
    <ascii> If TRUE, it is expected that the end of the message data is
    delimited by a '\0'.
    <out_fd> The file descriptor to which the processed message should be
    written.
    [RETURNS] TRUE on success, else FALSE.
*/
{
    flag remove_secring;
    int pgp_in_fd = -1;
    int pgp_out_fd = out_fd;
    int pgp_err_fd = ERROR_FD;
    int child;
    int status;
    int len, count, bytes_written;
    char tmp_passphrase[STRING_LENGTH];
    char *argv[7];
    extern char my_userid[STRING_LENGTH];
    extern char passphrase[STRING_LENGTH];

    /*  Setup arguments to PGP  */
    if (sign)
    {
	/*  pgp -fsta +clearsig=on +batchmode -u <userid>  */
	argv[0] = "pgp";
	argv[1] = "-fsta";
	argv[2] = "+clearsig=on";
	argv[3] = "+batchmode";
	argv[4] = "-u";
	argv[5] = my_userid;
	argv[6] = NULL;
	/*  The following line is a hack so that PGPsendmail won't drop the
	    first line, which is the signed message delimiter.
	    Maybe one day I'll work out *why* PGP often (but not always!)
	    drops the first line of a message it encrypts.  */
	write (out_fd, "\n", 1);
	fprintf (stderr, "Request to sign a message\n");
    }
    else
    {
	/*  pgp -f +batchmode  */
	argv[0] = "pgp";
	argv[1] = "-f";
	argv[2] = "+batchmode";
	argv[3] = NULL;
	fprintf (stderr, "Request to decrypt a message\n");
    }
    remove_secring = write_secring ();
    /*  Start PGP  */
    if ( ( child = spawn_job ("pgp", argv,
			      &pgp_in_fd, &pgp_out_fd, &pgp_err_fd) ) < 0 )
    {
	fprintf (stderr, "Error spawning PGP\n");
	if (remove_secring) scrub_secring ();
	return (FALSE);
    }
    /*  Send passphrase  */
    len = strlen (passphrase);
    /*  Simple decryption of passphrase  */
    for (count = 0; count < len; ++count)
	tmp_passphrase[count] = passphrase[count] ^ xor_char;
    bytes_written = write (pgp_in_fd, tmp_passphrase, len);
    m_clear (tmp_passphrase, len);
    if (bytes_written < len)
    {
	fprintf (stderr, "Error writing passphrase\t%s\n", ERRSTRING);
	close (pgp_in_fd);
	waitpid (child, &status, 0);
	if (remove_secring) scrub_secring ();
	return (FALSE);
    }
    /*  Copy input message to PGP  */
    if ( !copy_data (pgp_in_fd, in_fd, ascii) )
    {
	fprintf (stderr, "Error writing data\t%s\n", ERRSTRING);
	close (pgp_in_fd);
	waitpid (child, &status, 0);
	if (remove_secring) scrub_secring ();
	return (FALSE);
    }
    if (remove_secring) scrub_secring ();
    if (close (pgp_in_fd) == -1)
    {
	fprintf (stderr, "Error closing connection to PGP\t%s\n", ERRSTRING);
	waitpid (child, &status, 0);
	return (FALSE);
    }
    if (waitpid (child, &status, 0) < 0)
    {
	fprintf (stderr, "Error reaping child\t%s\n", ERRSTRING);
	return (FALSE);
    }
    if ( WIFEXITED (status) )
    {
	/*  Should exit with 0. DECRYPT is known to sometimes exit with 1  */
	if ( (WEXITSTATUS (status) != 0) &&
	     ( !sign && (WEXITSTATUS (status) != 1) ) )
	{
	    fprintf ( stderr, "Bad child status: %d\n", WEXITSTATUS (status) );
	    return (FALSE);
	}
    }
    else if ( WIFSIGNALED (status) )
    {
	fprintf ( stderr, "Child caught signal: %d\n", WTERMSIG (status) );
	return (FALSE);
    }
    else if ( WIFSTOPPED (status) )
    {
	fprintf ( stderr, "Child stopped by signal: %d\n", WSTOPSIG (status) );
	return (FALSE);
    }
    fprintf (stderr, "\n");
    return (TRUE);
}   /*  End Function process_message  */

static flag set_lockfile (filename, lock)
/*  This routine will create or delete a lockfile for a specified filename.
    The filename must be pointed to by  filename  .
    The routine creates or deletes the following lock files:
        <filename>.lock
	/tmp/<basename>.mlk
    where  basename  is the trailing path component of  filename  .
    The file will be created if the value of  lock  is TRUE, else it will be
    deleted.
    The routine returns TRUE on success, else it returns FALSE.
*/
char *filename;
flag lock;
{
    int fd;
    char *ptr;
    char lock_name[STRING_LENGTH];
    char mlk_name[STRING_LENGTH];

    strcpy (lock_name, filename);
    strcat (lock_name, ".lock");
    ptr = strrchr (filename, '/');
    ptr = (ptr == NULL) ? filename : ptr + 1;
    strcpy (mlk_name, "/tmp/");
    strcat (mlk_name, ptr);
    strcat (mlk_name, ".mlk");
    if (!lock)
    {
	if (unlink (lock_name) != 0)
	{
	    fprintf (stderr, "Error removing file: \"%s\"\t%s\n",
		     lock_name, ERRSTRING);
	    unlink (mlk_name);
	    return (FALSE);
	}
	if (unlink (mlk_name) != 0)
	{
	    fprintf (stderr, "Error removing file: \"%s\"\t%s\n",
		     mlk_name, ERRSTRING);
	    return (FALSE);
	}
	sync ();
	return (TRUE);
    }
    if ( ( fd = open (lock_name, O_RDWR | O_CREAT | O_EXCL,
		      S_IRUSR | S_IWUSR) ) < 0 )
    {
	if (errno == EEXIST)
	{
	    fprintf (stderr, "lockfile: \"%s\" exists\n", lock_name);
	    return (FALSE);
	}
	fprintf (stderr, "Error creating file: \"%s\"\t%s\n",
		 lock_name, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    fsync (fd);
    close (fd);
    if ( ( fd = open (mlk_name, O_RDWR | O_CREAT | O_EXCL,
		      S_IRUSR | S_IWUSR) ) < 0 )
    {
	unlink (lock_name);
	if (errno == EEXIST)
	{
	    fprintf (stderr, "lockfile: \"%s\" exists\n", mlk_name);
	    return (FALSE);
	}
	fprintf (stderr, "Error creating file: \"%s\"\t%s\n",
		 mlk_name, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    fsync (fd);
    close (fd);
    return (TRUE);
}   /*  End Function set_lockfile  */

static flag set_fd_lock (fd, lock, no_filelocking)
/*  This routine will (un)lock a file associated with a file descriptor.
    The file descriptor must be given by  fd  .
    The file will be lock if the value of  lock  is TRUE, else it will be
    unlocked.
    If the value of  no_filelocking  is TRUE, the routine ignores
    all requests for file locking/unlocking and returns TRUE.
    The routine returns TRUE on success, else it returns FALSE.
*/
int fd;
flag lock;
flag no_filelocking;
{
    long pos;
#ifdef F_SETLK
    struct flock fl;
#endif

    if (no_filelocking) return (TRUE);
#undef LOCKING_WORKS
#ifdef F_SETLK
#define LOCKING_WORKS
    fl.l_type = lock ? F_WRLCK : F_UNLCK;
    fl.l_whence = 0;
    fl.l_start = 0L;
    fl.l_len = 0L;
    if (fcntl (fd, F_SETLK, &fl) == -1)
    {
	fprintf (stderr, "Error %slocking file: %d with  fcntl\t%s\n",
		 lock ? "" : "un", fd, ERRSTRING);
	return (FALSE);
    }
    return (TRUE);
#endif

#if !defined(LOCKING_WORKS) && defined(F_LOCK)
#define LOCKING_WORKS
    if ( (pos = lseek (fd, 0L, 0) ) == -1)
    {
	fprintf (stderr, "Error seeking in file\t%s\n", ERRSTRING);
	return (FALSE);
    }
    if (lockf (fd, lock ? F_LOCK : F_ULOCK, 0L) != 0)
    {
	
	fprintf (stderr, "Error locking file with  lockf\t%s\n", ERRSTRING);
	return (FALSE);
    }
    if ( (pos = lseek (fd, pos, 0) ) == -1)
    {
	fprintf (stderr, "Error seeking in file\t%s\n", ERRSTRING);
	return (FALSE);
    }
    return (TRUE);
#endif

#ifndef LOCKING_WORKS
    fprintf (stderr, "File locking not implemented\n");
    my_exit (RV_UNDEF_ERROR);
    return (FALSE);
#endif
}   /*  End Function set_fd_lock  */

static flag check_and_process_mail ()
/*  [SUMMARY] Check for any incoming mail and will process that mail.
    [RETURNS] TRUE if there was mail to process, else FALSE.
*/
{
    flag pgp_data, bad_previous_pgp, read_more;
    flag remove_secring = FALSE;
    flag first_pgp = TRUE;
    int ch, len, count;
    int fd, oflags;
    int in_fd, out_fd, err_fd;
    int child = 0;
    int status;
    long bytes_read, pgp_offset;
    int line_length;
    FILE *in_fp, *out_fp;
    struct stat statbuf;
    char line[STRING_LENGTH], tmp_passphrase[STRING_LENGTH];
    char *argv[3];
    char *pgp_start_line = "-----BEGIN PGP MESSAGE-----\n";
    extern flag incoming_locked;
    extern flag no_filelocking;
    extern char incoming_spool[STRING_LENGTH];
    extern char outgoing_spool[STRING_LENGTH];
    extern char passphrase[STRING_LENGTH];

    if (stat (incoming_spool, &statbuf) != 0)
    {
	if (errno == ENOENT) return (FALSE);
	fprintf (stderr, "Error stat'ing file: \"%s\"\t%s\n",
		 incoming_spool, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    if (statbuf.st_size < 1) return (FALSE);
    fprintf (stderr, "%d bytes of mail...\n", statbuf.st_size);
    /*  Lock output file  */
    if ( !set_lockfile (outgoing_spool, TRUE) ) return (FALSE);
    if ( ( out_fp = fopen (outgoing_spool, "a") ) == NULL )
    {
	fprintf (stderr, "Error opening file: \"%s\"\t%s\n",
		 outgoing_spool, ERRSTRING);
	set_lockfile (outgoing_spool, FALSE);
	myexit (RV_SYS_ERROR);
    }
    if ( !set_fd_lock (fileno (out_fp), TRUE, no_filelocking) )
    {
	fprintf (stderr, "Error locking output spool: waiting\n");
	fclose (out_fp);
	set_lockfile (outgoing_spool, FALSE);
	return (FALSE);
    }
    /*  Now lock input file  */
    if ( !set_lockfile (incoming_spool, TRUE) )
    {
	set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
	fclose (out_fp);
	set_lockfile (outgoing_spool, FALSE);
	return (TRUE);
    }
    if ( ( in_fp = fopen (incoming_spool, "r+") ) == NULL )
    {
	fprintf (stderr, "Error opening file: \"%s\"\t%s\n",
		 incoming_spool, ERRSTRING);
	set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
	fclose (out_fp);
	set_lockfile (outgoing_spool, FALSE);
	set_lockfile (incoming_spool, FALSE);
	myexit (RV_SYS_ERROR);
    }
    if ( !set_fd_lock (fileno (in_fp), TRUE, no_filelocking) )
    {
	fprintf (stderr, "Error locking input spool: waiting\n");
	set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
	fclose (out_fp);
	set_lockfile (outgoing_spool, FALSE);
	fclose (in_fp);
	set_lockfile (incoming_spool, FALSE);
	return (TRUE);
    }
    /*  Process incoming and produce outgoing  */
    bytes_read = 0;
    pgp_data = FALSE;
    bad_previous_pgp = FALSE;
    fprintf (stderr, "Spoolfiles locked\n");
    read_more = TRUE;
    while (read_more)
    {
	if (fgets (line, STRING_LENGTH, in_fp) == NULL)
	{
	    /*  Must be EndOfFile  */
	    if (!pgp_data)
	    {
		/*  Nothing special, just stop trying to read  */
		read_more = FALSE;
		continue;
	    }
	    fprintf (stderr, "\nPGP message terminated prematurely\n");
	    fprintf (stderr, "Message not decrypted\n");
	    fprintf (out_fp, "PGP message terminated prematurely\n");
	    fprintf (out_fp, "Message not decrypted\n\n");
	    /*  PGP message terminated without end line: plain copy  */
	    if (child > 0)
	    {
		fprintf (stderr, "Killing PGP process\n");
		kill (child, SIGKILL);
		waitpid (child, &status, 0);
		child = 0;
	    }
	    if (fseek (in_fp, pgp_offset, SEEK_SET) != 0)
	    {
		fprintf (stderr, "Error seeking\t%s\n", ERRSTRING);
		myexit (RV_SYS_ERROR);
	    }
	    pgp_data = FALSE;
	    bad_previous_pgp = TRUE;
	    bytes_read = pgp_offset;
	    continue;
	}
	/*  New line of data is available  */
	line_length = strlen (line);
	bytes_read += line_length;
	if (pgp_data)
	{
	    /*  Check to see if PGP is still running  */
	    switch ( waitpid (child, &status, WNOHANG | WUNTRACED) )
	    {
	      case 0:
		/*  PGP still running  */
		break;
	      case -1:
		fprintf (stderr, "Error getting child status\t%s\n",
			 ERRSTRING);
		myexit (RV_SYS_ERROR);
		break;
	      default:
		if ( WIFSTOPPED (status) )
		{
		    /*  PGP was stopped: probably because message was encrypted
			with that fucking "For-your-eyes-only" option, and so
			PGP *insists* on asking if you want to view the
			message.  */
		    /*  Note that at this point, this condition is not likely,
			rather, reaping the child will show this condition.  */
		    fprintf (stderr,
			     "\nPGP stopped while decrypting, probably");
		    fprintf (stderr, " because it was encrypted with that\n");
		    fprintf (stderr,"fucking \"For-your-eyes-only\" option\n");
		    fprintf (out_fp, "PGP stopped while decrypting, probably");
		    fprintf (out_fp, " because it was encrypted with that\n");
		    fprintf (out_fp,"fucking \"For-your-eyes-only\" option. ");
		}
		else
		{
		    fprintf (stderr, "\nBad PGP child status: %d\n", status);
		    fprintf (out_fp, "Bad PGP child status: %d\n", status);
		}
		fprintf (stderr,
			 "Message not decrypted: killing PGP process\n");
		fprintf (out_fp, "Message not decrypted\n");
		kill (child, SIGKILL);
		waitpid (child, &status, 0);
		child = 0;
		if (fseek (in_fp, pgp_offset, SEEK_SET) != 0)
		{
		    fprintf (stderr, "Error seeking\t%s\n", ERRSTRING);
		    myexit (RV_SYS_ERROR);
		}
		pgp_data = FALSE;
		bad_previous_pgp = TRUE;
		bytes_read = pgp_offset;
		continue;
		/*break;*/
	    }
	}
	if (st_nicmp (line, "subject:", 8) == 0)
	{
	    fprintf (stderr, "> %s", line);
	}
	if ( (strcmp (line, pgp_start_line) == 0) && pgp_data )
	{
	    /*  Hm. Looks like the last PGP message wasn't terminated
		properly. I suppose PGP gave no output  */
	    fprintf (stderr, "\nStart of new PGP message without termination");
	    fprintf (stderr, " of previous message\nMessage not decrypted\n");
	    fprintf (out_fp, "Start of new PGP message without termination");
	    fprintf (out_fp, " of previous message\nMessage not decrypted\n");
	    /*  PGP message terminated without end line: plain copy  */
	    kill (child, SIGKILL);
	    waitpid (child, &status, 0);
	    child = 0;
	    if (fseek (in_fp, pgp_offset, SEEK_SET) != 0)
	    {
		fprintf (stderr, "Error seeking\t%s\n", ERRSTRING);
		myexit (RV_SYS_ERROR);
	    }
	    pgp_data = FALSE;
	    bad_previous_pgp = TRUE;
	    bytes_read = pgp_offset;
	    continue;
	}
	if ( (strcmp (line, pgp_start_line) == 0) &&
	     (passphrase[0] != '\0') && (!bad_previous_pgp) )
	{
	    pgp_data = TRUE;
	    pgp_offset = ftell (in_fp) - line_length;
	    in_fd = -1;
	    out_fd = fileno (out_fp);
	    err_fd = ERROR_FD;
	    fprintf (out_fp,
		     "PGPdaemon %s: automatically decrypted message:\n\n",
		     VERSION);
	    fflush (out_fp);
	    fprintf (stderr, "DECRYPTING...\n");
	    if (first_pgp)
	    {
		remove_secring = write_secring ();
		first_pgp = FALSE;
	    }
	    /*  Start PGP  */
	    /*  pgp -f  */
	    argv[0] = "pgp";
	    argv[1] = "-f";
	    argv[2] = NULL;
	    if (child > 0)
	    {
		fprintf (stderr, "Killing old child: %d before spawning\n",
			 child);
		kill (child, SIGKILL);
		waitpid (child, &status, 0);
	    }
	    if ( ( child = spawn_job ("pgp", argv, &in_fd, &out_fd, &err_fd) )
		 < 0 )
	    {
		fprintf (stderr, "Error spawning PGP\n");
		myexit (RV_SYS_ERROR);
	    }
	    /*  Send passphrase  */
	    len = strlen (passphrase);
	    /*  Simple decryption of passphrase  */
	    for (count = 0; count < len; ++count)
		tmp_passphrase[count] = passphrase[count] ^ xor_char;
	    if (write (in_fd, tmp_passphrase, len) < len)
	    {
		m_clear (tmp_passphrase, len);
		fprintf (stderr, "Error writing passphrase\t%s\n", ERRSTRING);
		close (in_fd);
		waitpid (child, &status, 0);
		myexit (RV_SYS_ERROR);
	    }
	    m_clear (tmp_passphrase, len);
	}
	if (pgp_data)
	{
	    if (write (in_fd, line, line_length) < line_length)
	    {
		if (errno == EPIPE)
		{
		    /*  PGP dropped out for some reason: plain copy  */
		    fprintf (stderr, "PGP closed pipe\n");
		    kill (child, SIGKILL);
		    waitpid (child, &status, 0);
		    child = 0;
		    if (fseek (in_fp, pgp_offset, SEEK_SET) != 0)
		    {
			fprintf (stderr, "Error seeking\t%s\n", ERRSTRING);
			myexit (RV_SYS_ERROR);
		    }
		    pgp_data = FALSE;
		    bad_previous_pgp = TRUE;
		    bytes_read = pgp_offset;
		    continue;
		}
		fprintf (stderr, "Error writing data to PGP\t%s\n", ERRSTRING);
		myexit (RV_SYS_ERROR);
	    }
	}
	else
	{
	    if (fputs (line, out_fp) == EOF)
	    {
		fprintf (stderr, "Error writing\t%s\n", ERRSTRING);
		fprintf (stderr, "Error copying data\n");
		set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
		fclose (out_fp);
		set_lockfile (outgoing_spool, FALSE);
		set_fd_lock (fileno (in_fp), FALSE, no_filelocking);
		fclose (in_fp);
		set_lockfile (incoming_spool, FALSE);
		myexit (RV_UNDEF_ERROR);
	    }
	}
	if ( (strcmp (line, pgp_start_line) == 0) && bad_previous_pgp )
	{
	    /*  This line started off a bad PGP message on the previous round.
		Now that we've gone past the line, we can clear the
		bad_previous_pgp  flag.  */
	    bad_previous_pgp = FALSE;
	    pgp_data = FALSE;  /*  What the hell, clear it just in case  */
	    continue;
	}
	if (strcmp (line, "-----END PGP MESSAGE-----\n") == 0)
	{
	    bad_previous_pgp = FALSE;
	    if (!pgp_data) continue;
	    pgp_data = FALSE;
	    close (in_fd);
	    waitpid (child, &status, WUNTRACED);
	    if (status != 0)
	    {
		if ( WIFSTOPPED (status) )
		{
		    /*  PGP was stopped: probably because message was encrypted
			with that fucking "For-your-eyes-only" option, and so
			PGP *insists* on asking if you want to view the
			message.  */
		    fprintf (stderr,
			     "\nPGP stopped after decrypting, probably");
		    fprintf (stderr, " because it was encrypted with that\n");
		    fprintf (stderr,"fucking \"For-your-eyes-only\" option\n");
		    fprintf (out_fp, "PGP stopped after decrypting, probably");
		    fprintf (out_fp, " because it was encrypted with that\n");
		    fprintf (out_fp,"fucking \"For-your-eyes-only\" option. ");
		    fprintf (stderr, "Killing PGP process\n");
		    kill (child, SIGKILL);
		    waitpid (child, &status, WUNTRACED);
		}
		else
		{
		    fprintf (stderr, "Bad PGP child status: %d\n", status);
		    fprintf (out_fp, "Bad PGP child status: %d\n", status);
		}
		child = 0;
		fprintf (stderr, "Message not decrypted\n");
		fprintf (out_fp, "Message not decrypted.\n\n");
		if (fseek (in_fp, pgp_offset, SEEK_SET) != 0)
		{
		    fprintf (stderr, "Error seeking\t%s\n", ERRSTRING);
		    myexit (RV_SYS_ERROR);
		}
		bad_previous_pgp = TRUE;
		bytes_read = pgp_offset;
		continue;
	    }
	    child = 0;
	    fprintf (stderr, "\nDECRYPTED\n");
	    fprintf (out_fp, "\nEnd automatically decrypted message\n");
	}
    }
    if (child > 0)
    {
	fprintf (stderr, "KILLING old child: %d after processing\n", child);
	kill (child, SIGKILL);
	waitpid (child, &status, WUNTRACED);
    }
    if (bytes_read < statbuf.st_size)
    {
	fprintf (stderr, "Error reading\t%s\n", ERRSTRING);
	fprintf (stderr, "Error copying data\n");
	set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
	fclose (out_fp);
	set_lockfile (outgoing_spool, FALSE);
	set_fd_lock (fileno (in_fp), FALSE, no_filelocking);
	fclose (in_fp);
	set_lockfile (incoming_spool, FALSE);
	myexit (RV_UNDEF_ERROR);
    }
    fflush (out_fp);
    fsync ( fileno (out_fp) );
    fprintf (stderr, "Unlocking spoolfiles...");
    /*  Unlock output spoolfile  */
    set_fd_lock (fileno (out_fp), FALSE, no_filelocking);
    fclose (out_fp);
    set_lockfile (outgoing_spool, FALSE);
#ifdef dummy
    /*  Remove input spoolfile  */
    if (unlink (incoming_spool) != 0)
    {
	fprintf (stderr, "Error removing file: \"%s\"\t%s\n",
		 incoming_spool, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    /*  Unlock input spoolfile  */
#else
    /*  Empty input spoolfile  */
    oflags = O_WRONLY | O_TRUNC;
#  ifdef O_SYNC
    oflags |= O_SYNC;
#  endif
    if ( ( fd = open (incoming_spool, oflags, 0) ) < 0 )
    {
	fprintf (stderr, "Error truncating file: \"%s\"\t%s\n",
		 incoming_spool, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
#  ifndef O_SYNC
    fsync (fd);
#  endif
    close (fd);
    set_fd_lock (fileno (in_fp), FALSE, no_filelocking);
#endif
    fclose (in_fp);
    set_lockfile (incoming_spool, FALSE);
    fprintf (stderr, "\tmail processed\n");
    if (remove_secring) scrub_secring ();
    return (TRUE);
}   /*  End Function check_and_process_mail  */

static void read_config (pgppath, mailwait, interval, check_mail)
/*  This routine will read the configuration file:  $PGPPATH/PGPdaemon.config
    The value of the PGPPATH environment variable must be pointed to by
    pgppath  .
    If the "MAILWAIT" keyword is present, the value TRUE will be written to the
    storage pointed to by  mailwait  .
    If the "MAIL_CHECK_INTERVAL" keyword is present, the associated integer
    value will be written to the storage pointed to by  interval  .
    If the value of  check_mail  is TRUE, the routine will check for the
    writability of the mail spool directory.
    The routine returns nothing.
*/
char *pgppath;
flag *mailwait;
long *interval;
flag check_mail;
{
    char *keyword, *rest;
    char *tmp;
    FILE *fp;
    char config_filename[STRING_LENGTH];
    char txt[STRING_LENGTH];
    extern char my_userid[STRING_LENGTH];
    extern char incoming_spool[STRING_LENGTH];
    extern char outgoing_spool[STRING_LENGTH];
    extern char logfile[STRING_LENGTH];

    sprintf (config_filename, "%s/PGPdaemon.config", pgppath);
    if ( ( fp = fopen (config_filename, "r") ) == NULL )
    {
	fprintf (stderr, "Error reading file: \"%s\"\t%s\n",
		 config_filename, ERRSTRING);
	myexit (RV_CANNOT_OPEN);
    }
    /*  Read in lines  */
    m_clear (my_userid, STRING_LENGTH);
    m_clear (incoming_spool, STRING_LENGTH);
    m_clear (outgoing_spool, STRING_LENGTH);
    m_clear (logfile, STRING_LENGTH);
    while (fgets (txt, STRING_LENGTH - 1, fp) != NULL)
    {
	/*  Strip newline  */
	txt[strlen (txt) - 1] = '\0';
	if ( (int) strlen (txt) < 1 ) continue;
	if (txt[0] == '#') continue;
	if ( ( keyword = ex_str (txt, &rest) ) == NULL ) continue;
	if (strcmp (keyword, "USERID") == 0) strcpy (my_userid, rest);
	else if (strcmp (keyword, "MAILWAIT") == 0) *mailwait = TRUE;
	else if (strcmp (keyword, "IN_SPOOL_DIR") == 0)
	{
	    strcpy (incoming_spool, rest);
	}
	else if (strcmp (keyword, "OUT_SPOOL_FILE") == 0)
	{
	    strcpy (outgoing_spool, rest);
	}
	else if (strcmp (keyword, "MAIL_CHECK_INTERVAL") == 0)
	{
	    if (rest == NULL)
	    {
		fprintf (stderr, "No interval specified\n");
		myexit (RV_BAD_DATA);
	    }
	    if ( ( *interval = atol (rest) ) < MIN_INTERVAL )
	    {
		fprintf (stderr, "Interval: %ld is less than minimum: %ld\n",
			 *interval, MIN_INTERVAL);
		myexit (RV_BAD_DATA);
	    }
	    if (*interval > MAX_INTERVAL)
	    {
		fprintf (stderr,"Interval: %ld is greater than maximum: %ld\n",
			 *interval, MAX_INTERVAL);
		myexit (RV_BAD_DATA);
	    }
	}
	else if (strcmp (keyword, "LOG_FILE") == 0) strcpy (logfile, rest);
	else
	{
	    fprintf (stderr, "Illegal config line: \"%s\"\n", txt);
	    myexit (RV_BAD_DATA);
	}
	m_free (keyword);
    }
    if (my_userid[0] == '\0')
    {
	fprintf (stderr, "No USERID specified in config file\n");
	myexit (RV_BAD_DATA);
    }
    if (my_userid[0] == '$')
    {
	if ( ( tmp = getenv (my_userid + 1) ) == NULL )
	{
	    fprintf (stderr, "Environment variable: \"%s\" not found\n",
		     my_userid + 1);
	    myexit (RV_BAD_DATA);
	}
	strcpy (my_userid, tmp);
    }
    if (incoming_spool[0] == '\0')
    {
	/*  No existing spool directory named  */
	if ( ( tmp = getenv ("MAIL") ) == NULL )
	{
	    strcpy (incoming_spool, _PATH_MAILDIR);
	    strcat (incoming_spool, "/");
	    strcat ( incoming_spool, getenv ("USER") );
	}
	else
	{
	    strcpy (incoming_spool, tmp);
	}
    }
    else
    {
	strcat (incoming_spool, "/");
	strcat ( incoming_spool, getenv ("USER") );
    }
    st_expr_expand (incoming_spool, STRING_LENGTH, incoming_spool, NULL);
    /*  Extract spool directory  */
    strcpy (txt, incoming_spool);
    if ( ( tmp = strrchr (txt, '/') ) == NULL )
    {
	fprintf (stderr, "No '/' in spool directory name: \"%s\"\n", txt);
	myexit (RV_BAD_DATA);
    }
    *tmp = '\0';
    if ( check_mail && (access (txt, W_OK) != 0) )
    {
	if (errno == ENOENT)
	{
	    fprintf (stderr, "Spool directory: \"%s\" does not exist\n", txt);
	    myexit (RV_FILE_NOT_FOUND);
	}
	if (errno == EACCES)
	{
	    fprintf (stderr, "Spool directory: \"%s\" is not writable\n", txt);
	    myexit (RV_WRITE_ERROR);
	}
	fprintf (stderr,
		 "Error checking access to spool directory: \"%s\"\t%s\n",
		 txt);
	myexit (RV_SYS_ERROR);
    }
    if (outgoing_spool[0] == '\0')
    {
	sprintf ( outgoing_spool, "%s/decrypted-mail", getenv ("HOME") );
    }
    else
    {
	st_expr_expand (outgoing_spool, STRING_LENGTH, outgoing_spool, NULL);
    }
    if (logfile[0] == '\0')
    {
	sprintf (logfile, "%s/PGPdaemon.log.%s", pgppath, hostname);
    }
    else st_expr_expand (logfile, STRING_LENGTH, logfile, NULL);
}   /*  End Function read_config  */

static flag write_secring ()
/*  This routine will write the secret keyring file.
    The routine returns TRUE if the file did not already exist,
    else it returns FALSE (indicating there is an existing file which will not
    be overwritten).
*/
{
    int fd;
    char filename[STRING_LENGTH];
    extern SECRING_SIZE_TYPE secring_bytes;
    extern char *secring_buf;
    extern char *pgppath;

    sprintf (filename, "%s/secring.pgp", pgppath);
    if ( ( fd = open (filename, O_WRONLY | O_CREAT | O_EXCL,
		      S_IRUSR | S_IWUSR) ) < 0 )
    {
	if (errno == EEXIST)
	{
	    return (FALSE);
	}
	fprintf (stderr, "Error creating file: \"%s\"\t%s\n",
		 filename, ERRSTRING);
	myexit (RV_SYS_ERROR);
    }
    if (write (fd, secring_buf, secring_bytes) < secring_bytes)
    {
	fprintf (stderr, "Error writing\t%s\n", ERRSTRING);
	close (fd);
	myexit (RV_SYS_ERROR);
    }
    close (fd);
    return (TRUE);
}   /*  End Function write_secring  */

static flag scrub_secring ()
/*  This routine will scrub the secret keyring file.
    The routine returns TRUE on success, else it returns FALSE.
*/
{
    int fd;
    char filename[STRING_LENGTH];
    extern char *pgppath;

    sprintf (filename, "%s/secring.pgp", pgppath);
    return scrub_file (filename, TRUE);
}   /*  End Function scrub_secring  */

static int get_rand ()
/*  This routine will compute a random number with a Uniform distribution.
    The routine will return the number.
*/
{   
    struct timeval tv;
    struct timezone tz;
    static int first_time = TRUE;

    if (first_time)
    {
	first_time = FALSE; 
	gettimeofday (&tv, &tz);
	tv.tv_sec ^= tv.tv_usec;
	srand (tv.tv_sec);
    }
    return rand ();
}   /*  End Function get_rand  */

static void invert_passphrase ()
/*  [SUMMARY] Invert the passphrase and XOR character.
    [PURPOSE] This function will invert the passphrase and XOR character. If
    called regularly, it ensures that the passphrase buffer contents do not
    remain static with time. This is important because it is possible to read
    the previous contents of a memory chip even after it has been turned off.
    The longer a bit in memory is unchanged, the longer it takes for the
    afterimage to fade beyond recovery. Regularly inverting the buffer contents
    serves to defeat attempts at data recovery. This function also helps to
    keep the passphrase out of swap space.
    [RETURNS] Nothing.
*/
{
    int count;
    int length = strlen (passphrase);

    for (count = 0; count < length; ++count)
    {
	passphrase[count] = ~passphrase[count];
    }
    xor_char = ~xor_char;
}   /*  End Function invert_passphrase  */
