#include "general.h"

void initialise(int argc, char **argv, int fd[2]) {
   char *chrootdir = NULL;
   char *keyfile = NULL;
   char *ptr;
   struct sockaddr_in servaddr;
   int uid = -1, gid = -1;
   int o, udp_probe = 1, opt = 1;
   int listenhost, listenport;
   long flags;

   opterr = 0;
   while((o = getopt(argc, argv, "?hvdel:c:u:g:s:k:t:")) != EOF) {
      switch(o) {
	 case 'h':
	 case '?':
	    usage(argv[0]);
	    break;
	 case 'v':
	    version();
	    break;
	 case 'd':
	    debug++;
	    break;
	 case 'e':
	    udp_probe = 0;
	    break;
	 case 'l':
	    max_simul_con = atoi(optarg);
	    break;
	 case 'c':
	    if((chrootdir = malloc(strlen(optarg)+1)) == NULL)
	       err_sys("memory allocation error");
	    strcpy(chrootdir, optarg);
	    break;
	 case 'u':
	    uid = atoi(optarg);
	    break;
	 case 'g':
	    gid = atoi(optarg);
	    break;
	 case 's':
	    sndbufsz = atoi(optarg);
	    break;
	 case 'k':
	    if((keyfile = malloc(strlen(optarg)+1)) == NULL)
               err_sys("memory allocation error");
            strcpy(keyfile, optarg);
            break;
	 case 't':
	    timeout = atoi(optarg);
	    break;
	 default:
	    fprintf(stderr, "Invalid option: -%c\n", o);
	    usage(argv[0]);
	    break;
      }
   }

   if(LoadPrivateKey(keyfile ? keyfile : PRIVATE_KEY_FILE, &ServerPrivateKey) != 0)
      exit(-1);         /* LoadPrivateKey prints error message */

   if(keyfile)
      free(keyfile);

   if((fd[0] = socket(AF_INET, SOCK_STREAM, 0)) == -1)
      err_sys("initialise: Can not SOCK_STREAM create socket");

   if(setsockopt(fd[0], SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(int)) == -1)
      err_sys("initialise: Can not set SO_REUSEADDR");

   if((flags = fcntl(fd[0], F_GETFL, 0)) == -1)
      err_sys("initialise: fcntl F_GETFL error");

   if(fcntl(fd[0], F_SETFL, flags | O_NONBLOCK) == -1)
      err_sys("initialise: fcntl F_SETFL error");

   if((argc - optind) > 0) {
      if((ptr = strchr(argv[optind], ':'))) {
         struct hostent *host;

         *ptr++ = '\0';
         listenport = atoi(ptr);

         if((host = gethostbyname2(argv[optind], AF_INET)) == NULL) {
            fprintf(stderr, "Error resolving %s: %s\n", argv[optind], hstrerror(h_errno));
            exit(-1);
         }
         listenhost = *(int *)host->h_addr_list[0];
      } else {
	 listenport = atoi(argv[optind]);
         listenhost = INADDR_ANY;
      }
   } else {
      listenport = LISTEN_PORT;
      listenhost = INADDR_ANY;
   }

   bzero(&servaddr, sizeof(servaddr));
   servaddr.sin_family = AF_INET;
   servaddr.sin_addr.s_addr = listenhost;
   servaddr.sin_port = htons(listenport);

   if(bind(fd[0], (struct sockaddr *)&servaddr, sizeof(servaddr)) == -1)
      err_sys("initialise: Can not bind SOCK_STREAM socket");

   if(listen(fd[0], SOMAXCONN) == -1)
      err_sys("initialise: Can not listen");

   if(udp_probe) {
      if((fd[1] = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
         err_sys("initialise: Can not create SOCK_DGRAM socket");
      if(bind(fd[1], (struct sockaddr *)&servaddr, sizeof(servaddr)) == -1)
         err_sys("initialise: Can not bind SOCK_DGRAM socket");
   } else {
      fd[1] = 0;
   }

   if(!sndbufsz) {
      socklen_t optlen = sizeof(int);

      if(getsockopt(fd[0], SOL_SOCKET, SO_SNDBUF, &sndbufsz, &optlen))
         err_sys("initialise: Can not get SO_SNDBUF");
   }

   if(!timeout)
      timeout = TIMEOUT;

   if(chrootdir) {
      if(chroot(chrootdir))
         err_sys("initialise: Can not chroot to %s", chrootdir);
      chdir("/");
      free(chrootdir);
   }

   if(gid != -1) {
      if(setregid(gid, gid))
         err_sys("initialise: Can not change gid to %d", gid);
   }

   if(uid != -1) {
      if(setreuid(uid, uid))
         err_sys("initialise: Can not change uid to %d", uid);
   }

   if(!debug) {
      if(fork()) {
	 exit(0);
      } else {
         setsid();
	 openlog(argv[0], LOG_PID, LOG_USER);
         daemon_proc = 1;
      }
   }
}

void usage(char *name) {
   fprintf(stderr, "aesopd: Advanced Encrypted Stackable Open Proxy v0.1\n");
   fprintf(stderr, "usage: %s [options] [[listenhost:]port]\n", name);
   fprintf(stderr, "where options are:\n");
   fprintf(stderr, "\t-h\t\t: Show this help\n");
   fprintf(stderr, "\t-d\t\t: Run the daemon in debug mode\n");
   fprintf(stderr, "\t-c dir\t\t: Chroot daemon to dir\n");
   fprintf(stderr, "\t-u uid\t\t: Run daemon under uid\n");
   fprintf(stderr, "\t-g gid\t\t: Run daemon under gid\n");
   fprintf(stderr, "\t-e\t\t: Disable support for udp probing\n");
   fprintf(stderr, "\t-l num\t\t: Maximum simultaneous connections\n");
   fprintf(stderr, "\t-s size\t\t: Set internal buffersize\n");
   fprintf(stderr, "\t-k file\t\t: Specify private keyfile\n");
   fprintf(stderr, "\t-t timeout\t: Set connection timeout\n");
   fprintf(stderr, "\t-v\t\t: Show version information\n");
   exit(0);
}

void version(void) {
   fprintf(stderr, "\tThis is aesop v%s\n", VERSION);
   fprintf(stderr, "webpage: http://www.kryptology.org/aesop/\n");
   fprintf(stderr, "author:  bighawk, bighawk@kryptology.org\n");
   exit(0);
}
