#include "aesopd.h"

/* Readdata() tries to read as much bytes from the local or remote fd specified
   by 'from' as there is free space available in the buffer, but no more than 
   'max' if 'max' is not zero.
   It checks for Out of Band data and if available it's pushed on the connection
   structures oob queue. Readdata() decrypts the readed data before it's added 
   depending on the mode the structure is in. Readdata() also modifies the global
   readset and writeset fd_set's depending on if there's data available to write
   out or in the situation that the buffer is full.
*/
void Readdata(Connection *con, int from, int max) {
   int n, count, fd;
   unsigned char *buf;
   char *name;

   if(from == FROM_LOCAL) {
      fd = con->localfd;
      name = "localfd";
      count = (sndbufsz - con->lpos);
      if(max)
	 count = MIN(count, max);
      if(sockatmark(con->localfd)) {
#ifdef DEBUG
	 err_msg("Readdata: Read OOB data from localfd=%d", con->localfd);
#endif
	 oob_add(&con->l_oobpos, &con->l_oobsize, &con->l_oobused, con->lpos);
      }
      buf = con->lbuf + con->lpos;
   }
   else {	/* FROM_REMOTE	*/
      fd = con->remotefd;
      name = "remotefd";
      count = (sndbufsz - con->rpos);
      if(max)
	 count = MIN(count, max);
      if(sockatmark(con->remotefd)) {
#ifdef DEBUG
	 err_msg("Readdata: Read OOB data from remotefd=%d", con->remotefd);
#endif
	 oob_add(&con->r_oobpos, &con->r_oobsize, &con->r_oobused, con->rpos);
      }
      buf = con->rbuf + con->rpos;
   }

   n = read(fd, buf, count);

   switch(n) {
      case -1:
#ifdef DEBUG
	err_ret("Readdata: Read error from %s=%d", name, fd);
#endif
	Connection_del(con);
	break;
      case 0: 	/* eof */
	Connection_eof(con, from == FROM_LOCAL ? EOF_LOCAL : EOF_REMOT);
	break;
      default:
#ifdef DEBUG
	err_msg("Readdata: Read %d bytes from %s=%d", n, name, fd);
#endif
	if(from == FROM_LOCAL) {
	   arcfour_decrypt(&con->d, buf, buf, n); 
	   if(con->control & CNTRL_CHAIN)
	      arcfour_encrypt(&con->e2, buf, buf, n);
	   else if((con->control & CNTRL_ENDPO) && (con->status == ST_CONNECT))
	      arcfour_decrypt(&con->d2, buf, buf, n);
	   con->lpos += n;
	   if(con->lpos == sndbufsz) { /* Receive buffer is full, clear localfd from readset */
#ifdef DEBUG
	      err_msg("Readdata: Receive buffer for localfd=%d is full", con->localfd);
#endif
	      FD_CLR(con->localfd, &rdset);
	   }
	   FD_SET(con->remotefd, &wrset); /* There's data to write */
	} else {	/* FROM_REMOTE	*/
	   if(con->control & CNTRL_CHAIN)
	      arcfour_decrypt(&con->d2, buf, buf, n);
	   else if((con->control & CNTRL_ENDPO) && (con->status == ST_CONNECT))
	      arcfour_encrypt(&con->e2, buf, buf, n);
	   arcfour_encrypt(&con->e, buf, buf, n);
	   con->rpos += n;
	   if(con->rpos == sndbufsz) {	/* Receive buffer is full, clear remotefd from readset */
#ifdef DEBUG
	      err_msg("Readdata: Receive buffer for remotefd=%d is full", con->remotefd);
#endif
	      FD_CLR(con->remotefd, &rdset);
	   }
	   FD_SET(con->localfd, &wrset); /* There's data to write */
	}
	break;
   }
}

/* Writedata() tries to write the entire buffer contents out to the remote
   or local fd specified by 'to'. If connection structure is at a OOB position
   it will send it out as OOB byte. The internal buffers will be shifted by
   the amount of data written out. And a vector correction on the queue with OOB
   positions will be done. Writedata() modifies the readset and writeset fd_set's
   depending on if there is still data left in the buffers or not.
*/
void Writedata(Connection *con, int to) {
   int n, fd;
   char *name;

   if(to == TO_LOCAL) {
      fd = con->localfd;
      name = "localfd";
      if(con->r_oobused && (con->r_oobpos[0] == 0)) {
#ifdef DEBUG
	 err_msg("Writedata: Send 1 byte of OOB data to localfd=%d", con->localfd);
#endif
	 n = send(con->localfd, con->rbuf, 1, MSG_OOB);	/* We send 1 byte of OOB data	*/
	 oob_shift(con->r_oobpos, &con->r_oobused);
      }
      else
	 n = write(con->localfd, con->rbuf, con->r_oobused ? con->r_oobpos[0] : con->rpos);
   } else {
      fd = con->remotefd;
      name = "remotefd";
      if(con->l_oobused && (con->l_oobpos[0] == 0)) {
#ifdef DEBUG
	 err_msg("Writedata: Send 1 byte of OOB data to remotefd=%d", con->remotefd);
#endif
	 n = send(con->remotefd, con->lbuf, 1, MSG_OOB); /* We send 1 byte of OOB data	*/
	 oob_shift(con->l_oobpos, &con->l_oobused);
      }
      else
	 n = write(con->remotefd, con->lbuf, con->l_oobused ? con->l_oobpos[0] : con->lpos);
   }

   switch(n) {
      case -1:
#ifdef DEBUG
	err_ret("Writedata: Write error to %s=%d", name, fd);
#endif
	Connection_del(con);
	break;	
      case 0:	/* Nothing written? strange.. */
#ifdef DEBUG
	err_msg("Writedata: Write to %s=%d returned zero..", name, fd);
#endif
	 break;
      default:
#ifdef DEBUG
	err_msg("Writedata: Wrote %d bytes to %s=%d", n, name, fd);
#endif
	if(to == TO_LOCAL) {
	   con->rpos -= n;
	   if(con->r_oobused)
	      oob_corvec(con->r_oobpos, con->r_oobused, -n);
	   memmove(con->rbuf, (con->rbuf + n), con->rpos);
	   if(con->remotefd != -1 && con->status != ST_FINISHR)
	      FD_SET(con->remotefd, &rdset);	/* We wrote something, so there must be space in the buffer */
	   if(con->rpos == 0) {	/* We wrote out all data */
	      FD_CLR(con->localfd, &wrset);
	      if(con->status == ST_FINISHR)
		 Connection_eof(con, EOF_REMOT);
	   }
	} else {
	   con->lpos -= n;
	   if(con->l_oobused)
	      oob_corvec(con->l_oobpos, con->l_oobused, -n);
	   memmove(con->lbuf, (con->lbuf + n), con->lpos);
	   if(con->remotefd != -1 && con->status != ST_FINISHL)
	      FD_SET(con->localfd, &rdset);	/* We wrote something, so there must be space in the buffer */
	   if(con->lpos == 0) {	/* We wrote out all data */
	      FD_CLR(con->remotefd, &wrset);
	      if(con->status == ST_FINISHL)
		 Connection_eof(con, EOF_LOCAL);
	   }
	}
	break;
   }
}

/* CheckConnection() checks if the remote connection succeeded or not.
   If the connection failed the connection structure will be deallocated.
   If the connection succeeded and the proxy functions as standalone or
   endpoint for that proxy the state will change to ST_CONNECT else the
   connection structure will be passed to Connection_DH_init() to setup the
   DH negotiation.
*/
void CheckConnection(Connection *con) {
   int error = 0;
   int optlen = sizeof(int);

   if(getsockopt(con->remotefd, SOL_SOCKET, SO_ERROR, &error, &optlen) == -1)
      err_sys("main: getsockopt SO_ERROR error");

   if(error) {   /* connection did not succeed */
#ifdef DEBUG
      err_msg("main: Connection for remotefd=%d did not succeed", con->remotefd);
#endif
      Connection_del(con);
      return;
   }

#ifdef DEBUG
   err_msg("main: Connection for remotefd=%d did succeed", con->remotefd);
#endif

   FD_CLR(con->remotefd, &wrset);

   if(con->control & CNTRL_CHAIN)
      Connection_DH_init(con, AS_CHAIN);
   else {
      FD_SET(con->localfd, &rdset);
      FD_SET(con->remotefd, &rdset);
      con->status = ST_CONNECT;
   }
}
