#include "libaesop.h"

static int init_done = 0;
static int ipv4ready = 0;
static int ipv6ready = 0;
static int sockchange = 0;
static struct sockaddr_in sin;
static socklen_t sinlen = 0;
#ifdef IPV6
static struct sockaddr_in6 sin6;
static socklen_t sin6len = 0;
#endif

static int (*_r_connect)(int, const struct sockaddr *, socklen_t) = NULL;

void _init(void) {
#ifdef IPV6
   int n;
#endif
   void *handle;
   char *localaddr, *localport;
   struct hostent *ent;
#ifdef IPV6
   struct addrinfo hints, *res;
#endif

#ifndef have_RTLD_NEXT
   if((handle = dlopen(LIBC, RTLD_LAZY)) == NULL) {
      fprintf(stderr, "libaesop: dlopen error for %s: %s\n", LIBC, dlerror());
      exit(-1);
   }
#else
   handle = RTLD_NEXT;
#endif

   if((_r_connect  = dlsym(handle, "connect")) == NULL) {
       char *errmsg;
       if((errmsg = dlerror()) != NULL) {
	  fprintf(stderr, "libaesop: dlsym error for connect(): %s\n", errmsg);
	  exit(-1);
       }
   }

#ifndef have_RTLD_NEXT
   dlclose(handle);
#endif

   if(getenv("AESOP_ALLOW_IPV4TOIPV6") != NULL)
      sockchange = 1;

   localaddr = getenv("AESOP_TUNNEL_ADDR");
   if((localport = getenv("AESOP_TUNNEL_PORT")) == NULL)
      localport = "3000";

   if((ent = gethostbyname(localaddr == NULL ? "127.0.0.1" : localaddr)) != NULL) {
      ipv4ready = 1;
      sin.sin_family = AF_INET;
      sin.sin_port = htons(atoi(localport));
      memcpy(&sin.sin_addr.s_addr, ent->h_addr_list[0], ent->h_length);
      sinlen = sizeof(struct sockaddr_in);
   }

#ifdef IPV6
   bzero(&hints, sizeof(struct addrinfo));
   hints.ai_family = AF_INET6;
   hints.ai_socktype = SOCK_STREAM;
   if((n = getaddrinfo(localaddr == NULL ? "::1" : localaddr, localport, &hints, &res)) != 0) {
      if(ipv4ready) {
	 sin6.sin6_family = AF_INET6;
	 sin6.sin6_port = htons(atoi(localport));
	 /* IPv4-mapped-IPv6 address	*/
	 memset(sin6.sin6_addr.s6_addr + 10, 0xff, 2);
	 memcpy(sin6.sin6_addr.s6_addr + 12, &sin.sin_addr.s_addr, 4);
	 sin6len = sizeof(struct sockaddr_in6);
	 ipv6ready = 1;
      }
   } else {
      ipv6ready = 1;
      sin6len = res->ai_addrlen;
      sin6 = *(struct sockaddr_in6 *)res->ai_addr;
      freeaddrinfo(res);
   }
#endif   

   if(!ipv4ready && !ipv6ready) {
      fprintf(stderr, "libaesop: aesoptunnel address could not be resolved\n");
      exit(-1);
   }

   init_done = 1;
}

static ssize_t writen(int fd, const void *vptr, size_t n) {
   size_t          nleft;
   ssize_t         nwritten;
   const char      *ptr;

   ptr = vptr;
   nleft = n;
   while (nleft > 0) {
      if((nwritten = write(fd, ptr, nleft)) <= 0) {
	 if(errno == EINTR)
	    nwritten = 0;
	 else
	    return -1;
      }

      nleft -= nwritten;
      ptr   += nwritten;
   }
   return(n);
}

static void restore_socket(int sockfd, int family) {
   int flags;
   int sd;
   struct sockaddr sa;
   int salen = sizeof(struct sockaddr);

   if((flags = fcntl(sockfd, F_GETFL, 0)) == -1)
      flags = 0;

   if(family) {
      sd = socket(family, SOCK_STREAM, 0);
   } else {
      if(getsockname(sockfd, &sa, &salen) == -1)
	 sd = socket(AF_INET, SOCK_STREAM, 0);
      else
	 sd = socket(sa.sa_family, SOCK_STREAM, 0);
   }

   dup2(sd, sockfd);
   close(sd);
   fcntl(sockfd, F_SETFL, flags);
}

int connect(int sockfd, const struct sockaddr *serv_addr, socklen_t addrlen) {
   int socktype = -1;
   int socktypelen = sizeof(socktype);

   if(!init_done)
      _init();

   getsockopt(sockfd, SOL_SOCKET, SO_TYPE, &socktype, &socktypelen);

   if((socktype == SOCK_STREAM) && ((serv_addr->sa_family == AF_INET)
#ifdef IPV6
       || (serv_addr->sa_family == AF_INET6)
#endif
   )) {
      int flags;
#ifdef IPV6
      int changed = 0;
#endif
      desthdr init;
      struct sockaddr *sv;      
      socklen_t svlen;

      memset(&init, 0, sizeof(init));

      init.magic = htonl(MAGIC);

      if(serv_addr->sa_family == AF_INET) {
#ifdef IPV6
	 if(!ipv4ready) {
	    if(sockchange) {
	       restore_socket(sockfd, AF_INET6);
	       changed = 1;
	    } else {
	       fprintf(stderr, "libaesop: Can not connect to IPv6 aesoptunnel-address on IPv4 socket\n");
	       fprintf(stderr, " You can try to set the AESOP_ALLOW_IPV4TOIPV6 environmental variable.\n");
	       fprintf(stderr, " Consult the manual page for more information.\n");
	       exit(-1);
	    }
	 }
#endif
	 init.targetport = ((struct sockaddr_in *)serv_addr)->sin_port;
	 memcpy(init.targetip, &((struct sockaddr_in *)serv_addr)->sin_addr.s_addr, 4);
#ifdef IPV6
	 if(changed) {
	    sv = (struct sockaddr *)&sin6;
	    svlen = sin6len;
	 } else {
#endif
	    sv = (struct sockaddr *)&sin;
	    svlen = sinlen;
	 }
#ifdef IPV6
      }
      else {
	 init.targetport = ((struct sockaddr_in6 *)serv_addr)->sin6_port;
	 memcpy(init.targetip, ((struct sockaddr_in6 *)serv_addr)->sin6_addr.s6_addr, 16);
	 init.control = htons(CNTRL_INET6);
	 sv = (struct sockaddr *)&sin6;
	 svlen = sin6len;
      }     
#endif

       if((flags = fcntl(sockfd, F_GETFL, 0)) == -1) {
	  flags = 0;
       }

       if((flags & O_NONBLOCK) != 0) {
	  fcntl(sockfd, F_SETFL, flags & (~O_NONBLOCK));
       }

       if((*_r_connect)(sockfd, sv, svlen) == -1)
	   return -1;

       if(writen(sockfd, &init, sizeof(init)) != sizeof(init)) {
	  errno = ECONNREFUSED;
	  restore_socket(sockfd, 0);
	  fcntl(sockfd, F_SETFL, flags);
	  return -1;
       }

       fcntl(sockfd, F_SETFL, flags);
       return 0;
   }
   return (*_r_connect)(sockfd, serv_addr, addrlen);
}
