#include "general.h"

#define MAXLINE		4096

int debug = 0;
int daemon_proc = 0;

static void err_doit(int, const char *, va_list);

/* Nonfatal error related to a system call.
 * Print a message and return. */

void err_ret(const char *fmt, ...) {
   va_list ap;

   if(!debug)
      return;

   va_start(ap, fmt);
   err_doit(1, fmt, ap);
   va_end(ap);
   return;
}

/* Fatal error related to a system call.
 * Print a message and terminate. */

void err_sys(const char *fmt, ...) {
   va_list ap;

   va_start(ap, fmt);
   err_doit(1, fmt, ap);
   va_end(ap);
   exit(1);
}

/* Nonfatal error unrelated to a system call.
 * Print a message and return. */

void err_msg(const char *fmt, ...) {
   va_list ap;

   if(!debug)
      return;

   va_start(ap, fmt);
   err_doit(0, fmt, ap);
   va_end(ap);
   return;
}

/* Fatal error unrelated to a system call.
 * Print a message and terminate. */

void err_quit(const char *fmt, ...) {
   va_list ap;

   va_start(ap, fmt);
   err_doit(0, fmt, ap);
   va_end(ap);
   exit(1);
}

/* Print a message and return to caller.
 * Caller specifies "errnoflag" and "level". */

static void err_doit(int errnoflag, const char *fmt, va_list ap) {
   int errno_save, n;
   char	buf[MAXLINE];

   errno_save = errno;		/* value caller might want printed */
   vsnprintf(buf, sizeof(buf)-1, fmt, ap);	/* this is safe */
   n = strlen(buf);
   if(errnoflag)
      snprintf(buf+n, sizeof(buf)-n-1, ": %s", strerror(errno_save));
   strcat(buf, "\n");

   if(daemon_proc) {
      aesoplog("%s", buf);
   } else {
      fflush(stdout);		/* in case stdout and stderr are the same */
      fputs(buf, stderr);
      fflush(stderr);
   }
   return;
}
