/*
   CIPE - encrypted IP over UDP tunneling

   cipe.h - contains definitions, includes etc. common to all modules

   Copyright 1996 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: cipe.h,v 1.12.4.2 1998/11/14 21:14:16 olaf Exp $ */

#ifndef _CIPE_H_
#define _CIPE_H_

#include "crypto.h"


/*** The kernel/user IOCTL interface ***/

/* ioctls for setup and key exchange */
/* #define SIOCxIFCIPxxx   (SIOCDEVPRIVATE+x) */
/* All ioctls are passed a struct ifreq <net/if.h> which contains the
   device name in ifr_name and a pointer to the actual control struct
   in ifr_data. */

/* Get interface parameters. */
#define SIOCGIFCIPPAR   (SIOCDEVPRIVATE+0)
struct  siocgifcippar {
    unsigned long   magic;
    /* SOCKS5 relayer */
    unsigned long   sockshost;
    unsigned short  socksport;
    /* Timeouts (in seconds) */
    int             tmo_keyxchg;
    int             tmo_keylife;
    /* Flags */
    int             mayclear;
    int		    cttl;
};

/* Set interface parameters. */
#define SIOCSIFCIPPAR   (SIOCDEVPRIVATE+1)
struct  siocsifcippar {
    unsigned long   magic;
    /* SOCKS5 relayer */
    unsigned long   sockshost;
    unsigned short  socksport;
    /* Timeouts (in seconds) */
    int             tmo_keyxchg;
    int             tmo_keylife;
    /* Flags */
    int             mayclear;
    int		    cttl;
};

/* Set a key. */
#define SIOCSIFCIPKEY   (SIOCDEVPRIVATE+2)
#define KEY_STATIC      1
#define KEY_SEND        2
#define KEY_RECV        3
#define KEY_INVAL       8
struct  siocsifcipkey {
    unsigned long   magic;
    int which;
    UserKey thekey;
};

/* Attach a socket. */
#define SIOCSIFCIPATT   (SIOCDEVPRIVATE+3)
struct  siocsifcipatt {
    unsigned long   magic;
    int fd;
};


/*** Key exchange related definitions ***/

/* Minimum kxc block. */
#define KEYXCHGBLKMIN    64
/* Maximum kxc block, padded with random bytes */
#define KEYXCHGBLKMAX    (KEYXCHGBLKMIN+256)

/* Type words. Only 8 are possible. */
#define TW_DATA         0x00
#define TW_NEWKEY       0x02
#define TW_CTRL         0x04
#define TW_RSVD2        0x06
#define TW_CTLBITS      0x06 /* Bit mask for all of above */

/* error indication, no valid type word */
#define TW_ERROR        0x01 

/* NEWKEY (key exchange mode 1) subtypes. */
#define NK_RREQ         0 /* not used in protocol */
#define NK_REQ          1 /* send me your new key */
#define NK_IND          2 /* this is my new key   */
#define NK_ACK          3 /* i have your new key  */


/*** Kernel-module internal stuff ***/

#ifdef __KERNEL__

#include <asm/byteorder.h>
#include <linux/types.h>
#include <linux/netdevice.h>
#include <linux/sockios.h>
#include <linux/sched.h>
#include <linux/if_ether.h>
#include <linux/net.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <net/sock.h>

/* We attach an ethernet header to the start of each packet */
#define cipe_pkthdr_len (sizeof(struct ethhdr))

/* The header we add to each packet */
#ifdef VER_SHORT
#define cipehdrlen (sizeof(struct iphdr)+sizeof(struct udphdr))
#else
#define cipehdrlen (sizeof(struct iphdr)+sizeof(struct udphdr)+blockSize)
#endif

/* In the xmit function we copy the packet to a new buffer as we do
 * not know the required space for the device we are going to be
 * sending over, and the protocol layers tend to not allocate enough
 * tail room for us. Encrypted packets also tend to be larger than the
 * unencrypted ones 
 */
#define cipexhdrl ((cipe_pkthdr_len+15)&~15)

/* max. padding at the end */
#if ProtocolVersion == 3 || ProtocolVersion == d
#define cipefootlen 12 /* 7 bytes pad, 1 byte type, 4 bytes CRC */
#else
#define cipefootlen 10 /* 8 bytes pad, 2 bytes CRC */
#endif

/* A CIPE device's parameter block */

#define CIPE_MAGIC 0x5387
struct cipe {
    unsigned short  magic;
    struct device   *dev;
     /* Set by user process */
    __u32           peeraddr;
    __u32           myaddr;
    __u16           peerport;
    __u16           myport;
    __u32           sockshost;
    __u16           socksport;
    short	    cttl;
#ifdef Crypto_IDEA
    Key             key_e, key_d, skey_e, rkey_d;
#endif
#ifdef Crypto_Blowfish
    Key             key, skey, rkey;
    #define key_e   key
    #define key_d   key
    #define skey_e  skey
    #define rkey_d  rkey
#endif
    unsigned long   tmo_keyxchg;
    unsigned long   tmo_keylife;
     /* Internal */
    unsigned long   timekx;
    unsigned long   timeskey;
    unsigned long   timerkey;
    int             cntskey;
    int             cntrkey;
    struct sock     *sock;
     /* Flags */
    char            mayclear;
    char            havekey;
    char            haveskey;
    char            haverkey;
     /* Statistics */
    struct enet_statistics stat;
     /* Socket interface stuff */
    struct proto    *udp_prot;
    struct proto    cipe_proto;
};

#define MAXBLKS     32767  /* max # blocks to encrypt using one key */

/* Define, init and check a struct cipe * variable. */
#define DEVTOCIPE(dev,c,err) \
    struct cipe *c = (struct cipe*)(dev->priv); \
    if (!c || c->magic!=CIPE_MAGIC) return err;

/* Master control struct */
struct cipe_ctrl {
    char             name[8];
    struct cipe      cipe;
    struct device    dev;
};

extern struct cipe_ctrl **cipe_ctrls;
extern int cipe_maxdev;

/* SOCKS5 encapsulation header */
struct sockshdr {
    char             rsv[2];
    char             frag;
    char             atyp;
    __u32            dstaddr    __attribute__((packed));
    __u16            dstport    __attribute__((packed));
};

#ifdef DEBUG
extern int cipe_debug;
#define DEB_CALL   1
#define DEB_INP    2
#define DEB_OUT    4
#define DEB_CRYPT  8
#define DEB_KXC   16
#define dprintk1(l,s,f,a)           if(cipe_debug&l){printk(s f,a);}
#define dprintk2(l,s,f,a,b)         if(cipe_debug&l){printk(s f,a,b);}
#define dprintk3(l,s,f,a,b,c)       if(cipe_debug&l){printk(s f,a,b,c);}
#define dprintk4(l,s,f,a,b,c,d)     if(cipe_debug&l){printk(s f,a,b,c,d);}
#define dprintk5(l,s,f,a,b,c,d,e)   if(cipe_debug&l){printk(s f,a,b,c,d,e);}
#else
#define dprintk1(l,s,f,a)
#define dprintk2(l,s,f,a,b)
#define dprintk3(l,s,f,a,b,c)
#define dprintk4(l,s,f,a,b,c,d)
#define dprintk5(l,s,f,a,b,c,d,e)
#endif /* DEBUG */

/* internal routines */
/* device.c */
extern void cipe_prnpad(unsigned char *buf, int len);
extern void cipe_close(struct cipe *c);
/* sock.c */
extern int cipe_attach(struct device *dev, struct siocsifcipatt *parm);
void cipe_fakenkey(struct cipe *c, char typ);
/* output.c */
extern int cipe_xmit(struct sk_buff *skb, struct device *dev);
/* encaps.c */
extern void cipe_encrypt(struct cipe *c, unsigned char *buf,
			 int *len, int typcode);
extern unsigned short cipe_decrypt(struct cipe *c, unsigned char *buf,
				   int *len);
#ifndef VER_SHORT
void cipe_cryptpad(unsigned char *buf);
#endif

#endif /* __KERNEL__ */

#ifdef VER_CRC32
/* crc32.c */
extern unsigned long crc32(const unsigned char *s, unsigned int len);
#else
/* crc.c */
extern unsigned short block_crc(unsigned char *d, int len);
#endif

#define MIN(a,b) (((a)<(b))?(a):(b))

#ifndef CTLDIR
#define CTLDIR "/etc/cipe/"
#endif

#ifndef DEVNAME
#define DEVNAME "cip" VERNAME CRNAME
#endif

#endif _CIPE_H_
