/*
   CIPE - encrypted IP over UDP tunneling

   device.c - the net device driver

   Copyright 1996 Olaf Titz <olaf@bigred.inka.de>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version
   2 of the License, or (at your option) any later version.
*/
/* $Id: device.c,v 1.15.4.2 1998/11/14 21:14:16 olaf Exp $ */

#include "cipe.h"
#include "version.h"
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/version.h>

/*** Globals ***/

static const char driver_version[]=VERSION;

struct cipe_ctrl **cipe_ctrls = NULL;
int cipe_maxdev = 4;            /* changeable via insmod */
#ifdef DEBUG
int cipe_debug = DEB_CALL;      /* changeable via insmod */
#endif

/* clear all potentially sensitive info */
static void cipe_zero_c(struct cipe *c)
{
    memset(&(c->peeraddr), 0, (char*)&(c->stat)-(char*)&(c->peeraddr));
    /* reset these to sensible values */
    c->tmo_keyxchg = 10*HZ;
    c->tmo_keylife = 10*60*HZ;
}

/* weak but fast PRNG, used for padding only */
static __u32 prnseed;
void cipe_prnpad(unsigned char *buf, int len)
{
    while (len>0) {
	prnseed=prnseed*0x01001001+1;
	if (len>=2) {
	    *(__u16 *)buf=prnseed>>16;
	    len-=2; buf+=2;
	} else {
	    *buf=(prnseed>>24)^jiffies; return;
	}
    }
}


/*** IOCTL handlers ***/

static int cipe_getpar(struct device *dev, struct siocgifcippar *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    parm->sockshost=c->sockshost;
    parm->socksport=c->socksport;
    parm->tmo_keyxchg=c->tmo_keyxchg/HZ;
    parm->tmo_keylife=c->tmo_keylife/HZ;
    parm->mayclear=c->mayclear;
    parm->cttl=c->cttl;
    return 0;
}

static int cipe_setpar(struct device *dev, struct siocsifcippar *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    if (parm->sockshost)
	c->sockshost=parm->sockshost;
    if (parm->socksport)
	c->socksport=parm->socksport;
    if (parm->tmo_keyxchg>10*60*HZ)
	return -EINVAL;
    if (parm->tmo_keyxchg)
	c->tmo_keyxchg=parm->tmo_keyxchg*HZ;
    if (parm->tmo_keylife>24*60*60*HZ)
	return -EINVAL;
    if (parm->tmo_keylife)
	c->tmo_keylife=parm->tmo_keylife*HZ;
    c->mayclear=parm->mayclear;
    c->cttl=parm->cttl;
    return 0;
}

static int cipe_setkey(struct device *dev, struct siocsifcipkey *parm)
{
    DEVTOCIPE(dev,c,-ENODEV);

    dprintk2(DEB_KXC, KERN_INFO, "%s: setkey %d\n", dev->name, parm->which);
    switch (parm->which) {
    case KEY_STATIC:
	ExpandUserKey(parm->thekey, c->key_e);
	InvertKey(c->key_e, c->key_d);
	c->havekey=1;
	break;
    case KEY_SEND:
	ExpandUserKey(parm->thekey, c->skey_e);
	c->timeskey=jiffies+c->tmo_keylife;
	c->cntskey=0;
	c->haveskey=1;
	break;
    case KEY_RECV:
	ExpandUserKey(parm->thekey, c->rkey_d);
	InvertKey(c->rkey_d, c->rkey_d);
	c->timerkey=jiffies+2*c->tmo_keylife; /* allow for fuzz */
	c->cntrkey=0;
	c->haverkey=1;
	break;
    case KEY_STATIC+KEY_INVAL:
	c->havekey=c->haveskey=c->haverkey=0;
	memset(&(c->key_e), 0, sizeof(c->key_e));
	memset(&(c->key_d), 0, sizeof(c->key_d));
	break;
    case KEY_SEND+KEY_INVAL:
	c->haveskey=0;
	memset(&(c->skey_e), 0, sizeof(c->skey_e));
	c->timeskey=jiffies+c->tmo_keyxchg;
	break;
    case KEY_RECV+KEY_INVAL:
	c->haverkey=0;
	memset(&(c->rkey_d), 0, sizeof(c->rkey_d));
	c->timerkey=jiffies+c->tmo_keyxchg;
	break;
    default:
	return -EINVAL;
    }
    return 0;
}


/*** Device operation handlers ***/

int cipe_dev_ioctl(struct device *dev, struct ifreq *ifr, int cmd)
{
    int e;
    if (!suser())
	return -EPERM;

#define doioctl(nam,fun,str) {                                              \
    struct str parm;                                                        \
    dprintk1(DEB_CALL, KERN_INFO, "%s: " nam "\n", dev->name);              \
    if ((e=verify_area(VERIFY_READ, ifr->ifr_data, sizeof(parm)))<0)        \
        return e;                                                           \
    memcpy_fromfs((void*)&parm, (void*)ifr->ifr_data, sizeof(parm));        \
    if (parm.magic!=VERSION_MAGIC) return -EINVAL;                          \
    if ((e=fun(dev, &parm))<0)                                              \
        return e;                                                           \
    if ((e=verify_area(VERIFY_WRITE, ifr->ifr_data, sizeof(parm)))<0)       \
        return e;                                                           \
    memcpy_tofs((void*)ifr->ifr_data, (void*)&parm, sizeof(parm));          \
    return 0; }

    switch (cmd) {
    case SIOCGIFCIPPAR:
	doioctl("getpar", cipe_getpar, siocgifcippar);
    case SIOCSIFCIPPAR:
	doioctl("setpar", cipe_setpar, siocsifcippar);
    case SIOCSIFCIPKEY:
	doioctl("setkey", cipe_setkey, siocsifcipkey);
    case SIOCSIFCIPATT:
	doioctl("attach", cipe_attach, siocsifcipatt);
    default:
	return -EINVAL;
    }

#undef doioctl
}

int cipe_dev_open(struct device *dev)
{
#if ProtocolVersion == d
    __u8 mac0[ETH_ALEN] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    DEVTOCIPE(dev,c,-ENODEV);
    if(memcmp(mac0, dev->dev_addr, ETH_ALEN) == 0)
	return -ENXIO;
#else
    DEVTOCIPE(dev,c,-ENODEV);
#endif
    if (!c->sock)
	return -ENXIO;
    dprintk1(DEB_CALL, KERN_INFO, "%s: opened\n", dev->name);
    return 0;
}

void cipe_close(struct cipe *c)
{
    cipe_zero_c(c);
    dprintk1(DEB_CALL, KERN_INFO, "%s: closed\n", c->dev->name);
}

int cipe_dev_close(struct device *dev)
{
    struct cipe *c = (struct cipe*)(dev->priv);
    if ((!c) || (c->magic!=CIPE_MAGIC)) {
	printk(KERN_WARNING "%s: cipe_close(): no valid struct\n", dev->name);
	return 0;
    }
    if (c->sock) {
	dprintk1(DEB_CALL, KERN_INFO, "%s: closing\n", c->dev->name);
	/* Tell the attached socket we're going down */
	c->sock->shutdown=SHUTDOWN_MASK;
	c->sock->zapped=1;
	c->sock->err=ENXIO;
	c->sock->error_report(c->sock);
    } else {
	cipe_close(c);
    }
    return 0;
}

struct enet_statistics *cipe_get_stats(struct device *dev)
{
    DEVTOCIPE(dev,c,NULL);
    return &(c->stat);
}

#if ProtocolVersion == d
/*** Frame Header Routines - stolen from net/ethernet/eth.c ***/


/* Multi-cast function, just a blank placeholder */

static void
cipe_set_multicast_list (struct device *dev) {

    if (dev->flags&IFF_PROMISC) {
    }

    else if((dev->flags&IFF_ALLMULTI)) {
    }
  
    else if(dev->mc_count) {
    }
  
    else {
    }
}


/*** Initialization and finalization stuff ***/

static int cipe_init_dev(struct device *dev)
{
    __u8 bmac[ETH_ALEN] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
    __u8 smac[ETH_ALEN] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    struct device *eth0; 
    struct cipe *c = (struct cipe*)(dev->priv);

    if (!c)
	return -ENODEV;

    memset(c, 0, sizeof(struct cipe)); /* zero the device struct along */
    c->magic       = CIPE_MAGIC;
    c->dev         = dev;
    cipe_zero_c(c);

    /* Find the address of ethernet 0, if possible and fill in our
     * device source address with it, this is a bit naughty, but 
     * it gives us a unique MAC address
     */
    eth0 = dev_get("eth0");

    if( eth0 && memcmp(eth0->dev_addr, smac, ETH_ALEN) != 0 ) {
	memcpy(smac, eth0->dev_addr, ETH_ALEN);
    }


    /* Call ether_setup to initialise most of the stuff - we will change
     * what we need to afterwards 
     */
    ether_setup(dev);

    /* Device parameters. */
    /* Procedural */
    dev->open                   = cipe_dev_open;
    dev->stop                   = cipe_dev_close;
    dev->hard_start_xmit        = cipe_xmit;
    dev->get_stats              = cipe_get_stats;
    dev->do_ioctl               = cipe_dev_ioctl;
    dev->set_multicast_list     = cipe_set_multicast_list;

    /* "Hardware" */
    /* dev->hard_header_len 	= cipe_pkthdr_len; */
    dev->mtu		        = ETH_DATA_LEN
	                          -cipehdrlen
	                          -cipefootlen
	                          -cipe_pkthdr_len
	                          -sizeof(struct sockshdr);
    memcpy(dev->broadcast, bmac, ETH_ALEN);
    memcpy(dev->dev_addr, smac, ETH_ALEN);

    /* New-style flags */
    dev->flags		|= IFF_NOTRAILERS;
    dev->metric         = 1;
    
    return 0;
}

#else /* if Protocol Version == d */
static int cipe_init_dev(struct device *dev)
{
    int i;
    struct cipe *c = (struct cipe*)(dev->priv);
    if (!c)
	return -ENODEV;

    memset(c, 0, sizeof(struct cipe)); /* zero the device struct along */
    c->magic       = CIPE_MAGIC;
    c->dev         = dev;
    cipe_zero_c(c);

    /* Device parameters. */
    /* Procedural */
    dev->open                   = cipe_dev_open;
    dev->stop                   = cipe_dev_close;
    dev->hard_start_xmit        = cipe_xmit;
    dev->get_stats              = cipe_get_stats;
    dev->hard_header           	= NULL;
    dev->rebuild_header         = NULL;
    dev->do_ioctl               = cipe_dev_ioctl;

    /* "Hardware" */
    dev->type		        = ARPHRD_TUNNEL;
    dev->hard_header_len 	= cipexhdrl;
    dev->mtu		        = ETH_DATA_LEN-cipehdrlen-
	                             cipefootlen-sizeof(struct sockshdr);
    dev->addr_len		= 4;            /* Dummy? */
    dev->tx_queue_len	        = 100; /* matches ethernet */

    for (i = 0; i < DEV_NUMBUFFS; i++)  {
	skb_queue_head_init(&dev->buffs[i]);
    }

    /* New-style flags */
    dev->flags		= IFF_POINTOPOINT|IFF_NOTRAILERS|IFF_NOARP;
    dev->family		= AF_INET;
    dev->pa_alen	= 4;
    dev->metric         = 1;

    return 0;
}

#endif


static int cipe_alloc(int n)
{
    struct cipe_ctrl *cc = kmalloc(sizeof(struct cipe_ctrl), GFP_KERNEL);
    if (!cc)
	return -ENOMEM;
    cipe_ctrls[n] = cc;

    memset((void *)cc, 0, sizeof(struct cipe_ctrl));
    sprintf(cc->name, DEVNAME "%d", n);
    cc->dev.name      = cc->name;
    cc->dev.base_addr = n; /* dummy */
    cc->dev.priv      = (void*)&(cc->cipe);
    cc->dev.next      = NULL;
    cc->dev.init      = cipe_init_dev; /* called by register_netdev */

    if (register_netdev(&(cc->dev))<0) {
	kfree(cc);
	cipe_ctrls[n]=NULL;
	printk(KERN_ERR
	       "%s: register_netdev() failed\n", cc->name);
    }
    return 0;
}

static void cipe_unalloc(int n)
{
    struct cipe_ctrl *cc=cipe_ctrls[n];
    if (!cc)
	return;
    unregister_netdev(&(cc->dev));
    kfree(cc);
    cipe_ctrls[n]=NULL;
}

#ifdef MODULE

int init_module(void)
{
    int i;

     /* sanity check on insmod-provided data */
    if (cipe_maxdev<1)  cipe_maxdev=1;
    if (cipe_maxdev>99) cipe_maxdev=99;

#ifdef DEBUG
    printk(KERN_INFO
	   DEVNAME ": CIPE driver vers %s (c) Olaf Titz 1996-1998, %d channels, debug=%d\n",
	   driver_version, cipe_maxdev, cipe_debug);
#else
    printk(KERN_INFO
	   DEVNAME ": CIPE driver vers %s (c) Olaf Titz 1996-1998, %d channels\n",
	   driver_version, cipe_maxdev);
#endif

    prnseed=~jiffies;
    cipe_ctrls = (struct cipe_ctrl **) kmalloc(sizeof(void*)*cipe_maxdev,
					       GFP_KERNEL);
    if (!cipe_ctrls) {
	printk(KERN_ERR
	       DEVNAME ": failed to allocate master control structure\n");
	return -ENOMEM;
    }
    memset(cipe_ctrls, 0, sizeof(void*)*cipe_maxdev);
    for (i=0; i<cipe_maxdev; ++i)
	if (cipe_alloc(i)<0)
	    return -ENOMEM;
    return 0;
}

void cleanup_module(void)
{
    int i;
    for (i=0; i<cipe_maxdev; ++i)
	cipe_unalloc(i);
    kfree(cipe_ctrls);
}

#else

#error "This does only work as a module."

#endif /* MODULE */

#if LINUX_VERSION_CODE < 131102 /* < 2.0.30 */
/* older kernel not always exported this */

int bad_user_access_length(void)
{
    panic("bad_user_access_length in" DEVNAME);
}

#endif
