#include "CipeIdeaEncryptor.h"
#include "hexdump.h"
#include "crc32.h"

//========================================================================================
//
//========================================================================================
CipeIdeaEncryptor::CipeIdeaEncryptor() : m_EncryptKey (KEY_LIMIT), m_DecryptKey (KEY_LIMIT), m_PreferredStaticKey (KEY_LIMIT)
   {
    for (unsigned long l_Index = 0; l_Index < KEY_LIMIT; ++l_Index) m_HaveKey [l_Index] = FALSE;
   }

//========================================================================================
//                              Key Manipulation
//========================================================================================
void CipeIdeaEncryptor::InstallKey (unsigned char *p_KeyData, CipeEncryptionKeyType p_KeyType)
   {
    SaveKeyData (p_KeyData, p_KeyType);
    InstallKey (p_KeyType);
   }

void CipeIdeaEncryptor::InstallKey (CipeEncryptionKeyType p_KeyType)
   {
    Idea_ExpandUserKey (m_KeyData [p_KeyType], m_Key [p_KeyType]);
    m_HaveKey [p_KeyType] = TRUE;
   }

void CipeIdeaEncryptor::InstallStaticKey (const char *p_TextKey)
   {
    unsigned char l_KeyData [IDEA_KEY_LENGTH + 1], l_Digit, *l_Ptr;
    unsigned long l_Length = strlen (p_TextKey);
    int l_Index = 0, l_Idx = 0;

    for (l_Index = 0; l_Index < l_Length && l_Index < 32; ++l_Index)
       {
        if (p_TextKey [l_Index] >= 'a' && p_TextKey [l_Index] <= 'f')
           l_Digit = p_TextKey [l_Index] + 10 - 'a';
        else if (p_TextKey [l_Index] >= 'A' && p_TextKey [l_Index] <= 'F')
           l_Digit = p_TextKey [l_Index] + 10 - 'A';
        else if (p_TextKey [l_Index] >= '0' && p_TextKey [l_Index] <= '9')
           l_Digit = p_TextKey [l_Index] - '0';
        else
           l_Digit = 0;

        if (l_Index % 2)
           l_KeyData [(int) (l_Index / 2)] |= (l_Digit & 0x0f);
        else
           l_KeyData [(int) (l_Index / 2)] = (l_Digit << 4) & 0xf0;
       }

    InstallKey (l_KeyData, KEY_STATIC_PRIMARY);

    for ((l_Ptr = (unsigned char *) p_TextKey) [l_Length] = l_Index = 0; *l_Ptr;)
       {
        l_KeyData [l_Index] = ((*l_Ptr) << 4) & 0xF0;
        ++l_Ptr;
        if (!*l_Ptr) break;
        l_KeyData [l_Index] |= *l_Ptr & 0x0F;
        ++l_Index;
        ++l_Ptr;
        if (l_Index >= 16) break;
       }

    InstallKey (l_KeyData, KEY_STATIC_SECONDARY);
   }

//========================================================================================
//
//========================================================================================
void CipeIdeaEncryptor::SaveKeyData (unsigned char *p_KeyData, CipeEncryptionKeyType p_KeyType)
   {
    memcpy (m_KeyData [p_KeyType], p_KeyData, IDEA_KEY_LENGTH);
   }

unsigned char *CipeIdeaEncryptor::GenerateDynamicKeyData (CipeEncryptionKeyType p_KeyType)
   {
    SaveKeyData (GenerateDynamicKeyData(), p_KeyType);
    return (unsigned char *) m_KeyData [p_KeyType];
   }

unsigned char *CipeIdeaEncryptor::GenerateDynamicKeyData()
   {
    static unsigned char l_KeyData [IDEA_KEY_LENGTH];
    CryptGenRandom (m_CryptoContext, sizeof (l_KeyData), l_KeyData);
    return (unsigned char *) l_KeyData;
   }

//========================================================================================
//
//========================================================================================
unsigned long CipeIdeaEncryptor::Encrypt (unsigned char *p_Out, unsigned char *p_In, unsigned long p_Length, CipeEncryptionKeyType p_KeyType, unsigned long p_Options)
   {
    #define t_Payload (p_Out + IDEA_DATA_SIZE)                                   // Offset of the payload data
    #define t_Padding (t_Payload + p_Length)                                     // Offset of the padding
    #define t_Options (t_Padding + l_PadLength)                                  // Offset of the Options byte
    #define t_CRC     ((unsigned long *) (t_Options + sizeof (unsigned char)))   // Offset of the CRC

    unsigned char l_PadLength = 7 - ((p_Length + sizeof (unsigned long) + IDEA_DATA_SIZE) & 7);
    unsigned long l_FullLength = p_Length + IDEA_DATA_SIZE + l_PadLength + sizeof (unsigned long) + sizeof (unsigned char);
    unsigned char l_KeyType;

    if (p_KeyType >= KEY_LIMIT)
       return 0;
    else if (m_HaveKey [p_KeyType] && p_KeyType == KEY_DYNAMIC_ENCRYPT)
       m_EncryptKey = p_KeyType;
    else if (m_PreferredStaticKey != KEY_LIMIT)
       m_EncryptKey = m_PreferredStaticKey;
    else if (m_HaveKey [KEY_STATIC_SECONDARY])
       m_EncryptKey = KEY_STATIC_SECONDARY;
    else if (m_HaveKey [KEY_STATIC_PRIMARY])
       m_EncryptKey = KEY_STATIC_PRIMARY;

    CryptGenRandom (m_CryptoContext, IDEA_DATA_SIZE, p_Out);
    CryptGenRandom (m_CryptoContext, l_PadLength, t_Padding);
    memcpy (t_Payload, p_In, p_Length);

    if (m_EncryptKey == KEY_DYNAMIC_ENCRYPT || m_EncryptKey == KEY_DYNAMIC_DECRYPT)
       p_Out [0] |= (unsigned char) 0x80; // Just raise the high bit
    else
       p_Out [0] &= (unsigned char) 0x7f; // Just lower the high bit

    t_Options [0] = ((NK_Type) p_Options & 0x06) | (l_PadLength << 4);
    t_CRC [0] = htonl (crc32 (t_Payload, p_Length + l_PadLength + sizeof (unsigned char)));

    return BlockEncrypt (p_Out, l_FullLength, m_Key [m_EncryptKey]);
   }

unsigned long CipeIdeaEncryptor::Decrypt (unsigned char *p_Out, unsigned char *p_In, unsigned long p_Length, CipeEncryptionKeyType p_KeyType, unsigned long &p_Options)
   {
    #define t_newlen (p_Length - (sizeof (unsigned long) + IDEA_DATA_SIZE)) // Remove CRC and random data from length
    #define t_statickeycount 2

    unsigned long *l_CRC = (unsigned long *) (p_Out + t_newlen);
    unsigned char *l_Options = (unsigned char *) (p_Out + t_newlen - sizeof (unsigned char));
    unsigned long l_DynamicKeyCount = 0, l_Length = 0, l_Index = 0, l_KeyCount = 0;
    CipeEncryptionKeyType l_KeyOrder [KEY_LIMIT], l_StaticOrder [t_statickeycount];

    l_StaticOrder [0] = KEY_STATIC_SECONDARY;
    l_StaticOrder [1] = KEY_STATIC_PRIMARY;

    if (p_In [0] & 0x80 && m_HaveKey [KEY_DYNAMIC_DECRYPT]) // Dynamic key
       {
        l_KeyOrder [l_KeyCount++] = KEY_DYNAMIC_DECRYPT;
       }

    l_DynamicKeyCount = l_KeyCount;

    if (m_PreferredStaticKey == KEY_STATIC_PRIMARY)
       {
        l_StaticOrder [0] = KEY_STATIC_PRIMARY;
        l_StaticOrder [1] = KEY_STATIC_SECONDARY;
       }

    for (l_Index = 0; l_Index < t_statickeycount; ++l_Index) if (m_HaveKey [l_StaticOrder [l_Index]])
       {
        l_KeyOrder [l_KeyCount++] = l_StaticOrder [l_Index];
       }

    for (l_Index = l_Length = 0; p_Length && ! l_Length && l_Index < l_KeyCount; ++l_Index)
       {
        memcpy (p_Out, p_In, p_Length);

        BlockDecrypt (p_Out, p_Length, m_Key [m_DecryptKey = l_KeyOrder [l_Index]]);

        if (htonl (crc32 (p_Out, t_newlen)) == l_CRC [0])
           {
            if (l_Index >= l_DynamicKeyCount) m_PreferredStaticKey = m_DecryptKey;
            l_Length = t_newlen - ((l_Options [0] >> 4) & 0x07) - 1;
            p_Options = l_Options [0] & 0x06;
           }
       }

    return l_Length;
   }

//========================================================================================
//
//========================================================================================
unsigned long CipeIdeaEncryptor::BlockEncrypt (unsigned char *p_Buffer, unsigned long p_Length, Idea_Key &p_Key)
   {
    for (unsigned long l_Index = 0; l_Index < p_Length; l_Index += IDEA_DATA_SIZE)
       {
        Idea_Crypt ((unsigned short *) p_Buffer + l_Index + IDEA_DATA_SIZE, (unsigned short *) p_Buffer + l_Index, p_Key);
       }

    return p_Length;
   }

unsigned long CipeIdeaEncryptor::BlockDecrypt (unsigned char *p_Buffer, unsigned long p_Length, Idea_Key &p_Key)
   {
    Idea_Key l_DecryptionKey;
    //================================================================================
    // Decryption advances the payload by IDEA_DATA_SIZE bytes, which places it at
    // the beginning of the buffer. The decrypted payload is followed by the options
    // byte and the longword crc. The returned length value should not include the crc
    // the options byte will still be preceeded by padding which will be removed by
    // the calling module
    //================================================================================
    Idea_InvertKey (p_Key, l_DecryptionKey);

    for (unsigned long l_Index = 0; l_Index < p_Length; l_Index += IDEA_DATA_SIZE)
       {
        Idea_Crypt ((unsigned short *) p_Buffer + l_Index + IDEA_DATA_SIZE, (unsigned short *) p_Buffer + l_Index, l_DecryptionKey);
       }

    return p_Length;
   }

//========================================================================================
//                                   End of Source
//========================================================================================
