/*
 *      Copyright (C) 2000,2001 Nikos Mavroyanopoulos
 *
 * This file is part of GNUTLS.
 *
 * GNUTLS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GNUTLS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#define LIBGNUTLS_VERSION "0.1.3"

#define GNUTLS_AES GNUTLS_RIJNDAEL

typedef enum BulkCipherAlgorithm { GNUTLS_NULL_CIPHER=1, GNUTLS_ARCFOUR, GNUTLS_3DES, GNUTLS_RIJNDAEL, GNUTLS_TWOFISH, GNUTLS_RIJNDAEL256 } BulkCipherAlgorithm;
typedef enum KXAlgorithm { GNUTLS_KX_RSA=1, GNUTLS_KX_DHE_DSS, GNUTLS_KX_DHE_RSA, GNUTLS_KX_DH_DSS, GNUTLS_KX_DH_RSA, GNUTLS_KX_DH_ANON, GNUTLS_KX_SRP } KXAlgorithm;
typedef enum MACAlgorithm { GNUTLS_NULL_MAC=1, GNUTLS_MAC_MD5, GNUTLS_MAC_SHA } MACAlgorithm;
typedef enum CompressionMethod { GNUTLS_NULL_COMPRESSION=1, GNUTLS_ZLIB } CompressionMethod;
typedef enum ConnectionEnd { GNUTLS_SERVER, GNUTLS_CLIENT } ConnectionEnd;
typedef enum AlertDescription { GNUTLS_CLOSE_NOTIFY, GNUTLS_UNEXPECTED_MESSAGE=10, GNUTLS_BAD_RECORD_MAC=20,
			GNUTLS_DECRYPTION_FAILED, GNUTLS_RECORD_OVERFLOW,  GNUTLS_DECOMPRESSION_FAILURE=30,
			GNUTLS_HANDSHAKE_FAILURE=40, GNUTLS_BAD_CERTIFICATE=42, GNUTLS_UNSUPPORTED_CERTIFICATE,
			GNUTLS_CERTIFICATE_REVOKED, GNUTLS_CERTIFICATE_EXPIRED, GNUTLS_CERTIFICATE_UNKNOWN,
			GNUTLS_ILLEGAL_PARAMETER, GNUTLS_UNKNOWN_CA, GNUTLS_ACCESS_DENIED, GNUTLS_DECODE_ERROR=50,
			GNUTLS_DECRYPT_ERROR, GNUTLS_EXPORT_RESTRICTION=60, GNUTLS_PROTOCOL_VERSION=70,
			GNUTLS_INSUFFICIENT_SECURITY, GNUTLS_INTERNAL_ERROR=80, GNUTLS_USER_CANCELED=90,
			GNUTLS_NO_RENEGOTIATION=100
			} AlertDescription;

#define GNUTLS_Version int
#define GNUTLS_TLS1 0
#define GNUTLS_SSL3 1

struct GNUTLS_STATE_INT;
typedef struct GNUTLS_STATE_INT* GNUTLS_STATE;

/* internal functions */

int gnutls_init(GNUTLS_STATE * state, ConnectionEnd con_end);
int gnutls_deinit(GNUTLS_STATE state);
int gnutls_close(int cd, GNUTLS_STATE state);
int gnutls_handshake(int cd, GNUTLS_STATE state);
int gnutls_check_pending(GNUTLS_STATE state);
int gnutls_send_hello_request(int cd, GNUTLS_STATE state);

AlertDescription gnutls_get_last_alert( GNUTLS_STATE state);

/* get information on the current state */
BulkCipherAlgorithm gnutls_get_current_cipher( GNUTLS_STATE state);
KXAlgorithm gnutls_get_current_kx( GNUTLS_STATE state);
MACAlgorithm gnutls_get_current_mac_algorithm( GNUTLS_STATE state);
CompressionMethod gnutls_get_current_compression_method( GNUTLS_STATE state);

/* the name of the specified algorithms */
char *gnutls_cipher_get_name(BulkCipherAlgorithm);
char *gnutls_mac_get_name(MACAlgorithm);
char *gnutls_compression_get_name(CompressionMethod);
char *gnutls_kx_get_name(KXAlgorithm algorithm);


/* error functions */
int gnutls_is_fatal_error( int error);
void gnutls_perror( int error);
char* gnutls_strerror(int error);

ssize_t gnutls_send(int cd, GNUTLS_STATE state, void *data, size_t sizeofdata, int flags);
ssize_t gnutls_recv(int cd, GNUTLS_STATE state, void *data, size_t sizeofdata, int flags);
ssize_t gnutls_write(int cd, GNUTLS_STATE state, void *data, size_t sizeofdata);
ssize_t gnutls_read(int cd, GNUTLS_STATE state, void *data, size_t sizeofdata);

/* functions to set priority of cipher suites */
void gnutls_set_cipher_priority( GNUTLS_STATE state, int num, ...);
void gnutls_set_mac_priority( GNUTLS_STATE state, int num, ...);
void gnutls_set_compression_priority( GNUTLS_STATE state, int num, ...);
void gnutls_set_kx_priority( GNUTLS_STATE state, int num, ...);


/* set our version - 0 for TLS 1.0 and 1 for SSL3 */
void gnutls_set_current_version(GNUTLS_STATE state, GNUTLS_Version version); 

/* get/set session */
int gnutls_set_current_session( GNUTLS_STATE state, void* session, int session_size);
int gnutls_get_current_session( GNUTLS_STATE state, void* session, int *session_size);
/* returns the session ID */
int gnutls_get_current_session_id( GNUTLS_STATE state, void* session, int *session_size);

int gnutls_set_lowat( GNUTLS_STATE state, int num);
int gnutls_set_cache_expiration( GNUTLS_STATE state, int seconds);
int gnutls_set_db_name( GNUTLS_STATE state, char* filename);	
int gnutls_clean_db( GNUTLS_STATE state);

/* returns libgnutls version */
const char* gnutls_check_version();

/* Functions for setting/clearing credentials */
int gnutls_clear_creds( GNUTLS_STATE state);
/* cred is a structure defined by the kx algorithm */
int gnutls_set_kx_cred( GNUTLS_STATE, KXAlgorithm kx, void* cred);
const void* gnutls_get_auth_info( GNUTLS_STATE);

/* Credential structures for SRP - used in gnutls_set_cred(); */
typedef struct {
	char* username;
	char* password;
} SRP_CLIENT_CREDENTIALS;

typedef struct {
	char* password_file;
	char* password_conf_file;
} SRP_SERVER_CREDENTIALS;

typedef struct {
	char username[256];
} SRP_AUTH_INFO;

/* Credential structures for DH_ANON */

typedef struct {
	int bits;
} DH_ANON_SERVER_CREDENTIALS;

typedef struct {
	int bits;
} DH_ANON_AUTH_INFO;


/* error codes appended here */
#define	GNUTLS_E_MAC_FAILED  -1
#define	GNUTLS_E_UNKNOWN_CIPHER -2
#define	GNUTLS_E_UNKNOWN_COMPRESSION_ALGORITHM -3
#define	GNUTLS_E_UNKNOWN_MAC_ALGORITHM -4
#define	GNUTLS_E_UNKNOWN_ERROR -5
#define	GNUTLS_E_UNKNOWN_CIPHER_TYPE -6
#define	GNUTLS_E_LARGE_PACKET -7
#define GNUTLS_E_UNSUPPORTED_VERSION_PACKET -8
#define GNUTLS_E_UNEXPECTED_PACKET_LENGTH -9
#define GNUTLS_E_INVALID_SESSION -10
#define GNUTLS_E_UNABLE_SEND_DATA -11
#define GNUTLS_E_FATAL_ALERT_RECEIVED -12
#define GNUTLS_E_RECEIVED_BAD_MESSAGE -13
#define GNUTLS_E_RECEIVED_MORE_DATA -14
#define GNUTLS_E_UNEXPECTED_PACKET -15
#define GNUTLS_E_WARNING_ALERT_RECEIVED -16
#define GNUTLS_E_CLOSURE_ALERT_RECEIVED -17
#define GNUTLS_E_ERROR_IN_FINISHED_PACKET -18
#define GNUTLS_E_UNEXPECTED_HANDSHAKE_PACKET -19
#define GNUTLS_E_UNKNOWN_KX_ALGORITHM -20
#define	GNUTLS_E_UNKNOWN_CIPHER_SUITE -21
#define	GNUTLS_E_UNWANTED_ALGORITHM -22
#define	GNUTLS_E_MPI_SCAN_FAILED -23
#define GNUTLS_E_DECRYPTION_FAILED -24
#define GNUTLS_E_MEMORY_ERROR -25
#define GNUTLS_E_DECOMPRESSION_FAILED -26
#define GNUTLS_E_COMPRESSION_FAILED -27
#define GNUTLS_E_AGAIN -28
#define GNUTLS_E_EXPIRED -29
#define GNUTLS_E_DB_ERROR -30
#define GNUTLS_E_PWD_ERROR -31
#define GNUTLS_E_INSUFICIENT_CRED -32
#define GNUTLS_E_HASH_FAILED -33
#define GNUTLS_E_PARSING_ERROR -34
#define	GNUTLS_E_MPI_PRINT_FAILED -35
#define GNUTLS_E_AUTH_FAILED -36
#define GNUTLS_E_GOT_HELLO_REQUEST -37
#define GNUTLS_E_GOT_APPLICATION_DATA -38
#define GNUTLS_E_UNIMPLEMENTED_FEATURE -50

