/* cipher.c - Cipher filters
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <stdio.h>
#include <gcrypt.h>

#include "opencdk.h"
#include "main.h"

/**
 * cdk_md_filter - MD filter.
 * @ctx: The filter context.
 * @control: Filter control id.
 * @buf: The iobuf to scan.
 *
 * Drop-in filter for the IOBUF implementation. The data that is
 * passed will be hashed with the GCRYPT MD context in the filter.
 **/
int
cdk_md_filter( md_filter_s *mfx, int control, CDK_IOBUF buf )
{
    CDK_BSTRING a;
    size_t len = 0;

    if ( !mfx || !buf || !mfx->md )
        return CDKERR_INV_VALUE;

    if ( control == IOBUF_CTRL_FLUSH ) {
        a = cdk_iobuf_read_mem( buf, 0 );
        if ( !a )
            return CDKERR_INV_VALUE;
        len = cdk_iobuf_get_length( buf );
        gcry_md_write( mfx->md, a->d, len );
        cdk_free( a );
    }
    
    return 0;
} /* cdk_md_filter */

static int
write_header( cipher_filter_s *cfx, CDK_IOBUF buf )
{
    CDK_PACKET pkt;
    cdkPKT_encrypted ed;
    byte temp[18];
    size_t blocksize = 0;
    int use_mdc = 0, cflags = 0;
    int nprefix = 0;
    int rc = 0;

    blocksize = gcry_cipher_get_algo_blklen( cfx->dek->algo );
    if ( blocksize < 8 || blocksize > 16 )
        return CDKERR_INV_ALGO;

    use_mdc = cfx->dek->use_mdc;
    if ( blocksize != 8 )
        use_mdc = 1; /* Hack: enable it for all modern ciphers */
  
    init_packet( &ed );
    ed.len = cfx->datalen;
    ed.extralen = blocksize + 2;
    if ( use_mdc ) {
        ed.mdc_method = GCRY_MD_SHA1;
        cfx->mdc_hash = gcry_md_open( GCRY_MD_SHA1, 0 );
        if ( !cfx->mdc_hash )
            return CDKERR_GCRY;
    }

    init_packet( &pkt );
    pkt.pkttype = use_mdc? PKT_ENCRYPTED_MDC: PKT_ENCRYPTED;
    pkt.pkt.encrypted = &ed;
    rc = cdk_pkt_build( buf, &pkt );
    if ( rc ) {
        gcry_md_close( cfx->mdc_hash );
        return rc;
    }
    nprefix = blocksize;
    gcry_randomize( temp, nprefix, GCRY_STRONG_RANDOM );
    temp[nprefix] = temp[nprefix-2];
    temp[nprefix+1] = temp[nprefix-1];
    cflags = GCRY_CIPHER_SECURE;
    if ( !use_mdc )
        cflags |= GCRY_CIPHER_ENABLE_SYNC;
    cfx->cipher_hd = gcry_cipher_open( cfx->dek->algo,
                                       OPENPGP_DEF_MODE,
                                       cflags );
    if ( !cfx->cipher_hd )
        return CDKERR_GCRY;
    gcry_cipher_setkey( cfx->cipher_hd, cfx->dek->key, cfx->dek->keylen );
    gcry_cipher_setiv( cfx->cipher_hd, NULL, 0 );
    if ( cfx->mdc_hash )
        gcry_md_write( cfx->mdc_hash, temp, nprefix + 2 );
    gcry_cipher_encrypt( cfx->cipher_hd, temp, nprefix + 2, NULL, 0 );
    gcry_cipher_sync( cfx->cipher_hd );
    rc = cdk_iobuf_write( buf, temp, nprefix + 2 );

    return rc;
} /* write_header */

int
cipher_encode( cipher_filter_s *cfx, CDK_IOBUF inp, CDK_IOBUF out )
{
    CDK_BSTRING plain;
    byte pktdata[22];
    int algo = GCRY_MD_SHA1;
    int rc = 0;

    if ( !cfx || !inp || !out )
        return CDKERR_INV_VALUE;
    
    plain = cdk_iobuf_read_mem( inp, 0 );
    if ( !plain )
        return CDKERR_INV_VALUE;
    if ( cfx->mdc_hash )
        gcry_md_write( cfx->mdc_hash, plain->d, plain->len );

    rc = gcry_cipher_encrypt( cfx->cipher_hd, plain->d, plain->len, NULL, 0 );
    if ( !rc )
        rc = cdk_iobuf_write( out, plain->d, plain->len );
    if ( !rc && cfx->mdc_hash ) {
        if ( gcry_md_get_algo_dlen( algo ) != 20 )
            return CDKERR_INV_ALGO;
        /* we must hash the prefix of the MDC packet here */
        pktdata[0] = 0xd3;
        pktdata[1] = 0x14;
        gcry_md_putc( cfx->mdc_hash, pktdata[0] );
        gcry_md_putc( cfx->mdc_hash, pktdata[1] );
        gcry_md_final( cfx->mdc_hash );
        memcpy( pktdata + 2, gcry_md_read( cfx->mdc_hash, algo ) , 20 );
        gcry_cipher_encrypt( cfx->cipher_hd, pktdata, 22, NULL, 0 );
        gcry_md_close( cfx->mdc_hash );
        cfx->mdc_hash = NULL;
        rc = cdk_iobuf_write( out, pktdata, 22 );
    }
    
    return rc;
} /* cipher_encode */

int
cdk_cipher_filter( cipher_filter_s *cfx, int control, CDK_IOBUF outp )
{
    int rc = 0;
    
    if ( control == IOBUF_CTRL_FLUSH ) {
        rc = write_header( cfx, outp );
        if ( !rc )
            rc = cipher_encode( cfx, cfx->plain, outp );
    }
    else if ( control == IOBUF_CTRL_UNDERFLOW ) {
    }
    
    return rc;
} /* cdk_cipher_filter */

