/* kbnode.c -  keyblock node utility functions
 *   Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.
 *   Copyright (C) 2002 Timo Schulz
 *
 * This file is part of GnuPG.
 *
 * GnuPG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GnuPG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"

#define USE_UNUSED_NODES 1

static CDK_KBNODE unused_nodes;

static CDK_KBNODE
alloc_node( void )
{
    CDK_KBNODE n;

    n = unused_nodes;
    if ( n )
        unused_nodes = n->next;
    else {
        n = cdk_alloc( sizeof *n );
        if ( !n )
            return NULL;
    }
    n->next = NULL;
    n->pkt = NULL;
    n->flag = 0;
    n->private_flag=0;
    
    return n;
} /* alloc_node */

static void
free_node( CDK_KBNODE n )
{
    if( n ) {
#if USE_UNUSED_NODES
        n->next = unused_nodes;
        unused_nodes = n;
#else
        cdk_free( n );
#endif
    }
} /* free_node */

CDK_KBNODE
cdk_kbnode_new( CDK_PACKET *pkt )
{
    CDK_KBNODE n = alloc_node();
    if ( !n )
        return NULL;
    n->pkt = pkt;
  
    return n;
} /* cdk_kbnode_new */

CDK_KBNODE
cdk_kbnode_clone( CDK_KBNODE node )
{
    CDK_KBNODE n = alloc_node();
    if ( !n )
        return NULL;
    n->pkt = node->pkt;
    n->private_flag = node->private_flag | 2; /* mark cloned */
  
    return n;
} /* cdk_kbnode_clone */

void
cdk_kbnode_release( CDK_KBNODE n )
{
    CDK_KBNODE n2;

    while ( n ) {
        n2 = n->next;
        n->pkt->pkttype = 0;
        if( !is_cloned_kbnode(n) )
            cdk_pkt_release( n->pkt );        
        free_node( n );
        n = n2;
    }
} /* cdk_kbnode_release */

/****************
 * Delete NODE.
 * Note: This only works with walk_kbnode!!
 */
void
cdk_kbnode_delete( CDK_KBNODE node )
{
    if ( node )
        node->private_flag |= 1;
} /* cdk_kbnode_delete */

/****************
 * Append NODE to ROOT.  ROOT must exist!
 */
void
cdk_kbnode_add( CDK_KBNODE root, CDK_KBNODE node )
{
    CDK_KBNODE n1;

    for(n1=root; n1->next; n1 = n1->next)
        ;
    n1->next = node;
} /* cdk_kbnode_add */

/****************
 * Insert NODE into the list after root but before a packet which is not of
 * type PKTTYPE (only if PKTTYPE != 0)
 */
void
cdk_kbnode_insert( CDK_KBNODE root, CDK_KBNODE node, int pkttype )
{
    if ( !pkttype ) {
        node->next = root->next;
        root->next = node;
    }
    else {
        CDK_KBNODE n1;

        for (n1=root; n1->next;	n1 = n1->next)
            if ( pkttype != n1->next->pkt->pkttype ) {
                node->next = n1->next;
                n1->next = node;
                return;
            }
        /* no such packet, append */
        node->next = NULL;
        n1->next = node;
    }
} /* cdk_kbnode_insert */

/****************
 * Find the previous node (if PKTTYPE = 0) or the previous node
 * with pkttype PKTTYPE in the list starting with ROOT of NODE.
 */
CDK_KBNODE
cdk_kbnode_find_prev( CDK_KBNODE root, CDK_KBNODE node, int pkttype )
{
    CDK_KBNODE n1;

    for ( n1=NULL; root && root != node; root = root->next ) {
        if ( !pkttype || root->pkt->pkttype == pkttype )
            n1 = root;
    }
  
    return n1;
} /* cdk_kbnode_find_prev */

/****************
 * Ditto, but find the next packet.  The behaviour is trivial if
 * PKTTYPE is 0 but if it is specified, the next node with a packet
 * of this type is returned.  The function has some knowledge about
 * the valid ordering of packets: e.g. if the next signature packet
 * is requested, the function will not return one if it encounters
 * a user-id.
 */
CDK_KBNODE
cdk_kbnode_find_next( CDK_KBNODE node, int pkttype )
{
    for( node=node->next ; node; node = node->next ) {
        if( !pkttype )
            return node;
        else if( pkttype == PKT_USER_ID 
                 && (	node->pkt->pkttype == PKT_PUBLIC_KEY
                        || node->pkt->pkttype == PKT_SECRET_KEY ) )
            return NULL;
        else if( pkttype == PKT_SIGNATURE
                 && (	node->pkt->pkttype == PKT_USER_ID
                        || node->pkt->pkttype == PKT_PUBLIC_KEY
                        || node->pkt->pkttype == PKT_SECRET_KEY ) )
            return NULL;
        else if( node->pkt->pkttype == pkttype )
            return node;
    }
  
    return NULL;
} /* cdk_kbnode_find_next */

CDK_KBNODE
cdk_kbnode_find( CDK_KBNODE node, int pkttype )
{
    for ( ; node; node = node->next ) {
        if( node->pkt->pkttype == pkttype )
            return node;
    }
  
    return NULL;
} /* cdk_kbnode_find */

/****************
 * Walk through a list of kbnodes. This function returns
 * the next kbnode for each call; before using the function the first
 * time, the caller must set CONTEXT to NULL (This has simply the effect
 * to start with ROOT).
 */
CDK_KBNODE
cdk_kbnode_walk( CDK_KBNODE root, CDK_KBNODE *context, int all )
{
    CDK_KBNODE n;

    do {
        if ( !*context ) {
            *context = root;
            n = root;
        }
        else {
            n = (*context)->next;
            *context = n;
        }
    } while( !all && n && is_deleted_kbnode(n) );

    return n;
} /* cdk_kbnode_walk */

void
cdk_kbnode_clear_flags( CDK_KBNODE n )
{
    for ( ; n; n = n->next ) {
        n->flag = 0;
    }
} /* cdk_kbnode_clear_flags */

/****************
 * Commit changes made to the kblist at ROOT. Note that ROOT my change,
 * and it is therefore passed by reference.
 * The function has the effect of removing all nodes marked as deleted.
 * returns true if any node has been changed
 */
int
cdk_kbnode_commit( CDK_KBNODE *root )
{
    CDK_KBNODE n, nl;
    int changed = 0;

    for ( n = *root, nl=NULL; n; n = nl->next ) {
        if ( is_deleted_kbnode(n) ) {
            if( n == *root )
                *root = nl = n->next;
            else
                nl->next = n->next;
            if( !is_cloned_kbnode(n) ) {
                cdk_pkt_release( n->pkt );
                cdk_free( n->pkt );
            }
            free_node( n );
            changed = 1;
        }
        else
            nl = n;
    }
    
    return changed;
} /* cdk_kbnode_commit */

void
cdk_kbnode_remove( CDK_KBNODE *root, CDK_KBNODE node )
{
    CDK_KBNODE n, nl;

    for( n = *root, nl=NULL; n; n = nl->next ) {
        if( n == node ) {
            if( n == *root )
                *root = nl = n->next;
            else
                nl->next = n->next;
            if( !is_cloned_kbnode(n) ) {
                cdk_pkt_release( n->pkt );
                cdk_free( n->pkt );
            }
            free_node( n );
        }
        else
            nl = n;
    }
} /* cdk_kbnode_remove */


/****************
 * Move NODE behind right after WHERE or to the beginning if WHERE is NULL.
 */
void
cdk_kbnode_move( CDK_KBNODE *root, CDK_KBNODE node, CDK_KBNODE where )
{
    CDK_KBNODE tmp, prev;

    if( !root || !*root || !node )
        return;  /* sanity check */
    for( prev = *root; prev && prev->next != node; prev = prev->next )
        ;
    if( !prev )
        return; /* node is not in the list */

    if( !where ) {  /* move node before root */
        if( node == *root ) /* move to itself */
            return;
        prev->next = node->next;
        node->next = *root;
        *root = node;
        return;
    }
    /* move it after where */
    if( node == where )
        return;
    tmp = node->next;
    node->next = where->next;
    where->next = node;
    prev->next = tmp;
} /* cdk_kbnode_move */

void
cdk_kbnode_dump( CDK_KBNODE node )
{
    for(; node; node = node->next ) {
        const char *s;
        switch( node->pkt->pkttype ) {
        case 0:		s="empty"; break;
        case PKT_PUBLIC_KEY:	s="public-key"; break;
        case PKT_SECRET_KEY:	s="secret-key"; break;
        case PKT_SECRET_SUBKEY: s= "secret-subkey"; break;
        case PKT_PUBKEY_ENC:	s="public-enc"; break;
        case PKT_SIGNATURE:	s="signature"; break;
        case PKT_ONEPASS_SIG: s="onepass-sig"; break;
        case PKT_USER_ID:	s="user-id"; break;
        case PKT_PUBLIC_SUBKEY: s="public-subkey"; break;
        case PKT_COMMENT:	s="comment"; break;
        case PKT_RING_TRUST:	s="trust"; break;
        case PKT_PLAINTEXT:	s="plaintext"; break;
        case PKT_COMPRESSED:	s="compressed"; break;
        case PKT_ENCRYPTED:	s="encrypted"; break;
        default:		s="unknown"; break;
        }
        fprintf(stderr, "node %p %02x/%02x type=%s",
                node, node->flag, node->private_flag, s);
        if( node->pkt->pkttype == PKT_USER_ID ) {
            cdkPKT_user_id *uid = node->pkt->pkt.user_id;
            fputs("  \"", stderr);
            fprintf( stderr, "%s", uid->name );
            fprintf (stderr, "\" .%c%c%c\n",
                     /* we don't have a expired flag */
                     uid->is_revoked? 'r':'.',
                     uid->created?    'v':'.',
                     uid->is_primary? 'p':'.' );
        }
        else if( node->pkt->pkttype == PKT_SIGNATURE ) {
            fprintf(stderr, "  class=%02x keyid=%08lX ts=%lu\n",
                    node->pkt->pkt.signature->sig_class,
                    (u32)node->pkt->pkt.signature->keyid[1],
                    (u32)node->pkt->pkt.signature->timestamp);
        }
        else if( node->pkt->pkttype == PKT_PUBLIC_KEY
                 || node->pkt->pkttype == PKT_PUBLIC_SUBKEY ) {
            cdkPKT_public_key *pk = node->pkt->pkt.public_key;
            fprintf(stderr, "  keyid=%08lX a=%d u=%d %c%c\n",
                    (u32)cdk_pk_get_keyid( pk, NULL ),
                    pk->pubkey_algo, pk->pubkey_usage,
                    pk->has_expired? 'e':'.',  
                    pk->is_revoked?  'r':'.');
        }
        else
            fputs("\n", stderr);
    }
} /* cdk_kbnode_dump */



