/* keyid.c - key ID and fingerprint handling
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "main.h"

/**
 * cdk_pk_get_fingerprint - Calculate the fingerprint of the PK.
 * @pk: The public key.
 * @fpr: The fingerprint of the key.
 *
 * Returns the fingerprint of the given public key.
 **/
int
cdk_pk_get_fingerprint( cdkPKT_public_key *pk, byte *fpr )
{
    GCRY_MD_HD hd;
    int digest_algo;
    int digest_size = 0;
  
    if ( !pk || !fpr )
        return CDKERR_INV_VALUE;
  
    if ( pk->version < 4 && is_RSA(pk->pubkey_algo) )
        digest_algo = GCRY_MD_MD5; /* special */
    else
        digest_algo = pk->version < 4? GCRY_MD_RMD160 : GCRY_MD_SHA1;
    digest_size = gcry_md_get_algo_dlen( digest_algo );
    hd = gcry_md_open( digest_algo, 0 );
    if ( !hd )
        return CDKERR_GCRY;
    cdk_hash_public_key( pk, hd );
    gcry_md_final( hd );
    memcpy( fpr, gcry_md_read( hd, digest_algo ), digest_size );
    cdk_md_close( &hd );
    if ( digest_size == 16 ) {
        fpr[16] = fpr[17] = 0;
        fpr[18] = fpr[19] = 0;
    }
  
    return 0;    
} /* cdk_pk_get_fingerprint */

static u32
fetch_v3_key_byfpr( const byte *fpr, size_t fprlen, u32 *keyid )
{
    CDK_KEYDB_HD khd;
    CDK_KBNODE pk, p;
    u32 lowbits = 0;
    int pos = 0, rc = 0;

    if ( !fpr || fprlen != 16 || !keyid )
        return CDKERR_INV_VALUE;
    
    rc = cdk_keydb_find_idx( 0, &pos );
    if ( rc )
        return 0;
    khd = cdk_keydb_get_ctx( pos );
    rc = cdk_keydb_get_byfpr( khd, fpr, &pk );
    if ( !rc && keyid ) {
        p = cdk_kbnode_find( pk, PKT_PUBLIC_KEY );
        if ( p ) {
            cdk_pk_get_keyid( p->pkt->pkt.public_key, keyid );
            lowbits = keyid[1];
        }
        cdk_kbnode_release( pk );
    }
    return lowbits;
} /* fetch_v3_key_byfpr */

u32
cdk_pk_fingerprint_get_keyid( const byte *fpr, size_t fprlen, u32 *keyid )
{
    u32 lowbits = 0;
  
    if ( !fpr || !fprlen )
        return CDKERR_INV_VALUE;

    /* in this case we say the key is a V3 RSA key and we can't
       use the fingerprint to get the keyid. */
    if ( fprlen != 20 )
        lowbits = fetch_v3_key_byfpr( fpr, fprlen, keyid );
    else if ( keyid ) {
        keyid[0] = (fpr[12] << 24)| (fpr[13] << 16) | (fpr[14] << 8) | fpr[15];
        keyid[1] = (fpr[16] << 24)| (fpr[17] << 16) | (fpr[18] << 8) | fpr[19];
        lowbits = keyid[1];
    }
    else
        lowbits = (fpr[16] << 24) | (fpr[17] << 16) | (fpr[18] << 8) | fpr[19];
  
    return lowbits;
} /* cdk_pk_fingerprint_get_keyid */

/**
 * cdk_pk_get_keyid - Extract the keyID of the PK.
 * @pk: The public key.
 * @keyid: The key ID of the key.
 *
 * Returns the key ID of the key.
 **/
u32
cdk_pk_get_keyid( cdkPKT_public_key *pk, u32 *keyid )
{
    u32 lowbits = 0;
    byte buf[24];
  
    if ( !pk )
        return 0;
  
    if ( !pk->keyid[0] || !pk->keyid[1] ) {
        if ( pk->version < 4 && is_RSA( pk->pubkey_algo ) ) {
            size_t n = pk->mpi[0]->bytes;
            const byte *p = pk->mpi[0]->data + 2;
            pk->keyid[0] = p[n-8] << 24 | p[n-7] << 16 | p[n-6] << 8 | p[n-5];
            pk->keyid[1] = p[n-4] << 24 | p[n-3] << 16 | p[n-2] << 8 | p[n-1];
        }
        else if ( pk->version == 4 ) {
            cdk_pk_get_fingerprint( pk, buf );
            pk->keyid[0] = buffer_to_u32( buf + 12 );
            pk->keyid[1] = buffer_to_u32( buf + 16 );
        }
    }
    lowbits = pk->keyid[1];
    if ( keyid ) {
        keyid[0] = pk->keyid[0];
        keyid[1] = pk->keyid[1];
    }
    return lowbits;
} /* cdk_pk_get_keyid */

u32
cdk_sk_get_keyid( cdkPKT_secret_key *sk, u32 *keyid )
{
    if ( !sk || !sk->pk )
        return 0;
  
    return cdk_pk_get_keyid( sk->pk, keyid );
} /* cdk_sk_get_keyid */

u32
cdk_sig_get_keyid( cdkPKT_signature *sig, u32 *keyid )
{
    u32 lowbits = 0;
  
    if ( !sig )
        return 0;

    lowbits = sig->keyid[1];
    if ( keyid ) {    
        keyid[0] = sig->keyid[0];
        keyid[1] = sig->keyid[1];
    }

    return lowbits;
} /* cdk_sig_get_keyid */


