/* main.h - Private header file
 *  Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef OPENCDK_MAIN_H
#define OPENCDK_MAIN_H

#define is_main_key(x) (((x) == PKT_PUBLIC_KEY) || ((x) == PKT_SECRET_KEY))
#define is_sub_key(x) (((x) == PKT_PUBLIC_SUBKEY ||((x) == PKT_SECRET_SUBKEY)))
#define is_public(x) (((x) == PKT_PUBLIC_KEY || ((x) == PKT_PUBLIC_SUBKEY)))
#define is_secret(x) (((x) == PKT_SECRET_KEY || ((x) == PKT_SECRET_SUBKEY)))

enum {
    OPENPGP_ARMOR = 100,
    OPENPGP_COMPRESS = 101,
    OPENPGP_TEXTMODE = 102,
    OPENPGP_DIGEST = 103,
    OPENPGP_CIPHER = 104
};

#define DEBUG_PKT (_cdk_get_logging() & 2)

#define KEY_CAN_ENCRYPT(a) (openpgp_pk_algo_usage((a)) & GCRY_PK_USAGE_ENCR)
#define KEY_CAN_SIGN(a) (openpgp_pk_algo_usage((a)) & GCRY_PK_USAGE_SIGN)

#ifndef DIM
#define DIM(v) (sizeof(v)/sizeof((v)[0]))
#define DIMof(type,member)   DIM(((type *)0)->member)
#endif

#define init_packet(ctx) memset( (ctx), 0, sizeof (ctx) )

void _cdk_iobuf_hexdump( CDK_IOBUF buf, size_t pos, size_t len );

/*-- logging.c --*/
int _cdk_get_logging( void );

/*-- encrypt.c --*/
CDK_IOBUF _cdk_make_plaintext( CDK_IOBUF inp, const char *filename, int mode );

/*-- misc.c --*/
u32 make_timestamp( void );

u32 buffer_to_u32( const byte *buffer );
void u32_to_buffer( u32 u, byte *buffer );

int openpgp_cipher_test_algo( int algo );
int openpgp_pk_test_algo( int algo, unsigned int usage_flags );
int openpgp_pk_algo_usage( int algo );
int openpgp_md_test_algo( int algo );

int openpgp_get_opt_val( GCRY_SEXP opt, int id );

void cdk_md_close( GCRY_MD_HD *r_hd );
void cdk_cipher_close( GCRY_CIPHER_HD *r_hd );

int ascii_strcasecmp( const char *a, const char *b );
const char* ascii_memistr( const char *buf, size_t buflen, const char *sub );
unsigned trim_trailing_ws( byte *line, unsigned len );

/*-- new-packet.c --*/
int subpkt_calc_size( struct cdk_subpkt_s *p );
int cdk_subpkt_calc_size( CDK_SUBPKT pkt );
int subpkt_copy( struct cdk_subpkt_s **r_dst, struct cdk_subpkt_s *src );
int cdk_subpkt_hash( CDK_SUBPKT pkt, size_t hashed_size, GCRY_MD_HD hd );
cdk_prefitem_t* _cdk_prefs_copy( const cdk_prefitem_t *prefs );

int _cdk_compare_pks( cdkPKT_public_key *a, cdkPKT_public_key *b );
int copy_pk_parts_to_sk( cdkPKT_public_key *pk, cdkPKT_secret_key *sk );

cdkPKT_signature* cdk_sig_new( void );
void cdk_sig_release( cdkPKT_signature *sig );
void cdk_sk_release( cdkPKT_secret_key *sk );
void cdk_pk_release( cdkPKT_public_key *pk );
void cdk_user_id_release( cdkPKT_user_id *uid );

int cdk_copy_uid( cdkPKT_user_id **dst, cdkPKT_user_id *src );
int cdk_copy_sig( cdkPKT_signature **dst, cdkPKT_signature *src );
int cdk_copy_pk( cdkPKT_public_key **dst, cdkPKT_public_key *src );
int cdk_copy_sk( cdkPKT_secret_key **dst, cdkPKT_secret_key *src );

/*-- sig-check.c --*/
int signature_check( cdkPKT_public_key *pk, cdkPKT_signature *sig,
                     GCRY_MD_HD digest, int *r_expired );

/*-- kbnode.c --*/
CDK_KBNODE cdk_kbnode_clone( CDK_KBNODE node );
void cdk_kbnode_delete( CDK_KBNODE node );
void cdk_kbnode_add( CDK_KBNODE root, CDK_KBNODE node );
void cdk_kbnode_insert( CDK_KBNODE root, CDK_KBNODE node, int pkttype );
CDK_KBNODE cdk_kbnode_find_prev( CDK_KBNODE root, CDK_KBNODE node,
                                 int pkttype );
CDK_KBNODE cdk_kbnode_find_next( CDK_KBNODE node, int pkttype );
CDK_KBNODE cdk_kbnode_find( CDK_KBNODE node, int pkttype );
CDK_KBNODE cdk_kbnode_walk( CDK_KBNODE root, CDK_KBNODE *context, int all );
void cdk_kbnode_clear_flags( CDK_KBNODE n );
int cdk_kbnode_commit( CDK_KBNODE *root );
void cdk_kbnode_remove( CDK_KBNODE *root, CDK_KBNODE node );
void cdk_kbnode_move( CDK_KBNODE *root, CDK_KBNODE node, CDK_KBNODE where );
void cdk_kbnode_dump( CDK_KBNODE node );

#define is_deleted_kbnode(a)  ((a)->private_flag & 1)
#define is_cloned_kbnode(a)   ((a)->private_flag & 2)

/*-- sesskey.c --*/
int _cdk_encode_sesskey( GCRY_MPI *esk, CDK_DEK *dek, int nbits );
int _cdk_encode_digest( byte **r_md, size_t *r_mdlen, int pk_algo,
                       const byte *md, int digest_algo, unsigned nbits );
int _cdk_dek_create( CDK_DEK *dek );

/*-- seckey.c --*/
int cdk_hash_passphrase( CDK_DEK *dek, char *pw, CDK_STRING2KEY *s2k,
                         int create );

/*-- strgutil.c --*/
CDK_STRLIST cdk_strlist_prev( CDK_STRLIST head, CDK_STRLIST node );
CDK_STRLIST cdk_strlist_last( CDK_STRLIST node );

/*-- keydb.c --*/
CDK_KBNODE _cdk_keydb_get_pkblock( u32 *keyid );
int _cdk_keydb_seckey_available( u32 *keyid );

/*-- pksign.c --*/
int cdk_sig_create( cdkPKT_public_key *pk, cdkPKT_signature *sig );
int _cdk_sig_hash_for( int pubkey_algo, int pkt_version );

/*-- encode-packet.c --*/
int _cdk_encode_plaintext( CDK_IOBUF inp, const char *name, int textmode );
int _cdk_encode_symkey_enc( CDK_IOBUF inp, char *pw );

#endif /* OPENCDK_MAIN_H */



