/* new-packet.c - General packet handling (freeing, copying, ...)
 *  Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#include <string.h>
#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "iobuf.h"

int
cdk_pkt_new( CDK_PACKET **r_pkt )
{
    CDK_PACKET *pkt;
    
    pkt = cdk_calloc( 1, sizeof *pkt );
    if ( !pkt )
        return CDKERR_OUT_OF_CORE;
    pkt->pkttype = 0;
    *r_pkt = pkt;
  
    return 0;
} /* cdk_pkt_new */

void
cdk_pkt_free( CDK_PACKET *pkt )
{
    if ( !pkt )
        return;

    switch ( pkt->pkttype ) {
    case PKT_ATTRIBUTE:
    case PKT_USER_ID:
        cdk_user_id_release( pkt->pkt.user_id );
        break;
          
    case PKT_PUBLIC_KEY:
    case PKT_PUBLIC_SUBKEY:
        cdk_pk_release( pkt->pkt.public_key );
        break;
          
    case PKT_SECRET_KEY: 
    case PKT_SECRET_SUBKEY:
        cdk_sk_release( pkt->pkt.secret_key );
        break;
          
    case PKT_SIGNATURE:
        cdk_sig_release( pkt->pkt.signature );
        break;
          
    case PKT_PUBKEY_ENC: {
        cdkPKT_pubkey_enc *enc = pkt->pkt.pubkey_enc;
        int nenc = cdk_pk_get_nenc( enc->pubkey_algo );
        while ( enc->mpi && nenc-- ) {
            cdk_free( enc->mpi[nenc] );
            enc->mpi[nenc] = NULL;
        }
        cdk_free( pkt->pkt.pubkey_enc );
        break; }

    case PKT_SYMKEY_ENC:
        cdk_free( pkt->pkt.symkey_enc );
        break;
          
    case PKT_MDC:
        cdk_free( pkt->pkt.mdc );
        break;
          
    case PKT_ENCRYPTED:
    case PKT_ENCRYPTED_MDC:
        if ( pkt->pkt.encrypted->buf ) {
            cdk_iobuf_close( pkt->pkt.encrypted->buf );
            pkt->pkt.encrypted->buf = NULL;
        }
        cdk_free( pkt->pkt.encrypted );
        break;

    case PKT_ONEPASS_SIG:
        cdk_free( pkt->pkt.onepass_sig );
        break;

    case PKT_PLAINTEXT:
        if ( pkt->pkt.plaintext->name )
            cdk_free( pkt->pkt.plaintext->name );
        if ( pkt->pkt.plaintext->buf )
            cdk_iobuf_close( pkt->pkt.plaintext->buf );
        cdk_free( pkt->pkt.plaintext );
        break;

    case PKT_COMMENT:
        cdk_free( pkt->pkt.comment );
        break;

    case PKT_COMPRESSED:
        cdk_free( pkt->pkt.compressed );
        break;

    default:
        break;
    }
} /* cdk_pkt_free */

void
cdk_pkt_release( CDK_PACKET *pkt )
{
    if ( pkt ) {
        cdk_pkt_free( pkt );
        cdk_free( pkt );
    }
} /* cdk_pkt_release */

int
cdk_pkt_alloc( CDK_PACKET **r_pkt, int id )
{
    CDK_PACKET *pkt;
    int rc = 0;

    rc = cdk_pkt_new( &pkt );
    if ( rc == CDKERR_OUT_OF_CORE )
        return rc;
  
    switch (id) {
    case PKT_USER_ID: 
        pkt->pkt.user_id = cdk_alloc_clear( sizeof *pkt->pkt.user_id );
        if ( !pkt->pkt.user_id )
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_PUBLIC_KEY: 
    case PKT_PUBLIC_SUBKEY:
        pkt->pkt.public_key = cdk_alloc_clear( sizeof *pkt->pkt.public_key );
        if ( !pkt->pkt.public_key )
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_SECRET_KEY: 
    case PKT_SECRET_SUBKEY:
        pkt->pkt.secret_key = cdk_alloc_clear(sizeof *pkt->pkt.secret_key); 
        pkt->pkt.secret_key->pk = cdk_alloc_clear(
            sizeof *pkt->pkt.secret_key->pk);
        if ( !pkt->pkt.secret_key || !pkt->pkt.secret_key->pk )
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_SIGNATURE: 
        pkt->pkt.signature = cdk_sig_new();
        if ( !pkt->pkt.signature )
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_PUBKEY_ENC: 
        pkt->pkt.pubkey_enc = cdk_alloc_clear(sizeof *pkt->pkt.pubkey_enc);
        if (!pkt->pkt.pubkey_enc)
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_MDC:
        pkt->pkt.mdc = cdk_alloc_clear(sizeof *pkt->pkt.mdc);
        if (!pkt->pkt.mdc)
            return CDKERR_OUT_OF_CORE;
        break;
      
    case PKT_ENCRYPTED_MDC:
    case PKT_ENCRYPTED:
        pkt->pkt.symkey_enc = cdk_alloc_clear(sizeof *pkt->pkt.symkey_enc);
        if (!pkt->pkt.symkey_enc)
            return CDKERR_OUT_OF_CORE;
        break;

    case PKT_PLAINTEXT:
        pkt->pkt.plaintext = cdk_alloc_clear( sizeof *pkt->pkt.plaintext );
        if ( !pkt->pkt.plaintext )
            return CDKERR_OUT_OF_CORE;
        break;
    }
    pkt->pkttype = id;
    *r_pkt = pkt;
            
    return 0;
} /* cdk_pkt_alloc */

cdkPKT_signature*
cdk_sig_new( void )
{
    cdkPKT_signature *sig = NULL;

    sig = cdk_alloc_clear( sizeof *sig );
    if ( !sig )
        return NULL;
    cdk_subpkt_new( &sig->hashed );
    cdk_subpkt_new( &sig->unhashed );
    if ( !sig->hashed || !sig->unhashed ) {
        cdk_free( sig ); sig = NULL;
    }
  
    return sig;
} /* cdk_sig_new */

void
cdk_sig_release( cdkPKT_signature *sig )
{
    int nsig = cdk_pk_get_nsig( sig->pubkey_algo );
    
    if ( sig ) {
        while ( sig->mpi && nsig-- ) {
            cdk_free( sig->mpi[nsig] );
            sig->mpi[nsig] = NULL;
        }
        cdk_free( sig->hashed ); sig->hashed = NULL;  
        cdk_free( sig->unhashed ); sig->unhashed = NULL;
        cdk_free( sig );
    }
} /* cdk_pkt_release */

/**
 * cdk_sk_release - Release a secret key.
 * @sk: the secret key.
 *
 * Special routine because the secret key contains sentensive
 * data stored in secure memory.
 **/
void
cdk_sk_release( cdkPKT_secret_key *sk )
{
    int nskey = cdk_pk_get_nskey( sk->pubkey_algo );
  
    if ( !sk )
        return;

    while ( sk->mpi && nskey-- ) {
        cdk_free( sk->mpi[nskey] );
        sk->mpi[nskey] = NULL; 
    }
    if ( sk->encdata ) { 
        cdk_free( sk->encdata );
        sk->encdata = NULL;
    }
    if ( sk->pk ) { 
        cdk_pk_release( sk->pk );
        sk->pk = NULL;
    }
    cdk_free( sk );
} /* cdk_sk_release */

void
cdk_pk_release( cdkPKT_public_key *pk )
{
    int npkey = cdk_pk_get_npkey( pk->pubkey_algo );
    
    if ( !pk )
        return;
  
    if ( pk->uid ) {
        cdk_user_id_release( pk->uid );
        pk->uid = NULL;
    }
    if ( pk->prefs ) {
        cdk_free( pk->prefs );
        pk->prefs = NULL;
    }
    while ( pk->mpi && npkey-- ) {
        cdk_free( pk->mpi[npkey] );
        pk->mpi[npkey] = NULL;
    }
    cdk_free( pk );
} /* cdk_pk_release */

void
cdk_user_id_release( cdkPKT_user_id *uid )
{
    if ( !uid )
        return;

    if ( uid->prefs ) { 
        cdk_free( uid->prefs );
        uid->prefs = NULL;
    }
    if ( uid->attrib_img ) {
        cdk_free( uid->attrib_img );
        uid->attrib_img = NULL;
    }
    cdk_free( uid );
} /* cdk_pkt_uid_release */

cdk_prefitem_t*
_cdk_prefs_copy( const cdk_prefitem_t *prefs )
{
    size_t n = 0;
    cdk_prefitem_t *new_prefs;
  
    if (!prefs)
        return NULL;

    for ( n = 0; prefs[n].type; n++ )
        ;
    new_prefs = cdk_alloc( sizeof *new_prefs * (n+1) );
    if ( !new_prefs )
        return NULL;
    for ( n = 0; prefs[n].type; n++ ) {
        new_prefs[n].type = prefs[n].type;
        new_prefs[n].value = prefs[n].value;
    }
    new_prefs[n].type = PREFTYPE_NONE;
    new_prefs[n].value = 0;
  
    return new_prefs;
} /* _cdk_prefs_copy */

int
cdk_copy_uid( cdkPKT_user_id **dst, cdkPKT_user_id *src )
{
    cdkPKT_user_id *u;
  
    if ( !dst || !src )
        return CDKERR_INV_VALUE;
  
    u = cdk_alloc_clear( sizeof *u + strlen( src->name ) + 1 );
    if ( !u )
        return CDKERR_OUT_OF_CORE;
    memcpy( u, src, sizeof *u );
    memcpy( u->name, src->name, strlen( src->name ) );
    u->prefs = _cdk_prefs_copy( src->prefs );
    *dst = u;

    return 0;
} /* cdk_copy_uid */

int
cdk_copy_pk( cdkPKT_public_key **dst, cdkPKT_public_key *src )
{
    cdkPKT_public_key *k;
    int i;
  
    if ( !dst || !src )
        return CDKERR_INV_VALUE;

    k = cdk_alloc_clear( sizeof *k );
    if ( !k )
        return CDKERR_OUT_OF_CORE;
    memcpy( k, src, sizeof *k );
    if ( src->uid )
        cdk_copy_uid( &k->uid, src->uid );
    if ( src->prefs )
        k->prefs = _cdk_prefs_copy( src->prefs );
    for ( i = 0; i < cdk_pk_get_npkey( src->pubkey_algo ); i++ ) {
        k->mpi[i] = cdk_alloc_clear( sizeof **k->mpi + src->mpi[i]->bytes + 2);
        if ( !k->mpi[i] )
            return CDKERR_OUT_OF_CORE;
        k->mpi[i]->bits = src->mpi[i]->bits;
        k->mpi[i]->bytes = src->mpi[i]->bytes;
        /* copy 2 extra bytes (prefix) */
        memcpy( k->mpi[i]->data, src->mpi[i]->data, src->mpi[i]->bytes + 2 );
    }
    *dst = k;
  
    return 0;
} /* cdk_copy_pk */

int
cdk_copy_sk( cdkPKT_secret_key **dst, cdkPKT_secret_key *src )
{
    cdkPKT_secret_key *k;
    int i;
  
    if ( !dst || !src )
        return CDKERR_INV_VALUE;

    k = cdk_alloc_clear( sizeof *k );
    if ( !k )
        return CDKERR_OUT_OF_CORE;
    memcpy( k, src, sizeof *k );
    cdk_copy_pk( &k->pk, src->pk );
  
    for ( i = 0; i < cdk_pk_get_nskey( src->pubkey_algo ); i++ ) {
        k->mpi[i] = cdk_alloc_clear( sizeof **k->mpi + src->mpi[i]->bytes + 2);
        if ( !k->mpi[i] )
            return CDKERR_OUT_OF_CORE;
        k->mpi[i]->bits = src->mpi[i]->bits;
        k->mpi[i]->bytes = src->mpi[i]->bytes;
        /* copy 2 extra bytes (prefix) */
        memcpy( k->mpi[i]->data, src->mpi[i]->data, src->mpi[i]->bytes + 2 );
    }
    *dst = k;

    return 0;
} /* cdk_copy_sk */

int
copy_pk_parts_to_sk( cdkPKT_public_key *pk, cdkPKT_secret_key *sk )
{
    if ( !pk || !sk )
        return CDKERR_INV_VALUE;

    sk->version = pk->version;
    sk->expiredate  = pk->expiredate;
    sk->pubkey_algo = pk->pubkey_algo;
    sk->has_expired = pk->has_expired;
    sk->is_revoked = pk->is_revoked;
    sk->main_keyid[0]= pk->main_keyid[0];
    sk->main_keyid[1]= pk->main_keyid[1];
    sk->keyid[0] = pk->keyid[0];
    sk->keyid[1] = pk->keyid[1];
  
    return 0;
} /* copy_pk_parts_to_sk */

int
cdk_pkt_sig_copy( cdkPKT_signature **dst, cdkPKT_signature *src )
{
    cdkPKT_signature *s = NULL;
    struct cdk_subpkt_s *res = NULL;
  
    if (!dst || !src)
        return CDKERR_INV_VALUE;

    s = cdk_alloc_clear( sizeof *s );
    if ( !s )
        return CDKERR_OUT_OF_CORE;
    memcpy( s, src, sizeof *src );
  
    subpkt_copy( &res, src->hashed );
    subpkt_copy( &s->hashed, res );
    cdk_subpkt_release( res ); res = NULL;
    subpkt_copy( &res, src->unhashed );
    subpkt_copy( &s->unhashed, res );
    cdk_subpkt_release( res ); res = NULL;
    *dst = s;

    return 0;
} /* cdk_pkt_sig_copy */

int
_cdk_compare_pks( cdkPKT_public_key *a, cdkPKT_public_key *b )
{
    int n, i;
  
    if( a->timestamp != b->timestamp )
        return -1;
    if( a->version < 4 && a->expiredate != b->expiredate )
        return -1;
    if( a->pubkey_algo != b->pubkey_algo )
        return -1;
    n = cdk_pk_get_npkey( a->pubkey_algo );
    if ( !n )
        return -1;
    for ( i = 0; i < n; i++ ) {
        if ( memcmp( a->mpi[i]->data, b->mpi[i]->data, a->mpi[i]->bytes) )
            return -1; 
    }

    return 0;
} /* _cdk_compare_pks */

int
cdk_subpkt_new( CDK_SUBPKT *ctx )
{
    CDK_SUBPKT c = NULL;

    c = cdk_calloc( 1, sizeof *c );
    if ( !c )
        return CDKERR_OUT_OF_CORE;
    *ctx = c;
  
    return 0;
} /* cdk_subpkt_new */

void
cdk_subpkt_release( CDK_SUBPKT ctx )
{
    struct cdk_subpkt_s *p2 = NULL;

    while ( ctx ) {
        p2 = ctx->next;
        cdk_free( ctx->data );
        cdk_free( ctx );
        ctx = p2;
    }
} /* cdk_subpkt_release */

struct cdk_subpkt_s*
cdk_subpkt_add( CDK_SUBPKT r_ctx )
{
    struct cdk_subpkt_s *c = NULL;

    c = cdk_calloc( 1, sizeof *c );
    if ( !c )
        return NULL;
    c->next = r_ctx;
  
    return c;
} /* cdk_subpkt_add */

struct cdk_subpkt_s*
cdk_subpkt_find( CDK_SUBPKT ctx, int type )
{
    struct cdk_subpkt_s *s = NULL;
    
    for ( s = ctx; s && s->size; s = s->next ) {
        if ( s->type == type )
            return s;
    }
    
    return NULL;
} /* cdk_subpkt_find */

int
cdk_subpkt_init( CDK_SUBPKT ctx, int type, byte *data, u32 size )
{
    if ( !ctx || !data || !size )
        return CDKERR_GENERAL;
        
    ctx->data = cdk_alloc( size+1 );
    if ( !ctx->data )
        return CDKERR_OUT_OF_CORE;
    memcpy( ctx->data, data, size );
    ctx->size = size;
    ctx->type = type;
  
    return 0;
} /* cdk_subpkt_init */

int
subpkt_calc_size( struct cdk_subpkt_s *p )
{
    size_t size = 0;

    if ( !p )
        return 0;
  
    if ( p->size < 192 )
        size++; /* 1 byte header */
    /* TODO: add all headers */
    size++; /* type */
    size += p->size;

    return size;
} /* subpkt_calc_size */

int
subpkt_copy( struct cdk_subpkt_s **r_dst, struct cdk_subpkt_s *src )
{
    struct cdk_subpkt_s *s = NULL;
    struct cdk_subpkt_s *p = NULL;
  
    if ( !src || !r_dst )
        return CDKERR_INV_VALUE;
  
    for ( p = src; p && p->size; p = p->next ) {
        s = cdk_subpkt_add( s );
        cdk_subpkt_init( s, p->type, p->data, p->size );
    }
    *r_dst = s;
  
    return 0;
} /* subpkt_copy */

int
cdk_subpkt_calc_size( CDK_SUBPKT pkt )
{
    struct cdk_subpkt_s *p = NULL;
    size_t size = 0;
        
    for ( p = pkt; p && p->size; p = p->next )
        size += subpkt_calc_size( p );
  
    return size;
} /* cdk_subpkt_calc_size */

int
cdk_subpkt_hash( CDK_SUBPKT hashed, size_t hashed_size, GCRY_MD_HD hd )
{
    struct cdk_subpkt_s *aback_list = NULL, *s = NULL;
    u32 datalen = 0;
    byte buf[2];
  
    if ( !hashed || !hd )
        return CDKERR_INV_VALUE;

    buf[0] = hashed_size >> 8;
    buf[1] = hashed_size;
    gcry_md_write( hd, buf, 2 );
    subpkt_copy( &aback_list, hashed );
    for ( s = aback_list; s && s->size; s = s->next ) {
        datalen = s->size + 1; /* type */
        if ( s->size < 192 )
            gcry_md_putc( hd, datalen );
        gcry_md_putc( hd, s->type );
        gcry_md_write( hd, s->data, s->size );
    }
    cdk_subpkt_release( aback_list );

    return 0;
} /* cdk_subpkt_hash */
