/* pkverify.c - Verify signatures
 *  Copyright (C) 2001, 2002 Timo Schulz 
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *  
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 *  
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */ 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <string.h>
#include <gcrypt.h>

#include "opencdk.h"
#include "main.h"
#include "iobuf.h"

typedef struct {
    const char *md_name;
    int digest_algo;
} clearsign_md;

clearsign_md clearsig_ctx[] = {
    {"MD5", GCRY_MD_MD5},
    {"SHA1", GCRY_MD_SHA1},
    {"RIPEMD160", GCRY_MD_RMD160},
    {"MD2", 0}, /* no support for MD2 */
    {"TIGER192", GCRY_MD_TIGER},
    {NULL, 0}
};

int
cdk_sig_from_iobuf( CDK_IOBUF buf,
                    cdkPKT_signature **r_sig,
                    cdkPKT_public_key **r_key )
{
    CDK_PACKET pkt = {0};
    cdkPKT_signature *sig = NULL;  
    int rc = 0;

    if (!r_sig || !r_key || !buf)
        return CDKERR_INV_VALUE;
  
    rc = cdk_pkt_parse( buf, &pkt );
    if (rc)
        return rc;
    if ( pkt.pkttype != PKT_SIGNATURE ) {
        rc = CDKERR_INV_PACKET;
        goto leave;
    }
    sig = pkt.pkt.signature;
    rc = cdk_keydb_get_pk( NULL, sig->keyid, r_key );
    if ( rc )
        goto leave;
    *r_sig = sig;
  
leave:
    return rc;
} /* cdk_sig_from_iobuf */

/**
 * cdk_verify_file - Verify a signature.
 * @file: The plaintext file.
 * @sig_file: The signature file.
 * @r_pk: The context to store the used PK.
 *
 * Verify a detached signature from @sig_file. The data must be in @file.
 * When the key, the signature was made with, was detected it'll be
 * returned in @r_pk.
 **/
int
cdk_verify_file( const char *file, const char *sig_file, 
                 cdkPKT_public_key **r_pk )
{
    CDK_IOBUF buf;
    cdkPKT_signature *sig = NULL;  
    cdkPKT_public_key *pk = NULL;  
    md_filter_s ctx = {0};
    int is_expired = 0;
    int rc = 0;

    if ( !file || !sig_file || !r_pk )
        return CDKERR_INV_VALUE;
  
    rc = cdk_iobuf_open( &buf, sig_file, IOBUF_MODE_RD );
    if ( rc == -1 )
        return CDKERR_FILE_OPEN;
    rc = cdk_sig_from_iobuf( buf, &sig, &pk );
    if ( rc )
        goto leave;
    cdk_iobuf_close( buf );
    ctx.md = gcry_md_open( sig->digest_algo, 0 );
    if ( !ctx.md ) {
        rc = CDKERR_GCRY;
        goto leave;
    }
    ctx.digest_algo = sig->digest_algo;
    rc = cdk_iobuf_open( &buf, file, IOBUF_MODE_RD );
    if ( rc )
        goto leave;
    rc = cdk_md_filter( &ctx, IOBUF_CTRL_FLUSH, buf );
    if ( rc ) {
        cdk_iobuf_close(buf);
        rc = CDKERR_FILTER;
        goto leave;
    }
    cdk_iobuf_close( buf );
    rc = signature_check( pk, sig, ctx.md, &is_expired );
    *r_pk = !rc? pk : NULL;
	
leave:
    if ( rc )
        cdk_pk_release( pk );
    cdk_sig_release( sig );
    cdk_md_close( &ctx.md );
  
    return rc;
} /* cdk_verify_file */

/**
 * cdk_verify_cleartext:
 *
 * Verify a clearsigned signature. The signature must be in @file.
 **/
int
cdk_verify_cleartext(const char *file, cdkPKT_public_key **r_pk)
{
#if 0
    CDK_IOBUF buf, a;
    md_filter_s ctx = {0};
    cdkPKT_signature *sig = NULL;
    cdkPKT_public_key *pk = NULL;
    char buffer[512], t[512];
    const char *s;
    int rc = 0;
    int i, is_expired = 0, is_signed = 0;
    int digest_algo = GCRY_MD_MD5;
    size_t n = 0;

    /* FIXME: use new filter code!! */
  
    rc = cdk_iobuf_open(&buf, file, IOBUF_MODE_RD);
    if (rc == -1)
        return CDKERR_FILE_OPEN;

    while (!cdk_iobuf_eof(buf)) {
        rc = cdk_iobuf_read_line(buf, buffer, sizeof(buffer)-1, &n);
        if ( rc == -1 ) {
            rc = CDKERR_FILE_READ;
            break;
        }
        if (!strncmp(buffer, "-----BEGIN PGP SIGNED MESSAGE-----", 35)) {
            is_signed = 1;
            break;
        }
    }
    
    if ( cdk_iobuf_eof(buf) || !is_signed ) {
        rc = CDKERR_ARMOR;
        goto leave;
    }
    
    while ( !cdk_iobuf_eof(buf) ) {
        rc = cdk_iobuf_read_line(buf, buffer, sizeof(buffer)-1, &n);
        if (rc == -1) {
            rc = CDKERR_FILE_READ;
            break;
        }        
        if (n == 1)
            break; /* the empty line */
        else if (!strncmp(buffer, "Hash: ", 6)) {
            for (i=0; (s=clearsig_ctx[i].md_name); i++)
            {
                if (!strcmp(buffer+6, s))
                    digest_algo = clearsig_ctx[i].digest_algo;
            }
        }            
    }
    
    ctx.md = gcry_md_open(digest_algo, 0);
    if (!ctx.md) {
        rc = CDKERR_GCRY;
        goto leave;
    }
    ctx.algo = digest_algo;
    rc = cdk_md_filter( &ctx, IOBUF_CTRL_FLUSH, buf );
    if (rc) {
        rc = CDKERR_FILTER; 
        goto leave;
    }
    while (!cdk_iobuf_eof(buf)) {
        rc = cdk_iobuf_read_line(buf, buffer, sizeof(buffer)-1, &n);
        if (rc == -1) {
            rc = CDKERR_FILE_READ;
            break;
        }
        if (!strncmp(buffer, "-----BEGIN PGP SIGNATURE-----", 28))
            break;
    }
  
    cdk_iobuf_ctrl_filter(buf, IOBUF_CTRL_FILTER, 0, cdk_md_filter);
    a = cdk_iobuf_temp();
    while ( !cdk_iobuf_eof(buf) ) {
        rc = cdk_iobuf_read_line(buf, buffer, sizeof(buffer)-1, &n);
        if (rc == -1) {
            rc = CDKERR_FILE_READ;
            goto leave;
        }
        if (*buffer == '=' && n == 6) {
            cdk_iobuf_read_line(buf, buffer, sizeof(buffer)-1, &n);
            if (!n || strncmp(buffer, "-----END PGP SIGNATURE-----", 26))
                rc = CDKERR_ARMOR;
            break;
        }
        n = base64_to_raw(t, buffer);
        cdk_iobuf_write(a, t, n);
    }
    
    cdk_iobuf_rewind(a);
    sig = cdk_sig_new();
    rc = cdk_pkt_read_signature(a, sig);
    if (rc)
        goto leave;

    rc = cdk_keydb_get_pk(NULL, &pk, sig->keyid);
    if (rc || !pk) {
        rc = CDKERR_NOKEY;
        goto leave; 
    }
        
    cdk_iobuf_close(a);
    rc = signature_check(pk, sig, ctx.md, &is_expired);
    if (!rc)
        *r_pk = pk;
    
leave:    
    if (rc)
        cdk_pk_release(pk);
    cdk_md_close( &ctx.md );
    cdk_pkt_release(sig);
    cdk_iobuf_close(buf);
    return rc;
#endif
    return 0;
} /* cdk_verify_cleartext */    




