/* sign.c - Signing routines
 *  Copyright (C) 2002 Timo Schulz
 *  Copyright (C) 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
#include <time.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "iobuf.h"

#if 0
static void
md_print_hash( GCRY_MD_HD hd )
{
    int i;
    byte md[24];

    if (!hd)
        return;
    memcpy( md, gcry_md_read( hd, GCRY_MD_SHA1 ),
            gcry_md_get_algo_dlen( GCRY_MD_SHA1 ) );
    for ( i=0; i<20; i++ )
        printf( "%02X", md[i] );
    printf( "\n" );
} /* md_print_hash */
#endif

int
_cdk_sig_hash_for( int pubkey_algo, int pkt_version )
{
    /* fixme: we should also use the preferences */
    if ( is_DSA(pubkey_algo) )
        return GCRY_MD_SHA1;
    else if ( is_RSA(pubkey_algo) && pkt_version < 4 )
        return GCRY_MD_MD5;

    return OPENPGP_DEF_MD; /* default message digest */
} /* _cdk_sig_hash_for */

/**
 * cdk_sig_create - Create the signature.
 * @pk: the public key.
 * @sig: the signature.
 *
 * Create a signature from the settings of the public key if
 * the key is != NULL. And additional all sub packets will be
 * initalized.
 **/
int
cdk_sig_create( cdkPKT_public_key *pk, cdkPKT_signature *sig )
{
    byte buf[8];
    struct cdk_subpkt_s *s = NULL;
  
    if ( !sig )
        return CDKERR_INV_VALUE;
  
    if ( pk ) {    
        sig->version = pk->version;
        sig->pubkey_algo = pk->pubkey_algo;
        sig->digest_algo = _cdk_sig_hash_for( pk->pubkey_algo, pk->version );
        cdk_pk_get_keyid( pk, sig->keyid );
    }
    sig->timestamp = make_timestamp();
  
    if ( !sig->hashed || !sig->unhashed )
        return CDKERR_INV_VALUE;

    u32_to_buffer( sig->keyid[0], buf );
    u32_to_buffer( sig->keyid[1], buf+4 );
    s = sig->unhashed = cdk_subpkt_add( sig->unhashed );
    cdk_subpkt_init( s, SIGSUBPKT_ISSUER, buf, 8 );
    sig->unhashed_size += subpkt_calc_size( s );

    u32_to_buffer( sig->timestamp, buf );
    s = sig->hashed = cdk_subpkt_add( sig->hashed );
    cdk_subpkt_init( s, SIGSUBPKT_SIG_CREATED, buf, 4 );
    sig->hashed_size += subpkt_calc_size( s );

    if ( sig->expiredate ) {
        u32 u = sig->expiredate-sig->timestamp;
        u32_to_buffer( u, buf );
        s = sig->hashed = cdk_subpkt_add( sig->hashed );
        cdk_subpkt_init( s, SIGSUBPKT_SIG_EXPIRE, buf, 4 );
        sig->hashed_size += subpkt_calc_size( s );
    }
  
    return 0;
} /* cdk_sig_create */

int
cdk_sign_file( GCRY_SEXP opts, CDK_STRLIST locusr, const char *file,
               const char *output, int detached )
{
    CDK_IOBUF inp = NULL, outp = NULL;
    md_filter_s mfx;
    CDK_KEYLIST skl;
    int rc = 0;
  
    if ( !file || !output )
        return CDKERR_INV_VALUE;
    if ( detached != 1 )
        return CDKERR_NOT_IMPLEMENT;

    rc = cdk_iobuf_open( &inp, file, IOBUF_MODE_RD );
    if ( rc )
        goto leave;

    rc = cdk_iobuf_create( &outp, output );
    if ( rc )
        goto leave;
    
    rc = cdk_sklist_build( locusr, &skl, 1, GCRY_PK_USAGE_SIGN );
    if ( rc )
        return rc;

    /* fixme: this procedure only works when all secret keys
       use the same hash algorithm (for the case we've more then one). */
  
    memset( &mfx, 0, sizeof mfx );
    mfx.digest_algo = _cdk_sig_hash_for( skl->key.sk->pubkey_algo,
                                         skl->key.sk->version );
    mfx.md = gcry_md_open( mfx.digest_algo, 0 );
    if ( !mfx.md ) {
        rc = CDKERR_GCRY;
        goto leave;
    }

    rc = cdk_md_filter( &mfx, IOBUF_CTRL_FLUSH, inp );
    if ( rc )
        goto leave;
    cdk_iobuf_close( inp ); inp = NULL;
  
    rc = cdk_sklist_write( skl, outp, mfx.md, 0x00 );
  
leave:
    cdk_iobuf_close( inp );
    cdk_iobuf_close( outp );
  
    return rc;
} /* cdk_sign_file */








