/* t-packet.c
 *  Copyright (C) 2001, 2002 Timo Schulz
 * 
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version. 
 *
 * OpenCDK is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 * 
 * You should have received a copy of the GNU General Public License 
 * along with OpenCDK; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <stdio.h> 
#include <gcrypt.h> 

#include "opencdk.h" 
#include "main.h" 

void 
dump_mpi_stats(CDK_MPI **m, int algo, int pk_algo, int version) 
{ 
    int i; 
    
    if ( algo != GCRY_CIPHER_NONE ) 
        fprintf(stderr, "encrypted stuff follows...algo %d\n", algo); 
    else { 
        for (i=0; i<cdk_pk_get_nskey(pk_algo); i++) 
            printf("mpi[%d] %d bits\n", i, m[i]->bits); 
    } 
}

void 
do_parse_pkts(const char *fname) 
{ 
    CDK_IOBUF buf; 
    CDK_PACKET *pkt; 
    int i; 
    int rc = 0; 
     
    cdk_secmem_init( 16384 ); 
    cdk_set_logging( OPENCDK_LOG_DEBUG ); 
    rc = cdk_iobuf_open(&buf, fname, IOBUF_MODE_RD); 
    if ( rc ) { 
        printf("Can't open `%s'\n", fname); 
        return; 
    } 
		
    for (;;) { 
        /*printf("rfc2440: %d type: %d (length: %d)\n",  
               pkt.rfc2440, pkt.type, pkt.len); */
        pkt = cdk_alloc_clear( sizeof *pkt ); 
        rc = cdk_pkt_parse(buf, pkt); 
        printf("pkttype `%d' `%d' \n", 
               pkt?pkt->pkttype:0, 
               pkt?pkt->pktlen:0); 
        if (rc) 
            break; 
        if (pkt->pkttype == PKT_USER_ID) { 
            cdkPKT_user_id *id = pkt->pkt.user_id;
            if ( id->attrib_img )
                printf("ATTR: [%d] %s\n", id->attrib_len, id->name );
            else
                printf( "UID: [r%d] %s\n", id->is_revoked, id->name ); 
        } 
        else if (pkt->pkttype == PKT_PUBLIC_KEY  
                 || pkt->pkttype == PKT_PUBLIC_SUBKEY ) { 
            u32 keyid[2]; 
            cdk_pk_get_keyid( pkt->pkt.public_key, keyid ); 
            printf("%s: %u/0x%08X%08X\n", 
                   pkt->pkttype == PKT_PUBLIC_KEY ? "pubkey" : "subkey", 
                   pkt->pkt.public_key->mpi[0]->bits, 
                   keyid[0], keyid[1]); 
        } 
        else if (pkt->pkttype == PKT_SIGNATURE) { 
            struct cdk_subpkt_s *p; 
            printf("hashed subpkt size %d\n", 
                   cdk_subpkt_calc_size(pkt->pkt.signature->hashed)); 
            { 
                u32 keyid[2]; 
                cdk_sig_get_keyid( pkt->pkt.signature, keyid ); 
                printf("sig: 0x%08X%08X\n", keyid[0], keyid[1]); 
            } 

            if ( pkt->pkt.signature->version > 3 ) { 
                printf("sig: type %X\n", pkt->pkt.signature->sig_class); 
                  
                /*for (p=pkt->pkt.signature->hashed; p && p->size; p=p->next) 
                    printf("hashed: type %u size %d\n",  
                           p->type, p->size); */
                /*for (p=pkt->pkt.signature->hunhashed; p && p->size;
                  p=p->next) 
                  printf("unhashed: type %d size %d\n", p->type, p->size); */
                    
            } 
        } 
        else if (pkt->pkttype == PKT_SECRET_KEY) { 
            u32 keyid[2]; 
            cdk_sk_get_keyid( pkt->pkt.secret_key, keyid ); 
            printf("seckey: 0x%08X%08X\n", keyid[0], keyid[1]); 
            dump_mpi_stats(pkt->pkt.secret_key->mpi, 
                           pkt->pkt.secret_key->protect.algo, 
                           pkt->pkt.secret_key->pk->pubkey_algo, 
                           pkt->pkt.secret_key->pk->version); 
        } 
        else if ( pkt->pkttype == PKT_SYMKEY_ENC ) { 
            printf("sym enc: symalg %02d; s2k-mode %02d; s2k-algo %02d\n", 
                   pkt->pkt.symkey_enc->cipher_algo, 
                   pkt->pkt.symkey_enc->s2k.mode, 
                   pkt->pkt.symkey_enc->s2k.hash_algo); 
        } 
        else if ( pkt->pkttype == PKT_PUBKEY_ENC ) { 
            printf("pubkey enc: pubalg %02d; key-id %08x\n", 
                   pkt->pkt.pubkey_enc->pubkey_algo, 
                   pkt->pkt.pubkey_enc->keyid[1]); 
        } 
        else if ( pkt->pkttype == PKT_ENCRYPTED 
                  || pkt->pkttype == PKT_ENCRYPTED_MDC ) { 
            printf("encrypted: mdc-method %02d; size %02d\n", 
                   pkt->pkt.encrypted->mdc_method, 
                   pkt->pkt.encrypted->buf? 
                   pkt->pkt.encrypted->len:0);
        } 
        else if ( pkt->pkttype == PKT_COMPRESSED ) { 
            printf("compressed: algo %02d size %02d\n", 
                   pkt->pkt.compressed->algorithm, 
                   pkt->pkt.compressed->buf? 
                   cdk_iobuf_get_length(pkt->pkt.compressed->buf):0); 
        } 
        else if ( pkt->pkttype == PKT_ONEPASS_SIG ) { 
            printf("one pass sig: class %02X md-algo %02d pk-algo %02d\n", 
                   pkt->pkt.onepass_sig->sig_class, 
                   pkt->pkt.onepass_sig->digest_algo, 
                   pkt->pkt.onepass_sig->pubkey_algo ); 
        } 
        else if ( pkt->pkttype == PKT_PLAINTEXT ) { 
            printf("plaintext (literal): mode %c time %08X name %s\n", 
                   pkt->pkt.plaintext->mode, 
                   pkt->pkt.plaintext->timestamp, 
                   pkt->pkt.plaintext->name); 
        } 
 	} 

    if (rc) 
        printf("Error `%s'\n", cdk_strerror(rc) ); 

    cdk_iobuf_close(buf); 
    cdk_secmem_end( );
}

void 
test_packets(const char *fname) 
{ 
    CDK_IOBUF buf; 
    CDK_KBNODE pkt, p; 
    int rc = 0; 
  
    rc = cdk_iobuf_open(&buf, fname, IOBUF_MODE_RD); 
    if ( rc ) { 
        printf("can't open `%s'\n", fname); 
        return; 
    } 
    
    rc = cdk_pkts_parse( buf, &pkt ); 

    #if 0
    printf("** done (rc=%d)\n", rc); 
    for (p=pkt; p; p=p->next) 
        printf("ID %02d\n", p->pkt->pkttype);
    #endif
    
    cdk_kbnode_release(pkt); 
    cdk_iobuf_close(buf); 
} 

void 
test_packet_pk(void) 
{ 
    cdkPKT_public_key *pk; 
    CDK_IOBUF a; 
    CDK_PACKET *pkt; 
    int rc; 
  
    cdk_keydb_add_resource("ts.gpg", 0); 
    rc=cdk_keydb_get_pk_byname( "Timo Schulz <ts@winpt.org>", &pk, 0 ); 
    printf("pk version: %d; bits %d\n", pk->version, pk->mpi[0]->bits ); 
    if (rc) { 
        printf("Can't find key: `%s'\n", cdk_strerror(rc)); 
        return; 
    } 
    else { 
        cdk_pkt_alloc(&pkt, PKT_PUBLIC_KEY); 
        a = cdk_iobuf_temp(); 
        pkt->pkttype = PKT_PUBLIC_KEY; 
        pkt->pkt.public_key = pk; 
        rc = cdk_pkt_build(a, pkt); 
        if (rc) 
            printf("Can't write public key: `%s'\n", cdk_strerror(rc)); 
        /*fprintf(stdout, "iobuf size `%d'\n", cdk_iobuf_get_length(a)); */
        #if 0
        while ( !cdk_iobuf_eof(a) ) { 
            byte buf[512]; 
            size_t n; 
            int rc; 

            rc=cdk_iobuf_read(a, buf, 512, &n); 
            if (rc == -1 || !n) 
                break; 
            /*fprintf(stderr, "(rc %d) n=%d\n", rc, n);*/
            fwrite(buf, 1, n, stderr); 
        }
        #endif
        cdk_pkt_release(pkt);
        printf("** pkt **\n");
        cdk_iobuf_close(a);
    }  
}

int
main(int argc, char **argv)
{
    /*test_encode_packet();
    return 0;*/
    if (argc == 2) { 
        /*test_packets(argv[1]);*/
        do_parse_pkts(argv[1]);
    }
    test_packet_pk();
    return 0;
}
    








