/* encrypt.c
 *       Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"
#include "filters.h"
#include "stream.h"
#include "packet.h"


struct mainproc_ctx_s
{
  CDK_DEK dek;
  CDK_STREAM s;
  CDK_KBNODE node;
  struct 
  {
    unsigned present:1;
    unsigned one_pass:1;
    GCRY_MD_HD md;
    int digest_algo;
    int is_expired;
    cdkPKT_public_key * pk;
    unsigned pt_offset;
  } sig;
  unsigned reset:1;
  unsigned eof_seen:1;
  char * file; /* for detached signatures */
};
typedef struct mainproc_ctx_s *CTX;


static int
sym_encrypt_file (CDK_HD hd, const char *file, const char *output)
{
  CDK_STREAM inp = NULL, out = NULL;
  CDK_S2K_HD *s2k = NULL;
  CDK_PACKET pkt;
  CDK_DEK dek = NULL;
  cdkPKT_symkey_enc *enc;
  char *pw = NULL;
  int rc = 0;

  if (!hd || !file || !output)
    return CDK_Inv_Value;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;
  
  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }
  
  pw = _cdk_passphrase_get ("Enter Passphrase: ");
  if (!pw)
    goto fail;

  s2k = cdk_calloc (1, sizeof *s2k);
  if (!s2k)
    {
      rc = CDK_Out_Of_Core;
      goto fail;
    }
  s2k->mode = hd->s2k.mode;
  s2k->hash_algo = hd->s2k.digest_algo;
  
  dek = cdk_passphrase_to_dek (hd->cipher_algo, s2k, 2, pw);
  if (!dek || !dek->keylen)
    {
      rc = CDK_General_Error;
      goto fail;
    }  

  cdk_stream_set_cache (out, 1);
  enc = cdk_calloc (1, sizeof *enc);
  if (!enc)
    {
      rc = CDK_Out_Of_Core;
      goto fail;
    }
  enc->version = 4;
  enc->cipher_algo = dek->algo;
  enc->s2k = *s2k;
  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_SYMKEY_ENC;
  pkt.pkt.symkey_enc = enc;
  rc = cdk_pkt_build (out, &pkt);
  cdk_free (enc);
  if (rc)
    goto fail;
  cdk_stream_set_cache (out, 0);

  cdk_stream_set_cipher_flag (out, dek, hd->opt.mdc);
  cdk_stream_set_compress_flag (out, hd->compress_algo);
  cdk_stream_set_literal_flag (out, 0, file);

  rc = cdk_stream_kick_off (inp, out);

fail:
  cdk_stream_close (inp);
  cdk_stream_close (out);
  cdk_free (s2k);
  cdk_free (dek);
  _cdk_passphrase_free (pw);
  return rc;
}


int
cdk_file_encrypt (CDK_HD hd, CDK_STRLIST remusr,
                  const char *file, const char *output)
{
  CDK_KEYLIST pkl = NULL;
  CDK_STREAM inp = NULL, out = NULL;
  CDK_DEK dek = NULL;
  int rc = 0;

  if (!file || !output)
    return CDK_Inv_Value;

  if (!remusr)
    return sym_encrypt_file (hd, file, output);

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;

  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = cdk_pklist_build (remusr, &pkl, GCRY_PK_USAGE_ENCR);
  if (rc)
    goto fail;

  rc = cdk_dek_new (&dek, 0);
  if (rc)
    goto fail;
  dek->algo = cdk_pklist_select_algo (pkl);
  dek->use_mdc = cdk_pklist_use_mdc (pkl);

  cdk_stream_set_cache (out, 1);
  rc = cdk_pklist_encrypt (pkl, dek, out);
  if (rc)
    goto fail;
  cdk_stream_set_cache (out, 0);

  cdk_stream_set_cipher_flag (out, dek, 0);
  cdk_stream_set_compress_flag (out, hd->compress_algo);
  cdk_stream_set_literal_flag (out, 0, file);

  rc = cdk_stream_kick_off (inp, out);

fail:
  cdk_stream_close (inp);
  cdk_stream_close (out);
  cdk_free (dek);
  cdk_pklist_release (pkl);
  
  return rc;
}


static void
set_reset (CTX c, CDK_STREAM inp)
{
  if (!c->reset)
    {
      cdk_stream_control (inp, CDK_STREAMCTL_RESET, 1);
      c->reset = 1;
    }
}


static int
hash_data_file (char * file, int digest_algo, GCRY_MD_HD * r_md)
{
  md_filter_s * mfx;
  CDK_STREAM s;
  int rc;

  if (strstr (file, ".asc") || strstr (file, ".gpg")
      || strstr (file, ".pgp") || strstr (file, ".sig"))
    file[strlen (file) - 4] = '\0';
  else
    return CDK_General_Error;

  rc = cdk_stream_open (file, &s);
  if (rc)
    return rc;
  cdk_stream_set_hash_flag (s, digest_algo);
  cdk_stream_read (s, NULL, 0);
  mfx = _cdk_stream_get_opaque (s, fHASH);
  if (mfx && mfx->md)
    {
      *r_md = gcry_md_copy (mfx->md);
      gcry_md_close (mfx->md);
    }
  cdk_stream_close (s);
  return 0;
}


static int
handle_symkey_enc (CTX c, CDK_PACKET * pkt)
{
  char * pw;
  cdkPKT_symkey_enc *key;
  int rc = 0;

  assert (pkt->pkttype == CDK_PKT_SYMKEY_ENC);

  if (c->dek)
    return 0; /* we already decrypted the session key */
  
  pw = _cdk_passphrase_get ("Enter Passphrase: ");
  if (!pw)
      return CDK_Out_Of_Core;

  key = pkt->pkt.symkey_enc;
  c->dek = cdk_passphrase_to_dek (key->cipher_algo, &key->s2k, 0, pw);
  if (!c->dek)
    rc = CDK_Error_No_Key;
  _cdk_passphrase_free (pw);
  return rc;
}


static int
handle_pubkey_enc (CTX c, CDK_PACKET * pkt)
{
  cdkPKT_pubkey_enc * enc;
  CDK_KBNODE snode, sk;
  int rc = 0;

  assert (pkt->pkttype == CDK_PKT_PUBKEY_ENC);
  
  if (c->dek)
    return 0; /* we already decrypted the session key */

  enc = pkt->pkt.pubkey_enc;
  snode = _cdk_keydb_get_skblock (enc->keyid);
  if (!snode)
    return 0;
  sk = _cdk_keydb_find_bykeyid (snode, enc->keyid);
  if (sk)
    {
      c->dek = cdk_pkcs1_to_dek (enc, sk->pkt->pkt.secret_key);
      if (!c->dek)
        rc = CDK_Error_No_Key;
    }
  cdk_kbnode_release (snode);
  return rc;
}


static int
handle_encrypted (CTX c, CDK_PACKET *pkt, int use_mdc)
{
  cdkPKT_encrypted *enc;
  int rc = 0;

  assert (pkt->pkttype == CDK_PKT_ENCRYPTED
          || pkt->pkttype == CDK_PKT_ENCRYPTED_MDC);
  
  enc = pkt->pkt.encrypted;
  set_reset (c, enc->buf);
  cdk_stream_set_cipher_flag (enc->buf, c->dek, use_mdc);
  rc = cdk_stream_read (enc->buf, NULL, 0);
  if (!rc)
    c->s = enc->buf;
  return rc;
}


static int
handle_compressed (CTX c, CDK_PACKET * pkt)
{
  cdkPKT_compressed * zip;
  int rc;

  assert (pkt->pkttype == CDK_PKT_COMPRESSED);
  
  zip = pkt->pkt.compressed;
  cdk_stream_set_compress_flag (c->s, zip->algorithm);
  rc = cdk_stream_read (c->s, NULL, 0);

  return rc;
}


static int
handle_onepass_sig (CTX c, CDK_PACKET * pkt)
{
  assert (pkt->pkttype == CDK_PKT_ONEPASS_SIG);
  
  c->sig.digest_algo = pkt->pkt.onepass_sig->digest_algo;
  c->sig.md = gcry_md_open (c->sig.digest_algo, 0);
  if (!c->sig.md)
    return CDK_Gcry_Error;
  
  return 0;
}


static int
handle_plaintext (CTX c, CDK_PACKET * pkt, CDK_STREAM out)
{
  plaintext_filter_s * pfx;
  cdkPKT_plaintext * pt;
  int rc;

  assert (pkt->pkttype == CDK_PKT_PLAINTEXT);

  pt = pkt->pkt.plaintext;
  cdk_stream_seek (c->s, c->sig.present? c->sig.pt_offset : 0);
  cdk_stream_set_literal_flag (c->s, 0, NULL);
  if (c->sig.present)
    {
      pfx = _cdk_stream_get_opaque (c->s, fPLAINTEXT);
      if (pfx)
        pfx->md = c->sig.md;
    }
  rc = cdk_stream_kick_off (c->s, out);
  
  return rc;
}


static int
handle_signature (CDK_HD hd, CTX c, CDK_PACKET * pkt)
{
  Verify_Result res;
  cdkPKT_signature * sig;
  u32 keyid[2];
  int rc;

  assert (pkt->pkttype == CDK_PKT_SIGNATURE);

  if (!c->sig.present)
    return CDK_Inv_Packet;

  _cdk_result_verify_free (hd->result.verify);
  res = hd->result.verify = _cdk_result_verify_new ();
  if (!hd->result.verify)
    return CDK_Out_Of_Core;
  
  sig = pkt->pkt.signature;
  if (!c->sig.one_pass && !c->sig.md)
    {
      rc = hash_data_file (c->file, sig->digest_algo, &c->sig.md);
      if (rc)
        return rc;
    }
  
  cdk_sig_get_keyid (sig, keyid);
  res->keyid[0] = keyid[0];
  res->keyid[1] = keyid[1];
  res->created = sig->timestamp;
  res->pubkey_algo = sig->pubkey_algo;
  res->digest_algo = sig->digest_algo;
  
  rc = cdk_keydb_get_pk (NULL, keyid, &c->sig.pk);
  if (rc)
    {
      res->sig_status = CDK_SIGSTAT_NOKEY;
      return rc;
    }

  rc = _cdk_signature_check (c->sig.pk, sig, c->sig.md, &c->sig.is_expired);
  res->sig_status = !rc? CDK_SIGSTAT_GOOD : CDK_SIGSTAT_BAD;
  if (!rc)
    _cdk_log_debug ("Good Signature from %08lX%08lX (expired %d)\n",
                    keyid[0], keyid[1], c->sig.is_expired);
  
  return rc;
}


static void
free_mainproc (CTX c)
{
  if (c)
    {
      if (c->dek)
        cdk_stream_close (c->s);
      cdk_kbnode_release (c->node);
      if (c->sig.present)
        {
          gcry_md_close (c->sig.md);
          _cdk_free_pubkey (c->sig.pk);
        }
      cdk_free (c->file);
      cdk_free (c->dek);
    }
}


static int
do_proc_packets (CDK_HD hd, CTX c, CDK_STREAM inp, CDK_STREAM out)
{
  CDK_PACKET *pkt = NULL;
  CDK_KBNODE n = NULL, node;
  const char *s;
  int rc = 0, npos;

  if (!hd || !c)
    return CDK_Inv_Value;

  s = _cdk_stream_get_fname (inp);
  c->file = cdk_strdup (s? s : " ");
  if (!c->file)
    {
      cdk_free (c);
      return CDK_Out_Of_Core;
    }
  
  while (!cdk_stream_eof (inp))
    {
      pkt = cdk_calloc (1, sizeof *pkt);
      if (!pkt)
        return CDK_Out_Of_Core;
      rc = cdk_pkt_parse (inp, pkt);
      _cdk_log_debug ("type=%d len=%d (%d)\n",pkt->pkttype, pkt->pktlen, rc);
      if (rc == CDK_EOF)
        c->eof_seen = 1;
      if (rc)
        break;
      
      n = cdk_kbnode_new (pkt);
      if (!c->node)
        c->node = n;
      else
        cdk_kbnode_add (c->node, n);
      
      switch (pkt->pkttype)
        {
        case CDK_PKT_SYMKEY_ENC:
          rc = handle_symkey_enc (c, pkt);
          _cdk_log_debug (" handle_symkey_enc (%d)\n", rc);
          break;
          
        case CDK_PKT_PUBKEY_ENC:
          rc = handle_pubkey_enc (c, pkt);
          _cdk_log_debug (" handle_pubkey_enc (%d)\n", rc); 
          break;
          
        case CDK_PKT_ENCRYPTED_MDC: 
        case CDK_PKT_ENCRYPTED:
          rc = handle_encrypted (c, pkt, pkt->pkttype==CDK_PKT_ENCRYPTED_MDC);
          _cdk_log_debug (" handle_encrypted (%d)\n", rc);
          if (!rc)
            inp = c->s;
          break;
          
        case CDK_PKT_COMPRESSED:
          if (!c->s)
            c->s = inp;
          set_reset (c, c->s);
          rc = handle_compressed (c, pkt);
          _cdk_log_debug (" handle_compressed (%d)\n", rc);
          break;

        case CDK_PKT_ONEPASS_SIG:
          if (!c->s)
            c->s = inp;
          _cdk_log_debug (" handle_onepass_sig (0)\n");
          c->sig.present = 1;
          c->sig.one_pass = 1;
          c->sig.pt_offset = cdk_stream_tell (c->s);
          break;

        case CDK_PKT_PLAINTEXT:
          /* skip rest of the packet */
          npos = cdk_stream_tell (c->s) + pkt->pkt.plaintext->len;
          cdk_stream_seek (c->s, npos);
          set_reset (c, c->s);
          break;
          
        case CDK_PKT_SIGNATURE:
          if (!c->sig.present)
            c->sig.present = 1;
          break; /* handle it later */

        default:
          rc = CDK_Inv_Packet;
          break; 
        }
      if (rc)
        break;
    }
  if (c->eof_seen == 1)
    rc = 0;
  for (node = c->node; !rc && node; node = node->next)
    {
      pkt = node->pkt;
      switch (pkt->pkttype)
        {
        case CDK_PKT_ONEPASS_SIG:
          rc = handle_onepass_sig (c, pkt);
          _cdk_log_debug (" _handle_onepass_sig (%d)\n", rc);
          break;
          
        case CDK_PKT_PLAINTEXT:
          rc = handle_plaintext (c, pkt, out);
          _cdk_log_debug (" _handle_plaintext (%d)\n", rc);
          break;

        case CDK_PKT_SIGNATURE:
          rc = handle_signature (hd, c, pkt);
          _cdk_log_debug (" _handle_signature (%d)\n", rc);
          break;

        default:
          break;
        }
      if (rc)
        break;
    }
  if (rc == CDK_EOF)
    rc = CDK_Wrong_Seckey;
  return rc;
}


static int
proc_encryption_packets (CDK_HD hd, CDK_STREAM inp, CDK_STREAM out)
{
  CTX c;
  int rc;

  if (!inp || !out)
    return CDK_Inv_Value;
  
  c = cdk_calloc (1, sizeof *c);
  if (!c)
      return CDK_Out_Of_Core; 
  
  rc = do_proc_packets (hd, c, inp, out);
  free_mainproc (c);
  return rc;
}


int
_cdk_proc_signature_packets (CDK_HD hd, GCRY_MD_HD md,
                             CDK_STREAM inp, CDK_STREAM out)
{
  CTX c;
  int rc;

  if (!inp)
    return CDK_Inv_Value;
  
  c = cdk_calloc (1, sizeof *c);
  if (!c)
    return CDK_Out_Of_Core;
  if (md)
    c->sig.md = md;
  rc = do_proc_packets (hd, c, inp, out);
  free_mainproc (c);
  return rc;
}


int
cdk_file_decrypt (CDK_HD hd, const char *file, const char *output)
{
  CDK_STREAM inp, out;
  CDK_PACKET *pkt;
  int rc;
  
  if (!file || !output)
    return CDK_Inv_Value;

  pkt = cdk_calloc (1, sizeof *pkt);
  if (!pkt)
    return CDK_Out_Of_Core;

  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;

  rc = cdk_stream_new (output, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  rc = proc_encryption_packets (hd, inp, out);

  cdk_stream_close (inp);
  cdk_stream_close (out);
  
  return rc;
}

