/* kbnode.c -  keyblock node utility functions
 *        Copyright (C) 1998-2001 Free Software Foundation, Inc.
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"


#define USE_UNUSED_NODES 1
static CDK_KBNODE unused_nodes;


static CDK_KBNODE
alloc_node (void)
{
  CDK_KBNODE n;

  n = unused_nodes;
  if (n)
    unused_nodes = n->next;
  else
    {
      n = cdk_calloc (1, sizeof *n);
      if (!n)
	return NULL;
    }
  n->next = NULL;
  n->pkt = NULL;
  n->flag = 0;
  n->private_flag = 0;

  return n;
}


static void
free_node (CDK_KBNODE n)
{
  if (n)
    {
#if USE_UNUSED_NODES
      n->next = unused_nodes;
      unused_nodes = n;
#else
      cdk_free (n);
#endif
    }
}


CDK_KBNODE
cdk_kbnode_new (CDK_PACKET * pkt)
{
  CDK_KBNODE n = alloc_node ();
  if (!n)
    return NULL;
  n->pkt = pkt;

  return n;
}


CDK_KBNODE
cdk_kbnode_clone (CDK_KBNODE node)
{
  CDK_KBNODE n = alloc_node ();
  if (!n)
    return NULL;
  n->pkt = node->pkt;
  n->private_flag = node->private_flag | 2; /* mark cloned */

  return n;
}


void
cdk_kbnode_release (CDK_KBNODE n)
{
  CDK_KBNODE n2;

  while (n)
    {
      n2 = n->next;
      n->pkt->pkttype = 0;
      if (!is_cloned_kbnode (n))
	cdk_pkt_release (n->pkt);
      free_node (n);
      n = n2;
    }
}


/****************
 * Delete NODE.
 * Note: This only works with walk_kbnode!!
 */
void
cdk_kbnode_delete (CDK_KBNODE node)
{
  if (node)
    node->private_flag |= 1;
}


/****************
 * Append NODE to ROOT.  ROOT must exist!
 */
void
cdk_kbnode_add (CDK_KBNODE root, CDK_KBNODE node)
{
  CDK_KBNODE n1;

  for (n1 = root; n1->next; n1 = n1->next)
    ;
  n1->next = node;
}


/****************
 * Insert NODE into the list after root but before a packet which is not of
 * type PKTTYPE (only if PKTTYPE != 0)
 */
void
cdk_kbnode_insert (CDK_KBNODE root, CDK_KBNODE node, int pkttype)
{
  if (!pkttype)
    {
      node->next = root->next;
      root->next = node;
    }
  else
    {
      CDK_KBNODE n1;

      for (n1 = root; n1->next; n1 = n1->next)
	if (pkttype != n1->next->pkt->pkttype)
	  {
	    node->next = n1->next;
	    n1->next = node;
	    return;
	  }
      /* no such packet, append */
      node->next = NULL;
      n1->next = node;
    }
}


/****************
 * Find the previous node (if PKTTYPE = 0) or the previous node
 * with pkttype PKTTYPE in the list starting with ROOT of NODE.
 */
CDK_KBNODE
cdk_kbnode_find_prev (CDK_KBNODE root, CDK_KBNODE node, int pkttype)
{
  CDK_KBNODE n1;

  for (n1 = NULL; root && root != node; root = root->next)
    {
      if (!pkttype || root->pkt->pkttype == pkttype)
	n1 = root;
    }

  return n1;
}


/****************
 * Ditto, but find the next packet.  The behaviour is trivial if
 * PKTTYPE is 0 but if it is specified, the next node with a packet
 * of this type is returned.  The function has some knowledge about
 * the valid ordering of packets: e.g. if the next signature packet
 * is requested, the function will not return one if it encounters
 * a user-id.
 */
CDK_KBNODE
cdk_kbnode_find_next (CDK_KBNODE node, int pkttype)
{
  for (node = node->next; node; node = node->next)
    {
      if (!pkttype)
	return node;
      else if (pkttype == CDK_PKT_USER_ID
	       && (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
		   || node->pkt->pkttype == CDK_PKT_SECRET_KEY))
	return NULL;
      else if (pkttype == CDK_PKT_SIGNATURE
	       && (node->pkt->pkttype == CDK_PKT_USER_ID
		   || node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
		   || node->pkt->pkttype == CDK_PKT_SECRET_KEY))
	return NULL;
      else if (node->pkt->pkttype == pkttype)
	return node;
    }

  return NULL;
}


CDK_KBNODE
cdk_kbnode_find (CDK_KBNODE node, int pkttype)
{
  for (; node; node = node->next)
    {
      if (node->pkt->pkttype == pkttype)
	return node;
    }

  return NULL;
}


CDK_PACKET*
cdk_kbnode_find_packet (CDK_KBNODE node, int pkttype)
{
  CDK_KBNODE res;
  
  res = cdk_kbnode_find (node, pkttype);
  if (res)
    return node->pkt;
  return NULL;
}


/****************
 * Walk through a list of kbnodes. This function returns
 * the next kbnode for each call; before using the function the first
 * time, the caller must set CONTEXT to NULL (This has simply the effect
 * to start with ROOT).
 */
CDK_KBNODE
cdk_kbnode_walk (CDK_KBNODE root, CDK_KBNODE * context, int all)
{
  CDK_KBNODE n;

  do
    {
      if (!*context)
	{
	  *context = root;
	  n = root;
	}
      else
	{
	  n = (*context)->next;
	  *context = n;
	}
    }
  while (!all && n && is_deleted_kbnode (n));

  return n;
}


void
cdk_kbnode_clear_flags (CDK_KBNODE n)
{
  for (; n; n = n->next)
    {
      n->flag = 0;
    }
}


/****************
 * Commit changes made to the kblist at ROOT. Note that ROOT my change,
 * and it is therefore passed by reference.
 * The function has the effect of removing all nodes marked as deleted.
 * returns true if any node has been changed
 */
int
cdk_kbnode_commit (CDK_KBNODE * root)
{
  CDK_KBNODE n, nl;
  int changed = 0;

  for (n = *root, nl = NULL; n; n = nl->next)
    {
      if (is_deleted_kbnode (n))
	{
	  if (n == *root)
	    *root = nl = n->next;
	  else
	    nl->next = n->next;
	  if (!is_cloned_kbnode (n))
	    {
	      cdk_pkt_release (n->pkt);
	      cdk_free (n->pkt);
	    }
	  free_node (n);
	  changed = 1;
	}
      else
	nl = n;
    }

  return changed;
}


void
cdk_kbnode_remove (CDK_KBNODE * root, CDK_KBNODE node)
{
  CDK_KBNODE n, nl;

  for (n = *root, nl = NULL; n; n = nl->next)
    {
      if (n == node)
	{
	  if (n == *root)
	    *root = nl = n->next;
	  else
	    nl->next = n->next;
	  if (!is_cloned_kbnode (n))
	    {
	      cdk_pkt_release (n->pkt);
	      cdk_free (n->pkt);
	    }
	  free_node (n);
	}
      else
	nl = n;
    }
}


/****************
 * Move NODE behind right after WHERE or to the beginning if WHERE is NULL.
 */
void
cdk_kbnode_move (CDK_KBNODE * root, CDK_KBNODE node, CDK_KBNODE where)
{
  CDK_KBNODE tmp, prev;

  if (!root || !*root || !node)
    return;			/* sanity check */
  for (prev = *root; prev && prev->next != node; prev = prev->next)
    ;
  if (!prev)
    return;			/* node is not in the list */

  if (!where)
    {				/* move node before root */
      if (node == *root)	/* move to itself */
	return;
      prev->next = node->next;
      node->next = *root;
      *root = node;
      return;
    }
  /* move it after where */
  if (node == where)
    return;
  tmp = node->next;
  node->next = where->next;
  where->next = node;
  prev->next = tmp;
}


CDK_PACKET *
cdk_kbnode_get_packet (CDK_KBNODE node)
{
  if (node)
    return node->pkt;
  return NULL;
}
