/* keydb.c - Key database routines
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include "opencdk.h"
#include "main.h"
#include "packet.h"
#include "filters.h"
#include "stream.h"


#define KEYDB_CACHE_ENTRIES 8

typedef struct key_table_s * KEY_TABLE;
typedef struct key_idx_s * KEY_IDX;


static void keydb_cache_free (KEY_TABLE cache);
static int keydb_search_copy (CDK_DBSEARCH * r_dst, CDK_DBSEARCH src);

     
static char *
keydb_idx_mkname (const char * file)
{
  char * fname;
  fname = cdk_calloc (1, strlen (file) + 5);
  strcpy (fname, file);
  strcat (fname, ".idx");
  return fname;
}


/* This functions builds an index of the keyring into a separate file
   with the name keyring.ext.idx. It contains the offset of all public-
   and public subkeys. The format of the file is:
   --------
    4 octets offset of the packet
    8 octets keyid
   20 octets fingerprint
   --------
   We store the keyid and the fingerprint due to the fact we can't get
   the keyid from a v3 fingerprint directly.
*/
static int
keydb_idx_build (const char * file)
{
  CDK_PACKET pkt;
  CDK_STREAM inp, out;
  byte buf[8], fpr[20];
  u32 keyid[2];
  char * fname;
  long pos, pkttype;
  int rc;

  if (!file)
    return CDK_Inv_Value;
  
  fname = keydb_idx_mkname (file);
  
  rc = cdk_stream_open (file, &inp);
  if (rc)
    return rc;
  rc = cdk_stream_new (fname, &out);
  if (rc)
    {
      cdk_stream_close (inp);
      return rc;
    }

  while (!cdk_stream_eof (inp))
    {
      pos = cdk_stream_tell (inp);
      rc = cdk_pkt_parse (inp, &pkt);
      if (rc)
        break;
      pkttype = pkt.pkttype;
      if (pkttype == CDK_PKT_PUBLIC_KEY || pkttype == CDK_PKT_PUBLIC_SUBKEY)
        {
          _cdk_u32tobuf (pos, buf);
          cdk_stream_write (out, buf, 4);
          cdk_pk_get_keyid (pkt.pkt.public_key, keyid);
          _cdk_u32tobuf (keyid[0], buf);
          _cdk_u32tobuf (keyid[1], buf + 4);
          cdk_stream_write (out, buf, 8);
          cdk_pk_get_fingerprint (pkt.pkt.public_key, fpr);
          cdk_stream_write (out, fpr, 20);
        }
      cdk_pkt_free (&pkt);
    }
  cdk_stream_close (inp);
  cdk_stream_close (out);
  cdk_free (fname);
  return rc;
}


static int
keydb_idx_rebuild (CDK_KEYDB_HD hd)
{
  int rc;
  
  if (!hd || !hd->name)
    return CDK_Inv_Value;
  if (hd->secret)
    return 0;
  
  cdk_stream_close (hd->idx);
  if (!hd->idx_name)
      hd->idx_name = keydb_idx_mkname (hd->name);
  rc = keydb_idx_build (hd->name);
  if (!rc)
    rc = cdk_stream_open (hd->idx_name, &hd->idx);
  return rc;
}


static int
keydb_idx_parse (CDK_STREAM inp, KEY_IDX * r_idx)
{
  KEY_IDX idx;
  byte buf[4];
  int i;

  idx = cdk_calloc (1, sizeof * idx);
  if (!idx)
    return CDK_Out_Of_Core;

  while (!cdk_stream_eof (inp))
    {
      i = cdk_stream_read (inp, buf, 4);
      if (i == CDK_EOF)
        break;
      idx->offset = buf[0] << 24 | buf[1] << 16 | buf[2] << 8 | buf[3];
      cdk_stream_read (inp, buf, 4);
      idx->keyid[0] = buf[0] << 24 | buf[1] << 16 | buf[2] << 8 | buf[3];
      cdk_stream_read (inp, buf, 4);
      idx->keyid[1] = buf[0] << 24 | buf[1] << 16 | buf[2] << 8 | buf[3];
      cdk_stream_read (inp, idx->fpr, 20);
      #if 0
      _cdk_log_debug ("%08lX%08lX %06lu ",
                      idx->keyid[0], idx->keyid[1], idx->offset);
      for (i = 0; i < 20; i++)
        _cdk_log_debug ("%02X", idx->fpr[i]);
      _cdk_log_debug ("\n");
      #endif
      break; 
    }
  if (r_idx)
    *r_idx = idx;
  return cdk_stream_eof (inp)? CDK_EOF : 0;  
}


static int
keydb_idx_search (CDK_STREAM inp, u32 * keyid, const byte * fpr, u32 * r_off)
{
  KEY_IDX idx;

  if (keyid && fpr)
    return CDK_Inv_Mode;
  if (!keyid && !fpr)
    return CDK_Inv_Value;
  if (!r_off)
    return CDK_Inv_Value;

  *r_off = 0;
  cdk_stream_seek (inp, 0);
  while (keydb_idx_parse (inp, &idx) != CDK_EOF)
    {
      if (keyid && idx->keyid[0] == keyid[0] && idx->keyid[1] == keyid[1])
        {
          *r_off = idx->offset;
          break; 
        }
      else if (fpr && !memcmp (idx->fpr, fpr, 20))
        {
          *r_off = idx->offset;
          break; 
        }
      cdk_free (idx);
      idx = NULL; 
    }
  cdk_free (idx);
  return *r_off? 0 : CDK_EOF;
}


int
cdk_keydb_new (CDK_KEYDB_HD * r_hd, int type, void * data, size_t count)
{
  CDK_KEYDB_HD hd;

  hd = cdk_calloc (1, sizeof *hd);
  if (!hd)
    return CDK_Out_Of_Core;
  
  switch (type)
    {
    case CDK_DBTYPE_PK_KEYRING:
    case CDK_DBTYPE_SK_KEYRING:
      hd->name = cdk_calloc (1, count + 1);
      if (!hd->name)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      memcpy (hd->name, data, count);
      hd->name[count] = '\0';
      hd->type = type;
      break;
      
    case CDK_DBTYPE_DATA:
      hd->buf = cdk_stream_tmp_from_mem (data, count);
      if (!hd->buf)
        {
          cdk_free (hd);
          return CDK_Out_Of_Core;
        }
      hd->type = CDK_DBTYPE_DATA;
      break;
    }
  if (type == CDK_DBTYPE_SK_KEYRING)
    hd->secret = 1;
  if (r_hd)
    *r_hd = hd;
  return 0;
}


void
cdk_keydb_free (CDK_KEYDB_HD hd)
{
  if (hd)
    {
      if (hd->isopen && hd->name)
        {
          cdk_free (hd->name);
          cdk_stream_close (hd->buf);
          hd->buf = NULL;
        }
      if (!hd->secret)
        {
          cdk_stream_close (hd->idx);
          hd->idx = NULL;
        }
      hd->no_cache = 0;
      hd->secret = 0;
      hd->isopen = 0;
      cdk_free (hd);
    }
}


int
cdk_keydb_open (CDK_KEYDB_HD hd, CDK_STREAM * ret_kr)
{
  int rc = 0, ec, type;

  if (!hd)
    return CDK_Inv_Value;

  type = hd->type;
  if (type == CDK_DBTYPE_DATA)
    {
      if (hd->buf)
        cdk_stream_seek (hd->buf, 0);
    }
  else if (type == CDK_DBTYPE_PK_KEYRING || type == CDK_DBTYPE_SK_KEYRING)
    {
      if (!hd->isopen && hd->name)
        {
          rc = cdk_stream_open (hd->name, &hd->buf);
          if (rc)
            goto leave;
          if (cdk_armor_filter_use (hd->buf))
            cdk_stream_set_armor_flag (hd->buf, 0);
          hd->isopen = 1;
          cdk_free (hd->idx_name);
          hd->idx_name = keydb_idx_mkname (hd->name);
          ec = cdk_stream_open (hd->idx_name, &hd->idx);
          if (ec && !hd->secret)
            {
              rc = keydb_idx_build (hd->name);
              if (!rc)
                rc = cdk_stream_open (hd->idx_name, &hd->idx);
              if (!rc)
                _cdk_log_debug ("create key index table\n");
              if (rc)
                {
                  /* this is no real error, it just means we can't create
                     the index at the given directory. maybe we've no write
                     access. in this case, we simply disable the index. */
                  _cdk_log_debug ("disable key index table\n");
                  rc = 0;
                  hd->no_cache = 1;
                }
            }
        }
      else
        cdk_stream_seek (hd->buf, 0);
    }
  else
    return CDK_Inv_Mode;
  
 leave:
  if (rc)
    {
      cdk_stream_close (hd->buf);
      hd->buf = NULL;
    }
  if (ret_kr)
    *ret_kr = hd->buf;
  return rc;
}


static int
find_by_keyid (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  u32 keyid[2];
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_keyid_get (node->pkt, keyid);
          switch (ks->type)
            {
            case CDK_DBSEARCH_SHORT_KEYID:
              if (keyid[1] == ks->u.keyid[1])
                {
                  key_found = 1;
                  break;
                }
              break;

            case CDK_DBSEARCH_KEYID:
              if (keyid[1] == ks->u.keyid[1] && keyid[0] == ks->u.keyid[0])
                {
                  key_found = 1;
                  break;
                }
              break;
            }
        }
    }
  return key_found;
}


static int
find_by_fpr (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  int key_found = 0;
  byte fpr[20];
  
  if (ks->type != CDK_DBSEARCH_FPR)
    return key_found;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || node->pkt->pkttype == CDK_PKT_SECRET_KEY
          || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
        {
          _cdk_fingerprint_get (node->pkt, fpr);
          if (!memcmp (ks->u.fpr, fpr, 20))
            {
              key_found = 1;
              break;
            }
        }
    }
  return key_found;
}


static int
find_by_pattern (CDK_KBNODE knode, CDK_DBSEARCH ks)
{
  CDK_KBNODE node;
  size_t uidlen;
  char * name;
  int key_found = 0;

  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype != CDK_PKT_USER_ID)
        continue;
      uidlen = node->pkt->pkt.user_id->len;
      name = node->pkt->pkt.user_id->name;
      switch (ks->type)
        {
        case CDK_DBSEARCH_EXACT:
          if (name
              && (strlen (ks->u.pattern) == uidlen
              && !strncmp (ks->u.pattern, name, uidlen)))
            {
              key_found = 1;
              break;
            }
          break;

        case CDK_DBSEARCH_SUBSTR:
          if (uidlen > 65536)
            break;
          if (name && strlen (ks->u.pattern) > uidlen)
            break;
          if (name && _cdk_memistr (name, uidlen, ks->u.pattern))
            {
              key_found = 1;
              break;
            }
          break;
        }
    }
  return key_found;
}


static void
keydb_cache_free (KEY_TABLE cache)
{
  KEY_TABLE c2;

  while (cache)
    {
      c2 = cache->next;
      cache->offset = 0;
      cdk_keydb_search_free (cache->desc);
      cdk_free (cache);
      cache = NULL;
      cache = c2;
    }
}


static KEY_TABLE
keydb_cache_find (KEY_TABLE cache, CDK_DBSEARCH desc)
{
  KEY_TABLE t;
  
  for (t = cache; t; t = t->next)
    {
      if (t->desc->type != desc->type)
        continue;
      switch (t->desc->type)
        {
        case CDK_DBSEARCH_SHORT_KEYID:
        case CDK_DBSEARCH_KEYID:
          if (t->desc->u.keyid[0] == desc->u.keyid[0]
              && t->desc->u.keyid[1] == desc->u.keyid[1])
            return t;
          break;

        case CDK_DBSEARCH_EXACT:
          if (strlen (t->desc->u.pattern) == strlen (desc->u.pattern)
              && !strcmp (t->desc->u.pattern, desc->u.pattern))
            return t;
          break;

        case CDK_DBSEARCH_SUBSTR:
          if (strstr (t->desc->u.pattern, desc->u.pattern))
            return t;
          break;

        case CDK_DBSEARCH_FPR:
          if (!memcmp (t->desc->u.fpr, desc->u.fpr, 20))
            return t;
          break;
        }
    }
  return NULL;
}
  

static int
keydb_cache_add (CDK_KEYDB_HD hd, CDK_DBSEARCH dbs, u32 offset)
{
  KEY_TABLE k;

  if (hd->ncache > KEYDB_CACHE_ENTRIES)
    return 0;
  k = cdk_calloc (1, sizeof *k);
  if (!k)
    return CDK_Out_Of_Core;
  k->offset = offset;
  keydb_search_copy (&k->desc, dbs);
  k->next = hd->cache;
  hd->cache = k;
  hd->ncache++;
  _cdk_log_debug ("add entry [o=%d t=%d] to the cache\n", offset, dbs->type);
  return 0;
}

      
void
cdk_keydb_search_reset (CDK_KEYDB_HD hd)
{
  if (hd)
    {
      keydb_cache_free (hd->cache);
      hd->cache = NULL;
    }
}


static int
keydb_search_copy (CDK_DBSEARCH * r_dst, CDK_DBSEARCH src)
{
  CDK_DBSEARCH dst;
  
  if (!r_dst || !src)
    return CDK_Inv_Value;
  dst = cdk_calloc (1, sizeof *dst);
  if (!dst)
    return CDK_Out_Of_Core;
  dst->type = src->type;
  switch (src->type)
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      dst->u.pattern = cdk_strdup (src->u.pattern);
      break;

    case CDK_DBSEARCH_SHORT_KEYID:
    case CDK_DBSEARCH_KEYID:
      dst->u.keyid[0] = src->u.keyid[0];
      dst->u.keyid[1] = src->u.keyid[1];
      break;

    case CDK_DBSEARCH_FPR:
      memcpy (dst->u.fpr, src->u.fpr, 20);
      break;
    }
  *r_dst = dst;
  return 0;
}

  
int
cdk_keydb_search_new (CDK_DBSEARCH * r_dbs, int type, void * desc)
{
  CDK_DBSEARCH c;
  u32 * keyid;

  c = cdk_calloc (1, sizeof *c);
  if (!c)
    return CDK_Out_Of_Core;
  c->type = type;
  switch (type)
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      cdk_free (c->u.pattern);
      c->u.pattern = cdk_strdup (desc);
      if (!c->u.pattern)
        {
          cdk_free (c);
          return CDK_Out_Of_Core;
        }
      break;

    case CDK_DBSEARCH_SHORT_KEYID:
      keyid = desc;
      c->u.keyid[1] = keyid[0];
      break;
      
    case CDK_DBSEARCH_KEYID:
      keyid = desc;
      c->u.keyid[0] = keyid[0];
      c->u.keyid[1] = keyid[1];
      break;

    case CDK_DBSEARCH_FPR:
      memcpy (c->u.fpr, desc, 20);
      break;

    default:
      cdk_free (c);
      return CDK_Inv_Mode;
    }
  if (r_dbs)
    *r_dbs = c;
  return 0;
}


void
cdk_keydb_search_free (CDK_DBSEARCH dbs)
{
  if (dbs)
    {
      if (dbs->type == CDK_DBSEARCH_EXACT || dbs->type == CDK_DBSEARCH_SUBSTR)
        cdk_free (dbs->u.pattern);
      dbs->type = 0;
      cdk_free (dbs);
    }
}


static int
keydb_pos_from_cache (CDK_KEYDB_HD hd, CDK_DBSEARCH ks,
                      int * r_cache_hit, u32 * r_off)
{
  KEY_TABLE c;
  u32 off = 0;
  int cache_hit = 0;

  if (!hd)
    return CDK_Inv_Value;
  
  c = keydb_cache_find (hd->cache, ks);
  if (c)
    {
      _cdk_log_debug ("found entry in cache.\n");
      cache_hit = 1;
      off = c->offset;
    }

  if (hd->idx && !c)
    {
      if (ks->type == CDK_DBSEARCH_KEYID)
        {
          if (keydb_idx_search (hd->idx, ks->u.keyid, NULL, &off))
            return CDK_Error_No_Key;
          _cdk_log_debug ("found keyid entry in idx table.\n");
          cache_hit = 1;
        }
      else if (ks->type == CDK_DBSEARCH_FPR)
        {
          if (keydb_idx_search (hd->idx, NULL, ks->u.fpr, &off))
            return CDK_Error_No_Key;
          _cdk_log_debug ("found fpr entry in idx table.\n");
          cache_hit = 1;
        }
    }
  if (r_off)
    *r_off = off;
  if (r_cache_hit)
    *r_cache_hit = cache_hit;
  return 0;
}


int
cdk_keydb_search (CDK_KEYDB_HD hd, CDK_DBSEARCH ks, CDK_KBNODE * r_key)
{
  CDK_STREAM kr = NULL;
  CDK_KBNODE knode = NULL;
  int key_found = 0, cache_hit = 0;
  int rc = 0;
  u32 off = 0;
  size_t pos = 0;

  if (!hd || !ks)
    return CDK_Inv_Value;

  if (r_key)
    *r_key = NULL;

  rc = cdk_keydb_open (hd, &kr);
  if (rc)
    return rc;

  rc = keydb_pos_from_cache (hd, ks, &cache_hit, &off);
  if (rc)
    return rc;

  while (!key_found && !rc)
    {
      if (cache_hit)
        cdk_stream_seek (kr, off);
      pos = cdk_stream_tell (kr);
      rc = cdk_keydb_get_keyblock (kr, &knode);
      if (rc)
        {
          if (rc == CDK_EOF && knode)
            rc = 0;
          if (!knode && rc == CDK_EOF)
            rc = CDK_Error_No_Key;
          if (rc)
            break;
        }

      switch (ks->type)
        {
        case CDK_DBSEARCH_SHORT_KEYID:
        case CDK_DBSEARCH_KEYID:
          key_found = find_by_keyid (knode, ks);
          break;

        case CDK_DBSEARCH_FPR:
          key_found = find_by_fpr (knode, ks);
          break;
          
	case CDK_DBSEARCH_EXACT:
	case CDK_DBSEARCH_SUBSTR:
          key_found = find_by_pattern (knode, ks);
	  break;
	}

      if (key_found)
        {
          if (!keydb_cache_find (hd->cache, ks))
            keydb_cache_add (hd, ks, pos);
          break;
        }

      cdk_kbnode_release (knode);
      knode = NULL;
    }

  if (r_key)
    *r_key = key_found? knode : NULL;
  return rc;
}


int
cdk_keydb_get_bykeyid (CDK_KEYDB_HD hd, u32 * keyid, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!hd || !keyid || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SHORT_KEYID;
  ks.u.keyid[1] = keyid[1];

  return cdk_keydb_search (hd, &ks, ret_pk);
}


int
cdk_keydb_get_byfpr (CDK_KEYDB_HD hd, const byte * fpr, CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;

  if (!hd || !fpr || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_FPR;
  memcpy (ks.u.fpr, fpr, 20);

  return cdk_keydb_search (hd, &ks, ret_pk);
}


int
cdk_keydb_get_bypattern (CDK_KEYDB_HD hd, const char *patt,
			 CDK_KBNODE * ret_pk)
{
  struct cdk_keydb_search_s ks;
  int rc;

  if (!hd || !patt || !ret_pk)
    return CDK_Inv_Value;

  ks.type = CDK_DBSEARCH_SUBSTR;
  ks.u.pattern = cdk_strdup (patt);
  rc = cdk_keydb_search (hd, &ks, ret_pk);
  cdk_free (ks.u.pattern);
  return rc;
}


static int
keydb_check_key (CDK_PACKET * pkt)
{
  cdkPKT_public_key * pk;
  
  if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
      || pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
    pk = pkt->pkt.public_key;
  else if (pkt->pkttype == CDK_PKT_SECRET_KEY
      || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
    pk = pkt->pkt.secret_key->pk;
  else
    return 0;
  return !pk->is_revoked && !pk->has_expired;
}


static CDK_KBNODE
keydb_find_byusage (CDK_KBNODE root, int req_usage, int pk)
{
  CDK_KBNODE node;
  int pkttype = 0;

  if (!req_usage)
    return cdk_kbnode_find (root, CDK_PKT_PUBLIC_KEY);

  for (node = root; node; node = node->next)
    {
      pkttype = node->pkt->pkttype;
      if (pk && (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
                 || node->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
          && keydb_check_key (node->pkt)
          && (node->pkt->pkt.public_key->pubkey_usage & req_usage))
	return node;
      if (!pk && (node->pkt->pkttype == CDK_PKT_SECRET_KEY
                  || node->pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
          && keydb_check_key (node->pkt)
          && (node->pkt->pkt.secret_key->req_usage & req_usage))
	return node;
    }
  return NULL;
}


CDK_KBNODE
_cdk_keydb_find_bykeyid (CDK_KBNODE root, u32 * keyid)
{
  CDK_KBNODE node;
  u32 kid[2];

  for (node = root; node; node = node->next)
    {
      _cdk_keyid_get (node->pkt, kid);
      if (kid[1] == keyid[1])
	return node;
    }
  return NULL;
}


int
_cdk_keydb_get_sk_byname (CDK_KEYDB_HD hd, const char * name,
                          cdkPKT_secret_key ** ret_sk, int alloced)
{
  CDK_KBNODE knode = NULL, node = NULL;
  cdkPKT_secret_key * sk = NULL;
  int req_usage = 0;
  int rc = 0;

  if (!ret_sk)
    return CDK_Inv_Value;
  if (!hd)
    return CDK_Error_No_Keyring;
  
  rc = cdk_keydb_search_auto (hd, name, &knode);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_sk[0]->req_usage;
      cdk_free (ret_sk[0]);
      node = keydb_find_byusage (knode, req_usage, 0);
    }
  if (!node)
    {
      node = cdk_kbnode_find (knode, CDK_PKT_SECRET_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  sk = node->pkt->pkt.secret_key;
  _cdk_kbnode_clone (node);
  cdk_kbnode_release (knode);

leave:
  *ret_sk = sk;
  return rc;
}


int
_cdk_keydb_get_pk_byname (CDK_KEYDB_HD hd, const char * name,
                          cdkPKT_public_key ** ret_pk, int alloced)
{
  CDK_KBNODE knode, node = NULL;
  cdkPKT_public_key * pk = NULL;
  const char * s;
  int req_usage = 0;
  int rc = 0;

  if (!name)
    return CDK_Inv_Value;
  if (!hd)
    return CDK_Error_No_Keyring;
  
  rc = cdk_keydb_search_auto (hd, name, &knode);
  if (rc)
    goto leave;
  if (alloced)
    {
      req_usage = ret_pk[0]->pubkey_usage;
      cdk_free (ret_pk[0]);
      node = keydb_find_byusage (knode, req_usage, 1);
    }
  if (!node)
    {
      node = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
      if (!node)
	{
	  rc = CDK_Error_No_Key;
	  goto leave;
	}
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_USER_ID)
	{
	  s = node->pkt->pkt.user_id->name;
	  if (pk && !pk->uid && _cdk_memistr (s, strlen (s), name))
	    {
	      _cdk_copy_userid (&pk->uid, node->pkt->pkt.user_id);
	      break;
	    }
	}
    }
  cdk_kbnode_release (knode);

leave:
  if (ret_pk)
    *ret_pk = pk;
  return rc;
}


int
cdk_keydb_get_pk (CDK_KEYDB_HD hd, u32 * keyid, cdkPKT_public_key ** r_pk)
{
  struct cdk_keydb_search_s ks;
  CDK_KBNODE knode = NULL, node = NULL;
  cdkPKT_public_key * pk = NULL;
  int rc = 0;

  if (!keyid)
    return CDK_Inv_Value;
  if (!hd)
    return CDK_Error_No_Keyring;

  if (!keyid[0])
    {
      ks.type = CDK_DBSEARCH_SHORT_KEYID;
      ks.u.keyid[1] = keyid[1];
    }
  else
    {
      ks.type = CDK_DBSEARCH_KEYID;
      ks.u.keyid[0] = keyid[0];
      ks.u.keyid[1] = keyid[1];
    }
  rc = cdk_keydb_search (hd, &ks, &knode);
  if (rc)
    goto leave;
  node = _cdk_keydb_find_bykeyid (knode, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }
  _cdk_copy_pubkey (&pk, node->pkt->pkt.public_key);
  cdk_kbnode_release (knode);

leave:
  if (r_pk)
    *r_pk = pk;
  return rc;
}


int
cdk_keydb_get_sk (CDK_KEYDB_HD hd, u32 * keyid, cdkPKT_secret_key ** ret_sk)
{
  CDK_KBNODE snode, node;
  cdkPKT_secret_key * sk = NULL;
  int rc = 0;

  if (!keyid)
    return CDK_Inv_Value;
  if (!hd)
    return CDK_Error_No_Keyring;

  rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
  if (rc)
    goto leave;

  node = _cdk_keydb_find_bykeyid (snode, keyid);
  if (!node)
    {
      rc = CDK_Error_No_Key;
      goto leave;
    }

  sk = node->pkt->pkt.secret_key;
  _cdk_kbnode_clone (node);
  cdk_kbnode_release (snode);

leave:
  if (ret_sk)
    *ret_sk = sk;
  return rc;
}


static int
keydb_merge_selfsig (CDK_KBNODE key)
{
  CDK_KBNODE k, kbnode, unode;
  struct cdk_subpkt_s * s = NULL;
  cdkPKT_signature * sig = NULL;
  cdkPKT_user_id * uid = NULL;
  const byte * symalg = NULL, * hashalg = NULL, * compalg = NULL;
  size_t nsymalg = 0, nhashalg = 0, ncompalg = 0, n = 0;
  int key_usage = 0, key_expire = 0;

  if (!key)
    return CDK_Inv_Value;

  for (k = key; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE
	  && k->pkt->pkt.signature->sig_class == 0x13)
	{
	  unode = cdk_kbnode_find_prev (key, k, CDK_PKT_USER_ID);
	  if (!unode)
	    return CDK_Error_No_Key;
	  uid = unode->pkt->pkt.user_id;
	  sig = k->pkt->pkt.signature;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PRIMARY_UID);
	  if (s)
	    uid->is_primary = 1;
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_FEATURES);
	  if (s && s->size == 1 && s->d[0] & 0x01)
	    uid->mdc_feature = 1;
          s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_KEY_EXPIRE);
          if (s && s->size == 4)
            key_expire = _cdk_buftou32 (s->d);
          s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_KEY_FLAGS);
          if (s)
            {
              if (s->d[0] & 3) /* cert + sign data */
                key_usage |= GCRY_PK_USAGE_SIGN;
              if (s->d[0] & 12) /* encrypt comm. + storage */
                key_usage |= GCRY_PK_USAGE_ENCR;
            }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_SYM);
	  if (s)
	    {
	      symalg = s->d;
	      nsymalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_HASH);
	  if (s)
	    {
	      hashalg = s->d;
	      nhashalg = s->size;
	      n += s->size + 1;
	    }
	  s = cdk_subpkt_find (sig->hashed, SIGSUBPKT_PREFS_ZIP);
	  if (s)
	    {
	      compalg = s->d;
	      ncompalg = s->size;
	      n += s->size + 1;
	    }
	  if (!n || !hashalg || !compalg || !symalg)
	    uid->prefs = NULL;
	  else
	    {
	      uid->prefs = cdk_calloc (1, sizeof (*uid->prefs) * (n + 1));
	      if (!uid->prefs)
		return CDK_Out_Of_Core;
	      n = 0;
	      for (; nsymalg; nsymalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_SYM;
		  uid->prefs[n].value = *symalg++;
		}
	      for (; nhashalg; nhashalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_HASH;
		  uid->prefs[n].value = *hashalg++;
		}
	      for (; ncompalg; ncompalg--, n++)
		{
		  uid->prefs[n].type = CDK_PREFTYPE_ZIP;
		  uid->prefs[n].value = *compalg++;
		}
	      /* end of list marker */
	      uid->prefs[n].type = CDK_PREFTYPE_NONE;
	      uid->prefs[n].value = 0;
	      uid->prefs_size = n;

	      kbnode = cdk_kbnode_find_prev (key, k, CDK_PKT_PUBLIC_KEY);
	      if (kbnode)
		{
		  cdkPKT_public_key * pk = kbnode->pkt->pkt.public_key;
                  if (uid->prefs && n)
                    {
                      pk->prefs = _cdk_copy_prefs (uid->prefs);
                      pk->prefs_size = n;
                    }
                  if (key_expire)
                    {
                      pk->expiredate = pk->timestamp + key_expire;
                      pk->has_expired = pk->expiredate> _cdk_timestamp ()?0 :1;
                    } 
                  if (key_usage && !pk->pubkey_usage)
                    pk->pubkey_usage = key_usage;
                }
            }
        }
    }
  return 0;
}


static int
keydb_parse_allsigs (CDK_KBNODE knode, int check)
{
  CDK_KBNODE k, kb;
  cdkPKT_signature * sig;
  cdkPKT_public_key * pk;
  struct cdk_subpkt_s * s = NULL;
  u32 expiredate = 0, curtime = _cdk_timestamp (), keyid[2];
  int rc = 0;

  if (!knode)
    return CDK_Inv_Value;

  kb = cdk_kbnode_find (knode, CDK_PKT_SECRET_KEY);
  if (kb)
    return 0;

  /* reset */
  for (k = knode; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_USER_ID)
	k->pkt->pkt.user_id->is_revoked = 0;
      else if (k->pkt->pkttype == CDK_PKT_PUBLIC_KEY
               || k->pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
	k->pkt->pkt.public_key->is_revoked = 0;
    }

  kb = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
  if (!kb)
    return CDK_Inv_Packet;
  cdk_pk_get_keyid (kb->pkt->pkt.public_key, keyid);
  
  for (k = knode; k; k = k->next)
    {
      if (k->pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  sig = k->pkt->pkt.signature;
	  /* Revocation certificates for primary keys */
	  if (sig->sig_class == 0x20)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certificates for subkeys */
	  else if (sig->sig_class == 0x28)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
		  kb->pkt->pkt.public_key->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Revocation certifcates for user ID's */
	  else if (sig->sig_class == 0x30)
	    {
              if (sig->keyid[0] != keyid[0] || sig->keyid[1] != keyid[1])
                continue; /* revokes an earlier signature, no userID. */
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_USER_ID);
	      if (kb)
		{
		  kb->pkt->pkt.user_id->is_revoked = 1;
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for primary keys */
	  else if (sig->sig_class == 0x1F)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_KEY);
	      if (kb)
		{
                  pk = kb->pkt->pkt.public_key;
		  pk->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      expiredate = _cdk_buftou32 (s->d);
                      pk->expiredate = pk->timestamp + expiredate;
                      pk->has_expired = pk->expiredate > curtime? 0 : 1;
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	  /* Direct certificates for subkeys */
	  else if (sig->sig_class == 0x18)
	    {
	      kb = cdk_kbnode_find_prev (knode, k, CDK_PKT_PUBLIC_SUBKEY);
	      if (kb)
		{
                  pk = kb->pkt->pkt.public_key;
		  pk->is_invalid = 0;
		  s = cdk_subpkt_find (k->pkt->pkt.signature->hashed,
				       SIGSUBPKT_KEY_EXPIRE);
		  if (s)
		    {
		      expiredate = _cdk_buftou32 (s->d);
                      pk->expiredate = pk->timestamp + expiredate;
                      pk->has_expired = pk->expiredate > curtime? 0 : 1;
		    }
		  if (check)
		    ;
		}
	      else
		return CDK_Error_No_Key;
	    }
	}
    }
  k = cdk_kbnode_find (knode, CDK_PKT_PUBLIC_KEY);
  if (k) /* v3 public keys have no additional signatures for the key */
    {
      pk = k->pkt->pkt.public_key;
      if (pk->version == 3)
        pk->is_invalid = 0;
    }
  return rc;
}


int
cdk_keydb_get_keyblock (CDK_STREAM inp, CDK_KBNODE * r_knode)
{
  CDK_PACKET * pkt = NULL;
  CDK_KBNODE knode = NULL, node = NULL;
  u32 keyid[2], main_keyid[2];
  int rc = 0, rc2 = 0, old_off;
  int key_seen = 0, got_key = 0;

  if (!inp)
    return CDK_Inv_Value;

  memset (keyid, 0, sizeof keyid);
  memset (main_keyid, 0, sizeof main_keyid);
  
  while (1)
    {
      pkt = cdk_calloc (1, sizeof *pkt);
      if (!pkt)
	return CDK_Out_Of_Core;
      old_off = cdk_stream_tell (inp);
      rc = cdk_pkt_parse (inp, pkt);
      if (rc)
	break;
      if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
          || pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
          || pkt->pkttype == CDK_PKT_SECRET_KEY
          || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
	{
          if (key_seen && (pkt->pkttype == CDK_PKT_PUBLIC_KEY
                            || pkt->pkttype == CDK_PKT_SECRET_KEY))
	    {
              cdk_stream_seek (inp, old_off);
	      break;
	    }
          if (pkt->pkttype == CDK_PKT_PUBLIC_KEY
              || pkt->pkttype == CDK_PKT_SECRET_KEY)
            {
              _cdk_keyid_get (pkt, main_keyid);
              key_seen = 1;
            }
          else if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY
                   || pkt->pkttype == CDK_PKT_SECRET_SUBKEY)
            {
              if (pkt->pkttype == CDK_PKT_PUBLIC_SUBKEY)
		{
		  pkt->pkt.public_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.public_key->main_keyid[1] = main_keyid[1];
		}
	      else
		{
		  pkt->pkt.secret_key->main_keyid[0] = main_keyid[0];
		  pkt->pkt.secret_key->main_keyid[1] = main_keyid[1];
		}
	    }
	  /* we save this for the signature */
          _cdk_keyid_get (pkt, keyid);
          got_key = 1;
	}
      else if (pkt->pkttype == CDK_PKT_USER_ID)
        ;
      else if (pkt->pkttype == CDK_PKT_SIGNATURE)
	{
	  pkt->pkt.signature->key[0] = keyid[0];
	  pkt->pkt.signature->key[1] = keyid[1];
	}
      node = cdk_kbnode_new (pkt);
      if (!knode)
	knode = node;
      else
	_cdk_kbnode_add (knode, node);
    }

  if (got_key)
    {
      keydb_merge_selfsig (knode);
      rc2 = keydb_parse_allsigs (knode, 0);
      if (!rc)
	rc = rc2;
    }
  if (r_knode)
    *r_knode = got_key ? knode : NULL;
  return rc;
}


static int
classify_data (const byte * buf, size_t len)
{
  int type = 0;
  int i;

  if (buf[0] == '0' && buf[1] == 'x')
    {
      buf += 2;
      len -= 2;
    }

  if (len == 8 || len == 16 || len == 40)
    {
      for (i = 0; i < len; i++)
        {
          if (!isxdigit (buf[i]))
            break;
        }
      if (i == len)
        {
          switch (len)
            {
            case 8: type = CDK_DBSEARCH_SHORT_KEYID; break;
            case 16: type = CDK_DBSEARCH_KEYID; break;
            case 40: type = CDK_DBSEARCH_FPR; break;
            }
        }
    }
  if (!type)
    type = CDK_DBSEARCH_SUBSTR;
  return type;
}


static void
keyid_from_str (const char * buf, u32 * keyid)
{
  if (!strncmp (buf, "0x", 2))
    buf += 2;

  if (strlen (buf) == 8)
    {
      keyid[0] = 0;
      keyid[1] = strtoul (buf, NULL, 16);
    }
  else if (strlen (buf) == 16)
    {
      keyid[0] = strtoul (buf, NULL, 16);
      keyid[1] = strtoul (buf + 8, NULL, 16);
    }
}


static void
fpr_from_str (const char * buf, byte * fpr)
{
  char tmp[3];
  int i;
  
  if (strlen (buf) != 40)
    memset (fpr, 0, 20);
  else
    {
      for (i = 0; i < 20; i++)
        {
          tmp[0] = buf[2*i];
          tmp[1] = buf[2*i+1];
          tmp[2] = '\0';
          fpr[i] = strtoul (tmp, NULL, 16);
        }
    }
}
      

int
cdk_keydb_search_auto (CDK_KEYDB_HD hd, const char *desc, CDK_KBNODE * r_knode)
{
  u32 keyid[2];
  byte fpr[20];
  int rc, type;
  
  type = classify_data (desc, strlen (desc));
  switch (type)
    {
    case CDK_DBSEARCH_EXACT:
    case CDK_DBSEARCH_SUBSTR:
      rc = cdk_keydb_get_bypattern (hd, desc, r_knode);
      break;
      
    case CDK_DBSEARCH_KEYID:
    case CDK_DBSEARCH_SHORT_KEYID:
      keyid_from_str (desc, keyid);
      rc = cdk_keydb_get_bykeyid (hd, keyid, r_knode);
      break;

    case CDK_DBSEARCH_FPR:
      fpr_from_str (desc, fpr);
      rc = cdk_keydb_get_byfpr (hd, fpr, r_knode);
      break;

    default:
      rc = CDK_Inv_Mode;
      break;
    }
  return rc;
}


int
cdk_keydb_export (CDK_KEYDB_HD hd, CDK_STREAM out, CDK_STRLIST remusr)
{
  CDK_KBNODE knode, node;
  CDK_STRLIST r;
  int old_ctb = 0;
  int rc = 0;

  for (r = remusr; r; r = r->next)
    {
      rc = cdk_keydb_search_auto (hd, r->d, &knode);
      if (rc)
	break;
      for (node = knode; node; node = node->next)
	{
	  /* those packets are not intended for the real wolrd */
	  if (node->pkt->pkttype == CDK_PKT_RING_TRUST)
	    continue;
	  /* we never export local signed signatures */
	  if (node->pkt->pkttype == CDK_PKT_SIGNATURE &&
	      !node->pkt->pkt.signature->flags.exportable)
            continue;
          if (node->pkt->pkttype == CDK_PKT_PUBLIC_KEY
              && node->pkt->pkt.public_key->version == 3)
            old_ctb = 1;
          node->pkt->old_ctb = old_ctb;
	  rc = cdk_pkt_build (out, node->pkt);
	  if (rc)
	    break;
	}
      cdk_kbnode_release (knode);
      knode = NULL;
    }

  return rc;
}


static CDK_PACKET *
find_key_packet (CDK_KBNODE knode, int * r_secret)
{
  CDK_PACKET * pkt;

  pkt = cdk_kbnode_find_packet (knode, CDK_PKT_PUBLIC_KEY);
  if (!pkt)
    {
      pkt = cdk_kbnode_find_packet (knode, CDK_PKT_SECRET_KEY);
      *r_secret = pkt? 1 : 0;
    }
  return pkt;
}


int
cdk_keydb_import (CDK_KEYDB_HD hd, CDK_KBNODE knode, int result[4])
{
  CDK_KBNODE node, chk = NULL;
  CDK_PACKET * pkt;
  CDK_STREAM out;
  u32 keyid[2];
  int rc = 0, is_sk = 0;

  if (!hd || !knode)
    return CDK_Inv_Value;
  
  memset (result, 0, 4 * sizeof (int));
  pkt = find_key_packet (knode, &is_sk);
  if (!pkt)
      return CDK_Inv_Packet;
  result[is_sk] = 1;
  if (is_sk)
    cdk_sk_get_keyid (pkt->pkt.secret_key, keyid);
  else
    cdk_pk_get_keyid (pkt->pkt.public_key, keyid);
  cdk_keydb_get_bykeyid (hd, keyid, &chk);
  if (chk)
    {
      /* fixme: search for new signatures */
      cdk_kbnode_release (chk);
      return 0;
    }
  
  if (hd->buf)
    return CDK_Inv_Mode;
  rc = _cdk_stream_append (hd->name, &out);
  if (rc)
    return rc;
  
  for (node = knode; node; node = node->next)
    {
      if (node->pkt->pkttype == CDK_PKT_RING_TRUST)
        continue; /* No uniformed syntax for this packet */
      rc = cdk_pkt_build (out, node->pkt);
      if (rc)
        break;
    }
  if (!rc)
    result[is_sk? 3 : 2] = 1;
  cdk_stream_close (out);
  if (!hd->no_cache)
    keydb_idx_rebuild (hd);  
  return rc;
}  


/* internal functions */
CDK_KBNODE
_cdk_keydb_get_pkblock (CDK_KEYDB_HD hd, u32 * keyid)
{
  CDK_KBNODE kbnode = NULL;
  int rc;

  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &kbnode);
      if (rc)
        kbnode = NULL;
    }
  return kbnode;
}


CDK_KBNODE
_cdk_keydb_get_skblock (CDK_KEYDB_HD hd, u32 * keyid)
{
  CDK_KBNODE snode = NULL;
  int rc;

  if (hd)
    {
      rc = cdk_keydb_get_bykeyid (hd, keyid, &snode);
      if (rc)
        snode = NULL;
    }
  return snode;
}


