/* keygen.c - OpenPGP key generation
 *        Copyright (C) 2002 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif
#include <stdio.h>

#include "opencdk.h"
#include "packet.h"
#include "main.h"
#include "types.h"

struct one_key_s
{
  u32 expire_date;
  int algo;
  int len;
  GCRY_MPI resarr[6];
  size_t n;
  cdkPKT_public_key * pk;
  cdkPKT_secret_key * sk;
};
  
struct cdk_keygen_s
{
  struct one_key_s prim;
  char * user_id;
  cdkPKT_user_id * id;
  byte * sym_prefs;
  size_t sym_len;
  byte * hash_prefs;
  size_t hash_len;
  byte * zip_prefs;
  size_t zip_len;
  unsigned mdc_feature:1;
  struct one_key_s sub;
};


/* default preferences */
static byte def_sym_prefs[] = {GCRY_CIPHER_AES128, GCRY_CIPHER_CAST5,
                               GCRY_CIPHER_TWOFISH, GCRY_CIPHER_AES192,
                               GCRY_CIPHER_AES256, GCRY_CIPHER_3DES,
                               GCRY_CIPHER_BLOWFISH};
static byte def_hash_prefs[] = {GCRY_MD_SHA1, GCRY_MD_RMD160, GCRY_MD_MD5};
static byte def_zip_prefs[] = {CDK_COMPRESS_ZIP, CDK_COMPRESS_ZLIB};


/**
 * cdk_keygen_set_prefs: Set the preferences for the userID
 * @hd: the keygen object
 * @hd: the preference type
 * @array: one-octet array with algorithm numers
 *
 **/
int
cdk_keygen_set_prefs (CDK_KEYGEN_CTX hd, int type, byte * array, size_t n)
{
  
  if (!hd)
    return CDK_Inv_Value;

  switch (type)
    {
    case CDK_PREFTYPE_SYM:
      hd->sym_len = array? n : DIM (def_sym_prefs);
      hd->sym_prefs = cdk_calloc (1, hd->sym_len);
      memcpy (hd->sym_prefs, array? array : def_sym_prefs, hd->sym_len);
      break;
      
    case CDK_PREFTYPE_HASH:
      hd->hash_len = array? n : DIM (def_hash_prefs);
      hd->hash_prefs = cdk_calloc (1, hd->hash_len);
      memcpy (hd->hash_prefs, array? array : def_hash_prefs, hd->hash_len);
      break;
      
    case CDK_PREFTYPE_ZIP:
      hd->zip_len = array? n : DIM (def_zip_prefs);
      hd->zip_prefs = cdk_calloc (1, hd->zip_len);
      memcpy (hd->zip_prefs, array? array : def_zip_prefs, hd->zip_len);
      break;
      
    default:
      return CDK_Inv_Mode;
    }
  
  return 0;
}


/**
 * cdk_keygen_set_name: set the userid name for the key
 * @hd: the keygen object
 * @name: name
 *
 * The name will be encoded in UTF8 to avoid problems.
 **/
void
cdk_keygen_set_name (CDK_KEYGEN_CTX hd, const char * name)
{
  if (hd)
    {
      cdk_free (hd->user_id);
      hd->user_id = cdk_utf8_encode (name);
    }
}


/**
 * cdk_keygen_set_algo_info: set the length and type of the key
 * @hd: the keygen object.
 * @type: key type (primary=0, subkey=1)
 * @algo: algorithm compliant with rfc2440
 * @bits: lengt of the key in bits
 *
 **/
int
cdk_keygen_set_algo_info (CDK_KEYGEN_CTX hd, int type, int algo, int bits)
{
  int rc;
  
  if (!hd)
    return CDK_Inv_Value;

  if (bits % 128 != 0)
    bits = bits + (bits % 128);
  
  rc = _cdk_pk_test_algo (algo, type? GCRY_PK_USAGE_ENCR : GCRY_PK_USAGE_SIGN);
  if (rc)
    return rc;

  if (!type) /* primary */
    {
      hd->prim.algo = algo;
      hd->prim.len = bits;
    }
  else
    {
      hd->sub.algo = algo;
      hd->sub.len = bits;
    }

  /* sanity checks for minimal and maxmimal key length. */
  if (hd->prim.algo == GCRY_PK_DSA && hd->prim.len > 1024)
    hd->prim.len = 1024;
  if (hd->prim.len < 768)
    hd->prim.len = 768;
  if (hd->sub.len < 768)
    hd->sub.len = 768;
  if (hd->prim.len > 4096)
    hd->prim.len = 4096;
  if (hd->sub.len > 4096)
    hd->prim.len = 4096;

  return 0;
}


/**
 * cdk_keygen_set_mdc_feature: set the mdc feature for the key
 * @hd: keygen object
 * @val: boolean (yes=1, no=0)
 *
 * if you want a RFC2440 compliant key, you've to disable this feature
 * until the rfc2440-bis6 becomes the next standard.
 **/
void
cdk_keygen_set_mdc_feature (CDK_KEYGEN_CTX hd, int val)
{
  if (hd)
    hd->mdc_feature = val;
}


/**
 * cdk_keygen_set_expire_date: set the expire date of the primary key
 * @hd: keygen object
 * @type: key type (0=primary, 1=seconardy)
 * @timestamp: the date the key should expire
 *
 **/
void
cdk_keygen_set_expire_date (CDK_KEYGEN_CTX hd, int type, long timestamp)
{
  if (hd)
    {
      if (!type)
        hd->prim.expire_date = timestamp;
      else
        hd->sub.expire_date = timestamp;
    }
}


static int
read_single_mpi (GCRY_SEXP s_key, const char * val, GCRY_MPI * r_resarr)
{
  GCRY_SEXP list;

  if (!r_resarr)
    return CDK_Inv_Value;
  list = gcry_sexp_find_token (s_key, val, 0);
  if (list)
    *r_resarr = gcry_sexp_nth_mpi (list, 1, 0);
  gcry_sexp_release (list);
  return list? 0 : CDK_Gcry_Error;
}

  
static int
read_dsa_key (GCRY_SEXP s_key, GCRY_MPI * resarr)
{
  int rc = read_single_mpi (s_key, "p", &resarr[0]);
  if (!rc)
    rc = read_single_mpi (s_key, "q", &resarr[1]);
  if (!rc)
    rc = read_single_mpi (s_key, "g", &resarr[2]);
  if (!rc)
    rc = read_single_mpi (s_key, "y", &resarr[3]);
  if (!rc)
    rc = read_single_mpi (s_key, "x", &resarr[4]);
  return rc;
}


static int
read_elg_key (GCRY_SEXP s_key, GCRY_MPI * resarr)
{
  int rc = read_single_mpi (s_key, "p", &resarr[0]);
  if (!rc)
    rc = read_single_mpi (s_key, "g", &resarr[1]);
  if (!rc)
    rc = read_single_mpi (s_key, "y", &resarr[2]);
  if (!rc)
    rc = read_single_mpi (s_key, "x", &resarr[3]);
  return rc;  
}


static int
read_rsa_key (GCRY_SEXP s_key, GCRY_MPI * resarr)
{
  int rc = read_single_mpi (s_key, "n", &resarr[0]);
  if (!rc)
    rc =read_single_mpi (s_key, "e", &resarr[1]);
  if (!rc)
    rc = read_single_mpi (s_key, "d", &resarr[2]);
  if (!rc)
    rc = read_single_mpi (s_key, "p", &resarr[3]);
  if (!rc)
    rc = read_single_mpi (s_key, "q", &resarr[4]);
  if (!rc)
    rc = read_single_mpi (s_key, "u", &resarr[5]);
  return rc;
}
  

static int
generate_subkey (CDK_KEYGEN_CTX hd)
{
  GCRY_SEXP s_params = NULL, s_key;
  size_t n = hd->sub.len;
  int rc;

  if (is_DSA (hd->sub.algo))
    rc = gcry_sexp_build (&s_params, NULL, "(genkey(dsa(nbits %d)))", n);
  else if (is_ELG (hd->sub.algo))
    rc = gcry_sexp_build (&s_params, NULL, "(genkey(elg(nbits %d)))", n);
  else if (is_RSA (hd->sub.algo))
    rc = gcry_sexp_build (&s_params, NULL, "(genkey(rsa(nbits %d)))", n);
  else
      rc = CDK_Inv_Algo;
  if (!rc)
    rc = gcry_pk_genkey (&s_key, s_params);
  gcry_sexp_release (s_params);
  if (!rc)
    {
      if (is_DSA (hd->sub.algo))
        rc = read_dsa_key (s_key, hd->sub.resarr);
      else if (is_ELG (hd->sub.algo))
        rc = read_elg_key (s_key, hd->sub.resarr);
      else if (is_RSA (hd->sub.algo))
        rc = read_rsa_key (s_key, hd->sub.resarr);
    }
  gcry_sexp_release (s_key);
  return rc;
}
  

/**
 * cdk_keygen_start: kick off the key generation
 * @hd: the keygen object
 *
 **/
int
cdk_keygen_start (CDK_KEYGEN_CTX hd)
{
  GCRY_SEXP s_params = NULL, s_key = NULL;
  size_t n = hd->prim.len;
  int rc = 0;
  
  if (!hd || !hd->user_id)
    return CDK_Inv_Value;
  if (is_ELG (hd->prim.algo))
    return CDK_Inv_Mode;
  if (!hd->prim.len)
    hd->prim.len = 1024;

  if (!hd->sym_prefs)
    cdk_keygen_set_prefs (hd, CDK_PREFTYPE_SYM, NULL, 0);
  if (!hd->hash_prefs)
    cdk_keygen_set_prefs (hd, CDK_PREFTYPE_HASH, NULL, 0);
  if (!hd->zip_prefs)
    cdk_keygen_set_prefs (hd, CDK_PREFTYPE_ZIP, NULL, 0);

  if (is_DSA (hd->prim.algo))
    rc = gcry_sexp_build (&s_params, NULL, "(genkey(dsa(nbits %d)))", n);
  else if (is_RSA (hd->prim.algo))
    rc = gcry_sexp_build (&s_params, NULL, "(genkey(rsa(nbits %d)))", n);
  else
    rc = CDK_Inv_Algo;
  if (!rc)
    rc = gcry_pk_genkey (&s_key, s_params);
  gcry_sexp_release (s_params);
  if (!rc)
    {
      if (is_DSA (hd->prim.algo))
        rc = read_dsa_key (s_key, hd->prim.resarr);
      else if (is_RSA (hd->prim.algo))
        rc = read_rsa_key (s_key, hd->prim.resarr);
      hd->prim.n = cdk_pk_get_npkey (hd->prim.algo);
    }
  gcry_sexp_release (s_key);

  if (!rc)
    {
      if (hd->sub.algo && hd->sub.len)
        rc = generate_subkey (hd);
      hd->sub.n = cdk_pk_get_npkey (hd->sub.algo);
    }
  return rc;
}


static int
gcry_mpi_to_native (CDK_KEYGEN_CTX hd, size_t nkey, int type,
                    cdkPKT_public_key * pk, cdkPKT_secret_key * sk)
{
  GCRY_MPI * resarr;
  CDK_MPI a = NULL;
  size_t nbytes;
  int i = 0, j = 0, nbits;
  int rc = 0;

  if (!pk && !sk)
    return CDK_Inv_Value;

  resarr = type? hd->sub.resarr : hd->prim.resarr;
  if (sk)
    i += cdk_pk_get_npkey (sk->pubkey_algo);
  while (j != nkey)
    {
      nbits = gcry_mpi_get_nbits (resarr[i]);
      if (pk)
        a = cdk_calloc (1, sizeof * a + (nbits + 7) / 8 + 2 + 1);
      else if (sk)
        a = cdk_scalloc (sizeof * a + (nbits + 7) / 8 + 2 + 1);
      a->bits = nbits;
      a->bytes = (nbits + 7) / 8;
      nbytes = a->bytes;
      a->data[0] = nbits >> 8;
      a->data[1] = nbits;
      rc = gcry_mpi_print (GCRYMPI_FMT_USG, a->data + 2, &nbytes, resarr[i++]);
      if (rc)
        break;
      if (pk)
        pk->mpi[j++] = a;
      else if (sk)
        sk->mpi[j++] = a;
    }
  return rc;
}

  
static cdkPKT_public_key *
pk_create (CDK_KEYGEN_CTX hd, int type)
{
  cdkPKT_public_key * pk;
  int rc = 0;

  pk = cdk_calloc (1, sizeof * pk);
  if (!pk)
    return NULL;
  pk->version = 4;
  pk->pubkey_algo = type? hd->sub.algo : hd->prim.algo;
  pk->timestamp = _cdk_timestamp ();
  if (hd->prim.expire_date)
    pk->expiredate = (hd->prim.expire_date - pk->timestamp) / 86400;
  rc = gcry_mpi_to_native (hd, type? hd->sub.n: hd->prim.n, type, pk, NULL);
  if (rc)
    {
      cdk_free (pk);
      pk = NULL;
    }
  return pk;
}


cdkPKT_secret_key *
sk_create (CDK_KEYGEN_CTX hd, int type, cdkPKT_public_key * pk)
{
  cdkPKT_secret_key * sk;
  int nskey, rc = 0;

  sk = cdk_calloc (1, sizeof * sk);
  if (!sk)
    return NULL;
  _cdk_copy_pubkey (&sk->pk, pk);
  sk->version = 4;
  sk->pubkey_algo = type? hd->sub.algo : hd->prim.algo;
  sk->csum = 0;
  sk->is_protected = 0;
  nskey = cdk_pk_get_nskey (sk->pubkey_algo);
  rc = gcry_mpi_to_native (hd, nskey, type, NULL, sk);
  if (rc)
    {
      cdk_free (sk);
      sk = NULL;
    }
  return sk;
}


static cdkPKT_user_id *
uid_create (CDK_KEYGEN_CTX hd)
{
  cdkPKT_user_id * id;

  if (!hd->user_id)
    return NULL;
  id = cdk_calloc (1, sizeof * id + strlen (hd->user_id) + 1);
  if (!id)
    return NULL;
  strcpy (id->name, hd->user_id);
  id->len = strlen (hd->user_id);
  return id;
}


static cdkPKT_signature *
sig_subkey_create (CDK_KEYGEN_CTX hd,
                   cdkPKT_public_key * prim_pk,
                   cdkPKT_public_key * sub_pk,
                   cdkPKT_secret_key * prim_sk)
{
  GCRY_MD_HD md;
  cdkPKT_signature * sig;
  int rc;
  
  sig = cdk_calloc (1, sizeof * sig);
  if (!sig)
    return NULL;
  _cdk_sig_create (prim_pk, sig);
  sig->sig_class = 0x18;
  md = gcry_md_open (GCRY_MD_SHA1, 0);
  if (!md)
    {
      _cdk_free_signature (sig);
      return NULL;
    }
      
  _cdk_hash_pubkey (prim_pk, md, 0);
  _cdk_hash_pubkey (sub_pk, md, 0);
  rc = _cdk_sig_complete (sig, prim_sk, md);
  gcry_md_close (md);
  if (rc)
    {
      _cdk_free_signature (sig);
      return NULL;
    }
  return sig;
}
      
  
static cdkPKT_signature *
sig_self_create (CDK_KEYGEN_CTX hd,
                 cdkPKT_public_key * pk,
                 cdkPKT_secret_key * sk,
                 cdkPKT_user_id * id)
{
  GCRY_MD_HD md;
  cdkPKT_signature * sig;
  CDK_SUBPKT node;
  u32 keyid[2];
  int rc;

  sig = cdk_calloc (1, sizeof * sig);
  if (!sig)
    return NULL;
  sig->version = 4;
  sig->timestamp = _cdk_timestamp ();
  sig->sig_class = 0x13;
  sig->pubkey_algo = hd->prim.algo;
  sig->digest_algo = GCRY_MD_SHA1;

  sig->hashed = node = cdk_subpkt_new (4);
  if (node)
    {
      node->size = 4;
      node->type = SIGSUBPKT_SIG_CREATED;
      _cdk_u32tobuf (_cdk_timestamp (), node->d);      
    }

  node = cdk_subpkt_new (hd->sym_len + 1);
  if (node)
    {
      node->size = hd->sym_len;
      node->type = SIGSUBPKT_PREFS_SYM;
      memcpy (node->d, hd->sym_prefs, hd->sym_len);
      _cdk_subpkt_add (sig->hashed, node);
    }

  node = cdk_subpkt_new (hd->hash_len + 1);
  if (node)
    {
      node->size = hd->hash_len;
      node->type = SIGSUBPKT_PREFS_HASH;
      memcpy (node->d, hd->hash_prefs, hd->hash_len);
      _cdk_subpkt_add (sig->hashed, node);
    }

  node = cdk_subpkt_new (hd->zip_len + 1);
  if (node)
    {
      node->size = hd->zip_len;
      node->type = SIGSUBPKT_PREFS_ZIP;
      memcpy (node->d, hd->zip_prefs, hd->zip_len);
      _cdk_subpkt_add (sig->hashed, node);
    }

  if (hd->mdc_feature)
    {
      node = cdk_subpkt_new (1);
      if (node)
        {
          node->size = 1;
          node->type = SIGSUBPKT_FEATURES;
          node->d[0] = 0x01;
          _cdk_subpkt_add (sig->hashed, node);
        }
    }
  
  if (pk->expiredate)
    {
      node = cdk_subpkt_new (4);
      if (node)
        {
          node->size = 4;
          node->type = SIGSUBPKT_KEY_EXPIRE;
          _cdk_u32tobuf ((hd->prim.expire_date - pk->timestamp), node->d);
          _cdk_subpkt_add (sig->hashed, node);
        }
    }

  sig->unhashed = node = cdk_subpkt_new (8);
  if (node)
    {
      node->size = 8;
      node->type = SIGSUBPKT_ISSUER;
      cdk_pk_get_keyid (pk, keyid);
      _cdk_u32tobuf (keyid[0], node->d);
      _cdk_u32tobuf (keyid[1], node->d + 4);
    }

  md = gcry_md_open (sig->digest_algo, 0);
  if (!md)
    {
      _cdk_free_signature (sig);
      return NULL;
    }

  _cdk_hash_pubkey (pk, md, 0);
  _cdk_hash_userid (id, sig->version, md);
  rc = _cdk_sig_complete (sig, sk, md);
  gcry_md_close (md);
  if (rc)
    {
      _cdk_free_signature (sig);
      return NULL;
    }  
  return sig;
}


/**
 * cdk_keygen_save: save the generated keys to disk
 * @hd: the keygen object
 * @pub: name of the file to store the public key
 * @sec: name of the file to store the secret key
 *
 * hint: the secret is plaintext and not protected. this is subject
 * to change!
 **/
int
cdk_keygen_save (CDK_KEYGEN_CTX hd, const char * pub, const char * sec)
{
  CDK_STREAM out;
  CDK_PACKET pkt;
  int rc;

  hd->prim.pk = pk_create (hd, 0);
  if (!hd->prim.pk)
    return CDK_Inv_Packet;
  hd->prim.sk = sk_create (hd, 0, hd->prim.pk);
  if (!hd->prim.sk)
    return CDK_Inv_Packet;
  hd->id = uid_create (hd);
  if (!hd->id)
    return CDK_Inv_Packet;

  rc = cdk_stream_new (pub, &out);
  if (rc)
    return rc;
  
  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_PUBLIC_KEY;
  pkt.pkt.public_key = hd->prim.pk;
  rc = cdk_pkt_build (out, &pkt);
  if (rc)
    goto fail;
  
  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_USER_ID;
  pkt.pkt.user_id = hd->id;
  rc = cdk_pkt_build (out, &pkt);
  if (rc)
    goto fail;

  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_SIGNATURE;
  pkt.pkt.signature = sig_self_create (hd, hd->prim.pk, hd->prim.sk, hd->id);
  rc = cdk_pkt_build (out, &pkt);
  cdk_pkt_free (&pkt);
  if (rc)
    goto fail;

  if (hd->sub.algo)
    {
      init_packet (&pkt);
      pkt.pkttype = CDK_PKT_PUBLIC_SUBKEY;
      pkt.pkt.public_key = hd->sub.pk = pk_create (hd, 1);
      rc = cdk_pkt_build (out, &pkt);
      if (rc)
        goto fail;

      init_packet (&pkt);
      pkt.pkttype = CDK_PKT_SIGNATURE;
      pkt.pkt.signature = sig_subkey_create (hd, hd->prim.pk, hd->sub.pk,
                                             hd->prim.sk);
      rc = cdk_pkt_build (out, &pkt);
      cdk_pkt_free (&pkt);
      if (rc)
        goto fail;
    }
  
  cdk_stream_close (out);

  rc = cdk_stream_new (sec, &out);
  if (rc)
    goto fail;

  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_SECRET_KEY;
  pkt.pkt.secret_key = hd->prim.sk;
  rc = cdk_pkt_build (out, &pkt);
  if (rc)
    goto fail;

  init_packet (&pkt);
  pkt.pkttype = CDK_PKT_USER_ID;
  pkt.pkt.user_id = hd->id;
  rc = cdk_pkt_build (out, &pkt);

  if (hd->sub.algo)
    {
      init_packet (&pkt);
      pkt.pkttype = CDK_PKT_SECRET_SUBKEY;
      pkt.pkt.secret_key = hd->sub.sk = sk_create (hd, 1, hd->sub.pk);
      rc = cdk_pkt_build (out, &pkt);
      if (rc)
        goto fail;
    }  

  cdk_stream_close (out);

 fail:
  return rc;
}


/**
 * cdk_keygen_free: free the keygen object
 * @hd: the keygen object
 *
 **/
void
cdk_keygen_free (CDK_KEYGEN_CTX hd)
{
  if (hd)
    {
      _cdk_free_pubkey (hd->prim.pk);
      _cdk_free_pubkey (hd->sub.pk);
      _cdk_free_seckey (hd->prim.sk);
      _cdk_free_seckey (hd->sub.sk);
      _cdk_free_userid (hd->id);
      cdk_free (hd->sym_prefs);
      cdk_free (hd->hash_prefs);
      cdk_free (hd->zip_prefs);
      _cdk_free_mpi_array (hd->prim.n, hd->prim.resarr);
      _cdk_free_mpi_array (hd->sub.n, hd->sub.resarr);
      cdk_free (hd);
    }
}


/**
 * cdk_keygen_new:
 * @r_hd: the new object
 *
 **/
int
cdk_keygen_new (CDK_KEYGEN_CTX * r_hd)
{
  CDK_KEYGEN_CTX hd;

  hd = cdk_calloc (1, sizeof * hd);
  if (!hd)
    return CDK_Out_Of_Core;
  if (r_hd)
    *r_hd = hd;
  return 0;
}

      
