
/* This is an independent implementation of the encryption algorithm:   */
/*                                                                      */
/*         RIJNDAEL by Joan Daemen and Vincent Rijmen                   */
/*                                                                      */
/* which is a candidate algorithm in the Advanced Encryption Standard   */
/* programme of the US National Institute of Standards and Technology.  */
/*                                                                      */
/* Copyright in this implementation is held by Dr B R Gladman but I     */
/* hereby give permission for its free direct or derivative use subject */
/* to acknowledgment of its origin and compliance with any conditions   */
/* that the originators of the algorithm place on its exploitation.     */
/*                                                                      */
/* Dr Brian Gladman (gladman@seven77.demon.co.uk) 14th January 1999     */


/*
Algorithm rijndael (rijndael.c)

128 bit key:
Key Setup:    217/1297 cycles (encrypt/decrypt)
Encrypt:       363 cycles =    70.5 mbits/sec
Decrypt:       363 cycles =    70.5 mbits/sec
Mean:          363 cycles =    70.5 mbits/sec

192 bit key:
Key Setup:    214/1527 cycles (encrypt/decrypt)
Encrypt:       435 cycles =    58.9 mbits/sec
Decrypt:       430 cycles =    59.5 mbits/sec
Mean:          433 cycles =    59.2 mbits/sec

256 bit key:
Key Setup:    286/1841 cycles (encrypt/decrypt)
Encrypt:       489 cycles =    52.4 mbits/sec
Decrypt:       503 cycles =    50.9 mbits/sec
Mean:          496 cycles =    51.6 mbits/sec
*/

#include "rijndael.h"

u1byte  rij_pow_tab[256];
u1byte  rij_log_tab[256];
u1byte  rij_sbx_tab[256];
u1byte  rij_isb_tab[256];
u4byte  rij_rco_tab[ 10];
u4byte  rij_ft_tab[4][256];
u4byte  rij_it_tab[4][256];

#ifdef  RIJNDAEL_LARGE_TABLES
  u4byte  rij_fl_tab[4][256];
  u4byte  rij_il_tab[4][256];
#endif

#define ff_mult(a,b)    (a && b ? rij_pow_tab[(rij_log_tab[a] + rij_log_tab[b]) % 255] : 0)

#define f_rn(bo, bi, n, k)                          \
    bo[n] =  rij_ft_tab[0][byte(bi[n],0)] ^             \
             rij_ft_tab[1][byte(bi[(n + 1) & 3],1)] ^   \
             rij_ft_tab[2][byte(bi[(n + 2) & 3],2)] ^   \
             rij_ft_tab[3][byte(bi[(n + 3) & 3],3)] ^ *(k + n)

#define i_rn(bo, bi, n, k)                          \
    bo[n] =  rij_it_tab[0][byte(bi[n],0)] ^             \
             rij_it_tab[1][byte(bi[(n + 3) & 3],1)] ^   \
             rij_it_tab[2][byte(bi[(n + 2) & 3],2)] ^   \
             rij_it_tab[3][byte(bi[(n + 1) & 3],3)] ^ *(k + n)

#ifdef RIJNDAEL_LARGE_TABLES

#define ls_box(x)                \
    ( rij_fl_tab[0][byte(x, 0)] ^    \
      rij_fl_tab[1][byte(x, 1)] ^    \
      rij_fl_tab[2][byte(x, 2)] ^    \
      rij_fl_tab[3][byte(x, 3)] )

#define f_rl(bo, bi, n, k)                          \
    bo[n] =  rij_fl_tab[0][byte(bi[n],0)] ^             \
             rij_fl_tab[1][byte(bi[(n + 1) & 3],1)] ^   \
             rij_fl_tab[2][byte(bi[(n + 2) & 3],2)] ^   \
             rij_fl_tab[3][byte(bi[(n + 3) & 3],3)] ^ *(k + n)

#define i_rl(bo, bi, n, k)                          \
    bo[n] =  rij_il_tab[0][byte(bi[n],0)] ^             \
             rij_il_tab[1][byte(bi[(n + 3) & 3],1)] ^   \
             rij_il_tab[2][byte(bi[(n + 2) & 3],2)] ^   \
             rij_il_tab[3][byte(bi[(n + 1) & 3],3)] ^ *(k + n)

#else

#define ls_box(x)                            \
    ((u4byte)rij_sbx_tab[byte(x, 0)] <<  0) ^    \
    ((u4byte)rij_sbx_tab[byte(x, 1)] <<  8) ^    \
    ((u4byte)rij_sbx_tab[byte(x, 2)] << 16) ^    \
    ((u4byte)rij_sbx_tab[byte(x, 3)] << 24)

#define f_rl(bo, bi, n, k)                                      \
    bo[n] = (u4byte)rij_sbx_tab[byte(bi[n],0)] ^                    \
        rotl(((u4byte)rij_sbx_tab[byte(bi[(n + 1) & 3],1)]),  8) ^  \
        rotl(((u4byte)rij_sbx_tab[byte(bi[(n + 2) & 3],2)]), 16) ^  \
        rotl(((u4byte)rij_sbx_tab[byte(bi[(n + 3) & 3],3)]), 24) ^ *(k + n)

#define i_rl(bo, bi, n, k)                                      \
    bo[n] = (u4byte)rij_isb_tab[byte(bi[n],0)] ^                    \
        rotl(((u4byte)rij_isb_tab[byte(bi[(n + 3) & 3],1)]),  8) ^  \
        rotl(((u4byte)rij_isb_tab[byte(bi[(n + 2) & 3],2)]), 16) ^  \
        rotl(((u4byte)rij_isb_tab[byte(bi[(n + 1) & 3],3)]), 24) ^ *(k + n)

#endif

void rijndael_gen_tabs(void)
{   u4byte  i, t;
    u1byte  p, q;

    /* log and power tables for GF(2**8) finite field with  */
    /* 0x011b as modular polynomial - the simplest prmitive */
    /* root is 0x03, used here to generate the tables       */

    for(i = 0,p = 1; i < 256; ++i)
    {
        rij_pow_tab[i] = (u1byte)p; rij_log_tab[p] = (u1byte)i;

        p = p ^ (p << 1) ^ (p & 0x80 ? 0x01b : 0);
    }

    rij_log_tab[1] = 0; p = 1;

    for(i = 0; i < 10; ++i)
    {
        rij_rco_tab[i] = p; 

        p = (p << 1) ^ (p & 0x80 ? 0x1b : 0);
    }

    /* note that the affine byte transformation matrix in   */
    /* rijndael specification is in big endian format with  */
    /* bit 0 as the most significant bit. In the remainder  */
    /* of the specification the bits are numbered from the  */
    /* least significant end of a byte.                     */

    for(i = 0; i < 256; ++i)
    {   
        p = (i ? rij_pow_tab[255 - rij_log_tab[i]] : 0); q = p; 
        q = (q >> 7) | (q << 1); p ^= q; 
        q = (q >> 7) | (q << 1); p ^= q; 
        q = (q >> 7) | (q << 1); p ^= q; 
        q = (q >> 7) | (q << 1); p ^= q ^ 0x63; 
        rij_sbx_tab[i] = p; rij_isb_tab[p] = (u1byte)i;
    }

    for(i = 0; i < 256; ++i)
    {
        p = rij_sbx_tab[i]; 

#ifdef  RIJNDAEL_LARGE_TABLES        
        
        t = p; rij_fl_tab[0][i] = t;
        rij_fl_tab[1][i] = rotl(t,  8);
        rij_fl_tab[2][i] = rotl(t, 16);
        rij_fl_tab[3][i] = rotl(t, 24);
#endif
        t = ((u4byte)ff_mult(2, p)) |
            ((u4byte)p <<  8) |
            ((u4byte)p << 16) |
            ((u4byte)ff_mult(3, p) << 24);
        
        rij_ft_tab[0][i] = t;
        rij_ft_tab[1][i] = rotl(t,  8);
        rij_ft_tab[2][i] = rotl(t, 16);
        rij_ft_tab[3][i] = rotl(t, 24);

        p = rij_isb_tab[i]; 

#ifdef  RIJNDAEL_LARGE_TABLES        
        
        t = p; rij_il_tab[0][i] = t; 
        rij_il_tab[1][i] = rotl(t,  8); 
        rij_il_tab[2][i] = rotl(t, 16); 
        rij_il_tab[3][i] = rotl(t, 24);
#endif 
        t = ((u4byte)ff_mult(14, p)) |
            ((u4byte)ff_mult( 9, p) <<  8) |
            ((u4byte)ff_mult(13, p) << 16) |
            ((u4byte)ff_mult(11, p) << 24);
        
        rij_it_tab[0][i] = t; 
        rij_it_tab[1][i] = rotl(t,  8); 
        rij_it_tab[2][i] = rotl(t, 16); 
        rij_it_tab[3][i] = rotl(t, 24); 
    }

}

#define star_x(x) (((x) & 0x7f7f7f7f) << 1) ^ ((((x) & 0x80808080) >> 7) * 0x1b)

#define imix_col(y,x)       \
    u   = star_x(x);        \
    v   = star_x(u);        \
    w   = star_x(v);        \
    t   = w ^ (x);          \
   (y)  = u ^ v ^ w;        \
   (y) ^= rotr(u ^ t,  8) ^ \
          rotr(v ^ t, 16) ^ \
          rotr(t,24)

/* initialise the key schedule from the user supplied key   */

#define loop4(i)                                    \
{   t = ls_box(rotr(t,  8)) ^ rij_rco_tab[i];           \
    t ^= key->e_key[4 * i];     key->e_key[4 * i + 4] = t;    \
    t ^= key->e_key[4 * i + 1]; key->e_key[4 * i + 5] = t;    \
    t ^= key->e_key[4 * i + 2]; key->e_key[4 * i + 6] = t;    \
    t ^= key->e_key[4 * i + 3]; key->e_key[4 * i + 7] = t;    \
}

#define loop6(i)                                    \
{   t = ls_box(rotr(t,  8)) ^ rij_rco_tab[i];           \
    t ^= key->e_key[6 * i];     key->e_key[6 * i + 6] = t;    \
    t ^= key->e_key[6 * i + 1]; key->e_key[6 * i + 7] = t;    \
    t ^= key->e_key[6 * i + 2]; key->e_key[6 * i + 8] = t;    \
    t ^= key->e_key[6 * i + 3]; key->e_key[6 * i + 9] = t;    \
    t ^= key->e_key[6 * i + 4]; key->e_key[6 * i + 10] = t;   \
    t ^= key->e_key[6 * i + 5]; key->e_key[6 * i + 11] = t;   \
}

#define loop8(i)                                    \
{   t = ls_box(rotr(t,  8)) ^ rij_rco_tab[i];           \
    t ^= key->e_key[8 * i];     key->e_key[8 * i + 8] = t;    \
    t ^= key->e_key[8 * i + 1]; key->e_key[8 * i + 9] = t;    \
    t ^= key->e_key[8 * i + 2]; key->e_key[8 * i + 10] = t;   \
    t ^= key->e_key[8 * i + 3]; key->e_key[8 * i + 11] = t;   \
    t  = key->e_key[8 * i + 4] ^ ls_box(t);              \
    key->e_key[8 * i + 12] = t;                          \
    t ^= key->e_key[8 * i + 5]; key->e_key[8 * i + 13] = t;   \
    t ^= key->e_key[8 * i + 6]; key->e_key[8 * i + 14] = t;   \
    t ^= key->e_key[8 * i + 7]; key->e_key[8 * i + 15] = t;   \
}

void rijndael_set_key(rijndaelkey *key, const u1byte in_key[], const u4byte key_len)
{   u4byte  i, t, u, v, w;

    key->k_len = (key_len + 31) / 32;

    key->e_key[0] = u4byte_in(in_key     ); 
	key->e_key[1] = u4byte_in(in_key +  4);
    key->e_key[2] = u4byte_in(in_key +  8); 
	key->e_key[3] = u4byte_in(in_key + 12);

    switch(key->k_len)
    {
        case 4: t = key->e_key[3];
                for(i = 0; i < 10; ++i) 
                    loop4(i);
                break;

        case 6: key->e_key[4] = u4byte_in(in_key + 16); t = key->e_key[5] = u4byte_in(in_key + 20);
                for(i = 0; i < 8; ++i) 
                    loop6(i);
                break;

        case 8: key->e_key[4] = u4byte_in(in_key + 16); key->e_key[5] = u4byte_in(in_key + 20);
                key->e_key[6] = u4byte_in(in_key + 24); t = key->e_key[7] = u4byte_in(in_key + 28);
                for(i = 0; i < 7; ++i) 
                    loop8(i);
                break;
    }

    key->d_key[0] = key->e_key[0]; key->d_key[1] = key->e_key[1];
    key->d_key[2] = key->e_key[2]; key->d_key[3] = key->e_key[3];

    for(i = 4; i < 4 * key->k_len + 24; ++i)
    {
        imix_col(key->d_key[i], key->e_key[i]);
    }

    return;
}

/* encrypt a block of text  */

#define f_nround(bo, bi, k) \
    f_rn(bo, bi, 0, k);     \
    f_rn(bo, bi, 1, k);     \
    f_rn(bo, bi, 2, k);     \
    f_rn(bo, bi, 3, k);     \
    k += 4

#define f_lround(bo, bi, k) \
    f_rl(bo, bi, 0, k);     \
    f_rl(bo, bi, 1, k);     \
    f_rl(bo, bi, 2, k);     \
    f_rl(bo, bi, 3, k)

void rijndael_encrypt(const u1byte in_blk[16], u1byte out_blk[16],rijndaelkey *key)
{   u4byte  b0[4], b1[4], *kp;

    b0[0] = u4byte_in(in_blk    ) ^ key->e_key[0]; b0[1] = u4byte_in(in_blk +  4) ^ key->e_key[1];
    b0[2] = u4byte_in(in_blk + 8) ^ key->e_key[2]; b0[3] = u4byte_in(in_blk + 12) ^ key->e_key[3];

    kp = key->e_key + 4;

    if(key->k_len > 6)
    {
        f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    }

    if(key->k_len > 4)
    {
        f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    }

    f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    f_nround(b1, b0, kp); f_nround(b0, b1, kp);
    f_nround(b1, b0, kp); f_lround(b0, b1, kp);

    u4byte_out(out_blk,      b0[0]); u4byte_out(out_blk +  4, b0[1]);
    u4byte_out(out_blk +  8, b0[2]); u4byte_out(out_blk + 12, b0[3]);
}

/* decrypt a block of text  */

#define i_nround(bo, bi, k) \
    i_rn(bo, bi, 0, k);     \
    i_rn(bo, bi, 1, k);     \
    i_rn(bo, bi, 2, k);     \
    i_rn(bo, bi, 3, k);     \
    k -= 4

#define i_lround(bo, bi, k) \
    i_rl(bo, bi, 0, k);     \
    i_rl(bo, bi, 1, k);     \
    i_rl(bo, bi, 2, k);     \
    i_rl(bo, bi, 3, k)

void rijndael_decrypt(const u1byte in_blk[16], u1byte out_blk[16], rijndaelkey *key)
{   u4byte  b0[4], b1[4], *kp;

    b0[0] = u4byte_in(in_blk     ) ^ key->e_key[4 * key->k_len + 24]; 
	b0[1] = u4byte_in(in_blk +  4) ^ key->e_key[4 * key->k_len + 25];
    b0[2] = u4byte_in(in_blk +  8) ^ key->e_key[4 * key->k_len + 26]; 
	b0[3] = u4byte_in(in_blk + 12) ^ key->e_key[4 * key->k_len + 27];

    kp = key->d_key + 4 * (key->k_len + 5);

    if(key->k_len > 6)
    {
        i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    }

    if(key->k_len > 4)
    {
        i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    }

    i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    i_nround(b1, b0, kp); i_nround(b0, b1, kp);
    i_nround(b1, b0, kp); i_lround(b0, b1, kp);

    u4byte_out(out_blk,     b0[0]); u4byte_out(out_blk +  4, b0[1]);
    u4byte_out(out_blk + 8, b0[2]); u4byte_out(out_blk + 12, b0[3]);
}
