/* ipsec_output.c  */
   
/* This file contains the part of the core IPsec engine that encapsulates
   outgoing IP packets */

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/skbuff.h>
#include <linux/in6.h>
#include <asm/uaccess.h>
#include <asm/checksum.h>
#include <linux/netdevice.h>
#include <linux/in.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/if_arp.h>
#include <linux/mroute.h>
#include <linux/random.h>

#include <net/datalink.h>
#include <net/sock.h>
#include <net/ip.h>
#include <net/protocol.h>
#include <linux/udp.h>

#include "sadb.h"
#include "ipsec.h"
#include "transform.h"
#include <syslog.h>

static char addr_s1[18],addr_s2[18];

struct sk_buff *ipsec_output(struct sk_buff *skb, unsigned short next_protocol, unsigned long next_spi)
{
  	struct net_addr ipsec_addr;
  	unsigned long spi;
  	struct sadb_dst_node *d;
	struct ipsec_transform *transform;
	struct iphdr *iph=(struct iphdr *)(skb->data);
	int kick;

#define errout(e) { error = e; printk(KERN_INFO "IPSEC: Outbound: Protocol Error %d\n", e);goto ipsec_out_error;}

#ifdef MODULE
	ipsec_inc_mod_count();
#endif

  	ipsec_addr.addr_family = AF_INET;
  	ipsec_addr.addr_union.ip_addr.s_addr = iph->daddr;
	/* Obtain SADB Information */
	if ((d = (struct sadb_dst_node *)get_sadb_node(ipsec_addr,next_protocol,next_spi,1,1)) == 0)
	{
		if (next_spi){
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
      			printk(KERN_NOTICE "IPSEC: Outbound: No SA found: src %s dst %s protocol %d spi %lx \n", addr_s1,addr_s2,next_protocol, next_spi);
      			goto ipsec_out_error;
		}
			
		/* NON_PROTECTED policy check */
		if (!(ipsec_sys_policy.outbound.prot_flag&NON_PROTECTED)){

			if (iph->protocol == IPPROTO_UDP)
			{
                            if (registered_km_port == 
                                ntohs(UDP_SOURCE_PORT((unsigned char *)iph+(iph->ihl<<2))))
		 	    {
				if (local_addr(((struct rtable*)skb->dst)->u.dst.dev, iph->saddr))
				{
#ifdef MODULE
					ipsec_dec_mod_count();
#endif
					return skb;
				}
			     }
			}
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
 /* Try to kick key management */
#ifdef SI_TESTER_VERBOSE
			printk(KERN_INFO "IPSEC: Trying to negotiate SA using ISAKMP\n");
#endif
    			kick = sadb_kick_keymgmt(iph->daddr,0,0);
#ifdef SI_TESTER_VERBOSE
			switch (kick) 
			{
			case -1:
   				printk(KERN_INFO "IPSEC: ISAKMP busy: cannot negotiate for src %s dst %s\n", addr_s1, addr_s2);
				break;
			case  0:
   				printk(KERN_INFO "IPSEC: sent SADB_ACQUIRE to ISAKMP for src %s dst %s\n", addr_s1, addr_s2);
				break;
			default:
   				printk(KERN_INFO "IPSEC: ISAKMP already negotiating for src %s dst %s\n", addr_s1, addr_s2);
				break;
			}
#endif
      			goto ipsec_out_error;
		}
		else
		{
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return skb;
		}
  	}

	if (iph->protocol == IPPROTO_UDP)
	{
                 if (registered_km_port == 
                       ntohs(UDP_SOURCE_PORT((unsigned char *)iph+(iph->ihl<<2))))
	         {
			if (local_addr(((struct rtable*)skb->dst)->u.dst.dev, iph->saddr))
			{
#ifdef MODULE
				ipsec_dec_mod_count();
#endif
				return skb;
			}
		     }
	}

	/* TBD: Should we check to see if src!=local & tunnel? */

       	spi = d->sa_info.peer_spi;

	if (d->sa_info.flags & IPSEC_CHECK_REPLAY_FLAG)
	{
		if (d->sa_info.lastsn == MAX_RP_SN)
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
      			printk(KERN_NOTICE "IPSEC: Outbound: Sequence Number Wrapped: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
    			sadb_delete(d);
      			goto ipsec_out_error;
		}
		else if (d->sa_info.lastsn >= MAX_RP_SN-RP_SN_WARN)
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
    			if ((d->sa_info.flags & IPSEC_NEG_KM_FLAG) && 
                           (!(d->sa_info.flags & IPSEC_KM_KICKED_FLAG)))
			{
				sadb_kick_keymgmt(iph->daddr,d->sa_info.spi, d->sa_info.peer_spi);
				d->sa_info.flags |= IPSEC_KM_KICKED_FLAG;
			}
      			printk(KERN_NOTICE "IPSEC: Outbound: Sequence Number About to Wrap: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
		}
	}
	switch(d->sa_info.protocol)
	{
	case IPPROTO_ESP:

#ifdef SI_TESTER
       	printk(KERN_NOTICE "IPSEC[%lu]: Outbound: Pre-encapsulation:src %s:dst %s:prot %d:spi %lx:",  si_cnt++,
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif
		if ((transform = ipsec_get_transform(d->sa_info.alg.esp.crypto_alg_id, CIPHER_TRANSFORM))==NULL){
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
			printk(KERN_NOTICE  "IPSEC: Outbound: Invalid encryption alg: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      			goto ipsec_out_error;
		}
		if ((skb = transform->output_handler(skb,d,transform)) == NULL)
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
			printk(KERN_NOTICE  "IPSEC: Outbound: encryption failure: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      			goto ipsec_out_error;
		}

		if (d->sa_info.alg.esp.auth_alg_id != NO_ALG)
		{
			if ((transform = ipsec_get_transform(d->sa_info.alg.esp.auth_alg_id,AUTH_TRANSFORM))==NULL){
				strcpy(addr_s1,ntoa(iph->saddr));
				strcpy(addr_s2,ntoa(iph->daddr));
				printk(KERN_NOTICE  "IPSEC: Outbound: Invalid authentication alg: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      				goto ipsec_out_error;
			}
			if ((skb = transform->output_handler(skb,d,transform)) == NULL)
			{
				strcpy(addr_s1,ntoa(iph->saddr));
				strcpy(addr_s2,ntoa(iph->daddr));
				printk(KERN_NOTICE  "IPSEC: Outbound: authentication failure: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      				goto ipsec_out_error;
			}
		
		}
	break;
	case IPPROTO_AH:
#ifdef SI_TESTER
       	printk(KERN_NOTICE "IPSEC[%lu]: Outbound: Pre-encapsulation:src %s:dst %s:prot %d:spi %lx:",  si_cnt++,
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif
		if ((transform = ipsec_get_transform(d->sa_info.alg.ah.auth_alg_id,AUTH_TRANSFORM))==NULL){
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
			printk(KERN_NOTICE  "IPSEC: Outbound: Authenticator alg failure: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      			goto ipsec_out_error;
		}

		if ((skb = transform->output_handler(skb,d,transform)) == NULL)
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
			printk(KERN_NOTICE  "IPSEC: Outbound: Authenticator failure: src %s dst %s spi %lx\n", addr_s1,addr_s2,spi);
      			goto ipsec_out_error;
		}
	break;
	case PROTO_NULL_SA:
		if (ipsec_sys_policy.outbound.prot_flag&NULL_ASSOCIATION)
		{
#ifdef MODULE
			ipsec_dec_mod_count();
#endif
			return skb;
		}
		else
		{
			strcpy(addr_s1,ntoa(iph->saddr));
			strcpy(addr_s2,ntoa(iph->daddr));
      			printk(KERN_NOTICE "IPSEC: Outbound: NULL SA found but not permited: src %s dst %s spi %lx \n", addr_s1,addr_s2,spi);
      			goto ipsec_out_error;
		}
	break;
	}

	iph = (struct iphdr *)skb->data;
	iph->check = 0;
	iph->check = ip_fast_csum((unsigned char *)iph,iph->ihl);

	if (d->sa_info.lifetime.bytes_remaining > 0)
		d->sa_info.lifetime.bytes_remaining = IPSEC_MAX(0, d->sa_info.lifetime.bytes_remaining-(long)((d->sa_info.protocol == IPPROTO_ESP) ? skb->len-(iph->ihl<<2) : skb->len));

#ifdef SI_TESTER
       	printk(KERN_NOTICE "IPSEC[%lu]: Outbound: Post-encapsulation:src %s:dst %s:prot %d:spi %lx:",  si_cnt++,
 		strcpy(addr_s1,ntoa(iph->saddr)), strcpy(addr_s2,ntoa(iph->daddr)), iph->protocol, spi);
	si_bprintk(skb->data,skb->len);
	printk("\n");
#endif


	skb->nh.iph = iph;
	skb->h.raw = (unsigned char*)iph + (iph->ihl<<2);
#ifdef MODULE
	ipsec_dec_mod_count();
#endif
	if (d->sa_info.next_encap.spi)
	{
		skb = ipsec_output(skb,d->sa_info.next_encap.protocol, d->sa_info.next_encap.spi);	
		return skb;
	}else
	{
		return skb;
	}
ipsec_out_error:

	if (skb) kfree_skb(skb);
#ifdef MODULE
	ipsec_dec_mod_count();
#endif
	return NULL;
}
