/* ipsec_util.c  */

/* This file contains a set of utility routines used throughout the
   core IPsec engine */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/random.h>
#include <net/ip.h>
#include <asm/byteorder.h>
#include <linux/skbuff.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/in6.h>
#include <net/protocol.h>
#include <linux/version.h>
#include "sadb.h"
#include "ipsec.h"

static struct ipsec_control ipsec= {
	NULL,		/* init */
	NULL,		/* err */
	ipsec_input,	/* input */
	ipsec_output,	/* output */
};


#ifdef MODULE
void ipsec_inc_mod_count(void);
void ipsec_dec_mod_count(void);

void ipsec_inc_mod_count(void)
{
	
	MOD_INC_USE_COUNT;
}

void ipsec_dec_mod_count(void)
{
	MOD_DEC_USE_COUNT;
}
#endif

void generate_ivec(i,l)
unsigned char * i;
int l;
{
	get_random_bytes(i,l);
}

#ifdef SI_TESTER
unsigned long si_cnt = 0;

void si_bprintk(unsigned char * b, int len)
{
	int i;
	for (i=0;i<len;i++)
		printk("%02x ",b[i]);
}
#endif

void bprintk(unsigned char * b, int len, char * s)
{
	int i;
        int k=0;
	printk(KERN_INFO "%s:\n",s);
	printk(KERN_INFO "%03x: ",k);
	for (i=0;i<len;i++){
		printk("%02x ",b[i]);
		if (!((i+1)%16)){
			printk("\n");
			printk(KERN_INFO "%03x: ",++k);
		}
	}
	printk("\n");
}


char * ntoa(__u32 addr)
{

static char s[18];
  
   addr = ntohl(addr);
   sprintf(s,"%d.%d.%d.%d",(0xff000000&addr)>>24,
          (0x00ff0000&addr)>>16, (0x0000ff00&addr)>>8,
           0x000000ff&addr);

   return s;
}

int local_addr(struct device *dev, __u32 src)
{
	struct in_device *in_dev = dev->ip_ptr;	
	struct in_ifaddr *ifa;

	for (ifa = (in_dev)->ifa_list; ifa; ifa = ifa->ifa_next)
	{
		if (ifa->ifa_address == src)
			return 1;
	}
	return 0;
}

/* Zeroes out all IP options from a uchar buffer except for
   IPSO, CIPSO, and SATID */

void iph_ah_zerooptions(unsigned char *buff)
{
    struct iphdr *iph = (struct iphdr *)buff;
    register unsigned char *cp;
    int opt, optlen=0, cnt;

    cp = (unsigned char *)(iph + 1);
    cnt = (iph->ihl<<2)-sizeof(struct iphdr);
    for (;cnt>0;cnt -=optlen,cp+=optlen){
        opt = cp[IPOPT_OPTVAL];
        if ((opt == IPOPT_END) || (opt == IPOPT_NOOP))
           optlen = 1;
        else
           optlen = cp[IPOPT_OLEN];

        if (!((opt == IPOPT_IPSO_BSO) || (opt == IPOPT_IPSO_ESO) ||
            (opt == IPOPT_CIPSO) || (opt == IPOPT_SATID) || (opt = IPOPT_RR)))
           memset(cp,0,optlen);

    }
}

void iph_ah_prep(unsigned char *buff)
{
	struct iphdr *iph = (struct iphdr *)buff;
	
/* zero out checksum and TTL */
	iph->tos = 0;
	iph->frag_off = 0;
	iph->check = 0;
	iph->ttl = 0;

	iph_ah_zerooptions(buff);
}


struct sk_buff *ipsec_alloc_skb(struct sk_buff *skb, unsigned short len)
{
	struct sk_buff *tmpskb;

	if (!(tmpskb = alloc_skb(len,GFP_ATOMIC)))return NULL;
	tmpskb->dev = skb->dev;
	tmpskb->pkt_type = skb->pkt_type;
	tmpskb->priority = skb->priority;
	if (skb->sk)
		skb_set_owner_w(tmpskb,skb->sk);
	tmpskb->dst = dst_clone(skb->dst);
	tmpskb->security = skb->security;

	return tmpskb;
}

/* Returns 0 if packet disallowed, 1 if packet permitted */
int check_replay_window(unsigned long seq, struct sadb_dst_node *d) 
{
	unsigned short diff;
	/* first == 0 or wrapped */
	if (seq == 0) return 0; 

	/* new larger sequence number */
	if (seq > d->sa_info.lastsn) {    
		diff = seq - d->sa_info.lastsn;
		/* In window */
		if (diff < REPLAY_WINDOW_SIZE) { 
			/* set bit for this packet */
			d->sa_info.rpwin_bitmap = (d->sa_info.rpwin_bitmap << diff) | 1; 
		/* This packet has a "way larger" */
		} else d->sa_info.rpwin_bitmap = (__u64)1;  
		d->sa_info.lastsn = seq;
		/* larger is good */
		return 1;           
		}
	diff = d->sa_info.lastsn - seq;
	/* too old or wrapped */
	if (diff >= REPLAY_WINDOW_SIZE) return 0; 
	/* this packet already seen */
	if (d->sa_info.rpwin_bitmap & ((__u64)1 << diff)) return 0; 
	/* mark as seen */
	d->sa_info.rpwin_bitmap |= ((__u64)1 << diff);  
 	/* out of order but good */
	return 1;              
}

#ifdef MODULE
int init_module(void) 
#else
void ipsec_init(struct net_proto *pro)
#endif
{
	struct timeval tv;

#ifdef MODULE
	ipsec_inc_mod_count();
#endif
	printk(KERN_INFO "NIST Cerberus IPsec Version %d.%d Initialized\n",
		CERBERUS_VERSION_MAJOR, CERBERUS_VERSION_MINOR);
	do_gettimeofday(&tv);

	sadb_init();
	register_ipsec(&ipsec);
#ifdef MODULE
	ipsec_dec_mod_count();
	return 0;
#endif
}

#ifdef MODULE
void cleanup_module(void) 
{
	unregister_ipsec();

	sadb_close();
	printk(KERN_INFO "NIST Cerberus IPsec removed\n");
}
#endif
