/* sadb.h - Security Association Data Base Managment routines header file */

/* This header file contains the datatypes and definitions required by
   the SADB management routines */

#ifndef _SADB_H_
#define _SADB_H_

#include <linux/types.h>
#include <linux/socket.h>

#define PROTECTED 0
#define NON_PROTECTED 1
#define NULL_ASSOCIATION 2

#define PROTO_NULL_SA 2

#define MAX_KEY_LENGTH 128

#define MAX_SADB_HASH_SIZE 64
#define SADB_HASH_PLEN 16

#define MAX_SADB_HASH6_SIZE 64
#define SADB_HASH6_PLEN 8

/* Policy defines not current in use */
#define MAX_P_CHOICE 4

#define P_AH  0
#define P_AH_MUST  1
#define P_ESP  2
#define P_ESP_MUST  3
#define P_AH_ESP 4
#define P_AH_ESP_MUST 5
#define P_ESP_AH 6
#define P_ESP_AH_MUST 7

#define IPSEC_TUNNEL_FLAG 0x1
#define IPSEC_MANUAL_KM_FLAG 0x2
#define IPSEC_CHECK_REPLAY_FLAG 0x4
/* IPSEC_PARTIAL_FLAG - partial SA entered by SADBM_GETSPI; completed by SADBM_UPDATE    */
#define IPSEC_PARTIAL_FLAG 0x8
/* IPSEC_NEG_KM_FLAG - KM negotiated  */
#define IPSEC_NEG_KM_FLAG 0x10
/* IPSEC_INBOUND_FLAG - inbound SA  */
#define IPSEC_INBOUND_FLAG 0x20
/* IPSEC_OUTBOUND_FLAG - outbound SA  */
#define IPSEC_OUTBOUND_FLAG 0x40
/* IPSEC_LIFETIME_PERMANENT - can lifetime expire */
#define IPSEC_LIFETIME_PERMANENT_FLAG 0x80
/* IPSEC_KM_KICKED_FLAG - KM kicked flag - ignore additional requests */
#define IPSEC_KM_KICKED_FLAG 0x100

/* SAs can expire on both time and bytes processed.  These defines allow
cause the timers to issue warnings (causing IKE to renegotiate) and expire
a bit early.  The EXPIRED settings should probably be set to a random number
with that as a MAX value to allow for some jitter and to prevent both ends
of IKE from renegotiating at the same time.  This is TBD. */

#define SA_BYTES_EXPIRED 100 
#define SA_BYTES_EXPIRED_WARNING 1000
#define SA_TIME_EXPIRED 10
#define SA_TIME_EXPIRED_WARNING 30

/* How often to perform SA garbage collection */
#define SA_EXPIRE_CHECK 60 

/* Although SAs are typically unidirectional (by definition), 
   In this particular implementation it is possible to have 
   a single bi-directional SA.  In addition, to facilitate
   the use of what is termed "fast-keys", which can contain
   data relevant to the direction of the cipher (e.g. encrypt vs.
   decrypt), seperate inbound and outbound keys are passed to
   and stored in the kernel. To eliminate possible errors in
   typing in two long hex-strings, it is recommended that 
   manual interfaces only have users enter one of the keys and
   copy that key into the other structure.  */

typedef struct {
	    unsigned short key_len;
	    unsigned char key_data[MAX_KEY_LENGTH];
	    unsigned char *fast_key; /* processed when SA entry added */
	    } key_type;

struct net_addr {
	unsigned char addr_family;

    union {
	struct in_addr  ip_addr;
	 } addr_union;

	 };

struct esp_alg {
	unsigned short crypto_alg_id;
	unsigned short crypto_ivec_length;
	key_type outbound_crypto_key;
	key_type inbound_crypto_key;
	unsigned short auth_alg_id;
	unsigned short auth_ivec_length;
	key_type outbound_auth_key;
	key_type inbound_auth_key;
	unsigned short auth_data_length;
	};

struct ah_alg {
	unsigned short auth_alg_id;
	unsigned short auth_ivec_length;
	key_type outbound_auth_key;
	key_type inbound_auth_key;
	unsigned short auth_data_length;
	};


struct sa_node {
	unsigned long spi;
	uid_t  uid;
	struct net_addr peer_addr;
	unsigned long peer_spi;
	unsigned long sn;
	__u64 rpwin_bitmap;
	unsigned long lastsn;
	unsigned short mss_delta;
	unsigned short protocol;
	struct {
	long time_expired;
	long bytes_remaining;
	} lifetime;
	union {
		struct esp_alg esp;
		struct ah_alg ah;
		} alg;
	unsigned short flags;
	struct {
		unsigned short protocol;
		unsigned long spi;
		} next_encap;
	};

struct sadb_dst_node {
	struct net_addr dst;
	unsigned short	prefix_len;
	struct sa_node sa_info;
	struct sadb_dst_node *next_out;
	struct sadb_dst_node *next_out_hash;
	struct sadb_dst_node *next_in;
	struct sadb_dst_node *next_in_hash;
	};

struct sadb {
	struct sadb_dst_node *outbound[MAX_SADB_HASH_SIZE];
	struct sadb_dst_node *inbound[MAX_SADB_HASH_SIZE];
/* to be used for v6 IPsec  */	
	struct sadb_dst_node *outbound6[MAX_SADB_HASH6_SIZE];
	struct sadb_dst_node *inbound6[MAX_SADB_HASH6_SIZE];
	};

struct ipsec_system_policy {
	struct {
		unsigned short prot_flag;
		unsigned short priority[MAX_P_CHOICE]; /* TBD */
	} outbound;
	struct {
		unsigned short prot_flag;
	} inbound;
	};

#ifdef __KERNEL__
extern struct sadb sadb;
extern struct ipsec_system_policy ipsec_sys_policy;
extern __u16 registered_km_port;
#endif

/*
 * Structures for sadb messages.
 */
struct sadb_msghdr {
	unsigned short	sadbm_msglen;	/* to skip over non-understood messages */
	unsigned char	sadbm_version;	/* future binary compatability */
	unsigned char	sadbm_type;		/* message type */
	unsigned short	sadbm_errno;		/* why failed */
	unsigned short	sadbm_flags;		/* flags, incl. kern & message, e.g. DONE */
};

                
union sadb_union {
        struct sadb_dst_node dst;
        struct ipsec_system_policy sys_pol;
	__u16  udp_port;
};              

struct sadb_msg { 
        struct  sadb_msghdr m_sadbm;
        unsigned short request_type;
        union sadb_union sadb_info;
};
 

#define SADBM_VERSION	1	/* Up the ante and ignore older versions */

/* Message types */
#define SADBM_SET		0x1	
#define SADBM_DELETE	0x2
#define SADBM_CHANGE	0x3	
#define SADBM_GET		0x4	
#define SADBM_FLUSH		0x5
/* SADBM_ACQUIRE - # compatible with PF_KEY  */
#define	SADBM_ACQUIRE	0x6
/* SADBM_REGISTER - # compatible with PF_KEY  */
#define	SADBM_REGISTER	0x7
/* SADBM_GETSPI - # NOT compatible with PF_KEY (should be 1)  */
#define	SADBM_GETSPI	0x8
/* SADBM_UPDATE - # NOT compatible with PF_KEY (should be 2)  */
#define	SADBM_UPDATE	0x9
#define	SADBM_UNREGISTER	0xa

/* SADB Request Types for Commands other than GETSPI and REGISTER */
#define SADB_SYSPOL	0x1
#define SADB_DST	0x2

/* SADB Request Types for GETSPI */
#define SADB_ADD_PARTIAL_SA	0x4
#define SADB_NO_ADD_PARTIAL_SA	0x8

/* Temporary SADB Request Types for REGISTER */
#define KICK_KEY_MGMT		0x10
#define NO_KICK_KEY_MGMT	0x20

/* SADB Message Header flags */
#define SADBM_DONE 0x1
#define SADBM_MORE 0x2
#define SADBM_MANUAL 0x4
#define SADBM_IKE 0x8
#define SADBM_IKE_GETSPI 0x10

extern short get_sadb_hash(struct net_addr dst);
extern struct sadb_dst_node *get_sadb_node(struct net_addr dst,unsigned short protocol, unsigned long spi, unsigned short outbound, unsigned short check_expired);
extern void sadb_add(struct sadb_dst_node *d);
extern int sadb_kick_keymgmt(__u32 daddr, unsigned long spi, unsigned long peer_spi);
extern void sadb_keymgmt_delete_sa(struct sadb_dst_node *d);
extern int null_inbound_sa_found(struct net_addr dst);
extern void sadb_delete(struct sadb_dst_node *d);
extern void sadb_init(void);
extern void sadb_close(void);
extern void printk_key(key_type k);
extern void printk_net_addr(struct net_addr a);
extern void printk_sadb_dst_node(struct sadb_dst_node *d);
extern void sadb_expire_check(void);

#endif /* _SADB_H_ */
