/* transform.c */

/* The basic IPsec transform engine - This file is used to
   setup the algorithm transforms and load them into a hash
   table - similar to the INET protocol table.  This was done
   so that transforms could easily be written to run as independent
   Linux run-time modules */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/config.h>
#include <linux/netdevice.h>
#include "sadb.h"
#include "ipsec.h"
#include "transform.h"
#include "transforms/common_auth.h"
#include "transforms/common_cipher.h"
#include "transforms/aes_cipher.h"


#define NEXT_TRANSFORM NULL

#include "transforms/esp_des_cbc.h"
static struct ipsec_transform esp_des_cbc = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_des_cbc_sadb_add,
	esp_des_cbc_sadb_delete,
	ESP_DES_CBC,
	ESP_DES_CBC_ALG_NAME,
        CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_des_cbc

#include "transforms/hmacmd596.h"
static struct ipsec_transform hmacmd596 = {
	NULL,
	NULL,
	common_auth_input,
	common_auth_output,
	hmacmd596_sadb_add,
	hmacmd596_sadb_delete,
	HMACMD596,
	HMACMD596_NAME,
	AUTH_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &hmacmd596

#include "transforms/esp_null.h"
static struct ipsec_transform esp_null = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_null_sadb_add,
	esp_null_sadb_delete,
	ESP_NULL,
	ESP_NULL_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_null

#include "transforms/esp_3des_cbc.h"
static struct ipsec_transform esp_3des_cbc = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_3des_cbc_sadb_add,
	esp_3des_cbc_sadb_delete,
	ESP_3DES_CBC,
	ESP_3DES_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_3des_cbc

#include "transforms/esp_rc5_cbc.h"
static struct ipsec_transform esp_rc5_cbc = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_rc5_cbc_sadb_add,
	esp_rc5_cbc_sadb_delete,
	ESP_RC5_CBC,
	ESP_RC5_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_rc5_cbc

#include "transforms/esp_blf_cbc.h"
static struct ipsec_transform esp_blf_cbc = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_blf_cbc_sadb_add,
	esp_blf_cbc_sadb_delete,
	ESP_BLOWFISH_CBC,
	ESP_BLF_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_blf_cbc

#include "transforms/esp_idea_cbc.h"
static struct ipsec_transform esp_idea_cbc = {
	NULL,
	NULL,
	common_cipher_input,
	common_cipher_output,
	esp_idea_cbc_sadb_add,
	esp_idea_cbc_sadb_delete,
	ESP_IDEA_CBC,
	ESP_IDEA_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_idea_cbc

#include "transforms/esp_mars_cbc.h"
static struct ipsec_transform esp_mars_cbc = {
	NULL,
	NULL,
	aes_cipher_input,
	aes_cipher_output,
	esp_mars_cbc_sadb_add,
	esp_mars_cbc_sadb_delete,
	ESP_MARS_CBC,
	ESP_MARS_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_mars_cbc

#include "transforms/esp_rc6_cbc.h"
static struct ipsec_transform esp_rc6_cbc = {
	NULL,
	NULL,
	aes_cipher_input,
	aes_cipher_output,
	esp_rc6_cbc_sadb_add,
	esp_rc6_cbc_sadb_delete,
	ESP_RC6_CBC,
	ESP_RC6_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_rc6_cbc

#include "transforms/esp_rijndael_cbc.h"
static struct ipsec_transform esp_rijndael_cbc = {
	esp_rijndael_cbc_init,
	NULL,
	aes_cipher_input,
	aes_cipher_output,
	esp_rijndael_cbc_sadb_add,
	esp_rijndael_cbc_sadb_delete,
	ESP_RIJNDAEL_CBC,
	ESP_RIJNDAEL_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_rijndael_cbc

#include "transforms/esp_serpent_cbc.h"
static struct ipsec_transform esp_serpent_cbc = {
	NULL,
	NULL,
	aes_cipher_input,
	aes_cipher_output,
	esp_serpent_cbc_sadb_add,
	esp_serpent_cbc_sadb_delete,
	ESP_SERPENT_CBC,
	ESP_SERPENT_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};

#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_serpent_cbc

#include "transforms/esp_twofish_cbc.h"
static struct ipsec_transform esp_twofish_cbc = {
	esp_twofish_cbc_init,
	NULL,
	aes_cipher_input,
	aes_cipher_output,
	esp_twofish_cbc_sadb_add,
	esp_twofish_cbc_sadb_delete,
	ESP_TWOFISH_CBC,
	ESP_TWOFISH_CBC_ALG_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &esp_twofish_cbc

#include "transforms/hmacsha96.h"
static struct ipsec_transform hmacsha96 = {
	NULL,
	NULL,
	common_auth_input,
	common_auth_output,
	hmacsha96_sadb_add,
	hmacsha96_sadb_delete,
	HMACSHA96,
	HMACSHA96_NAME,
	AUTH_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &hmacsha96

#include "transforms/cipher_test.h"
static struct ipsec_transform cipher_test = {
	NULL,
	NULL,
	cipher_test_input,
	cipher_test_output,
	cipher_test_sadb_add,
	NULL,
	CIPHER_TEST_ALG,
	CIPHERTEST_NAME,
	CIPHER_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &cipher_test

#include "transforms/auth_test.h"
static struct ipsec_transform auth_test = {
	NULL,
	NULL,
	auth_test_input,
	auth_test_output,
	auth_test_sadb_add,
	NULL,
	AUTH_TEST_ALG,
	AUTHTEST_NAME,
	AUTH_TRANSFORM,
	NEXT_TRANSFORM
};
#undef NEXT_TRANSFORM
#define NEXT_TRANSFORM &auth_test


static struct ipsec_transform *ipsec_transform_base = NEXT_TRANSFORM;

static struct ipsec_transform *ipsec_transforms[MAX_IPSEC_TRANSFORMS] = 
{
	NULL
};

struct ipsec_transform  *ipsec_get_transform(unsigned char id, unsigned char type)
{
	unsigned char hash=0;
	struct ipsec_transform *t=NULL;

	if (id == NO_ALG) return NULL;
	if ((type != AUTH_TRANSFORM) && (type != CIPHER_TRANSFORM)) return NULL;
	hash = (id*type) & (MAX_IPSEC_TRANSFORMS - 1);
	for (t=ipsec_transforms[hash] ; t != NULL; t=t->next)
	{
		if ((t->id == id) && (t->type == type))
			return(t);
	}
	printk(KERN_INFO "IPSEC_GET_TRANSFORM: No such transform (id = %d, type = %d)\n",id,type);
	return NULL ;

}

void ipsec_add_transform(struct ipsec_transform *trans)
{
	unsigned char hash;
	struct ipsec_transform *t;
	
	if (trans->id == NO_ALG) return;
#ifdef MODULE
	ipsec_inc_mod_count();
#endif
	printk("%s ",trans->name);
	hash = (trans->id * trans->type) & (MAX_IPSEC_TRANSFORMS - 1);
	trans->next = ipsec_transforms[hash];
	ipsec_transforms[hash] = trans;
	if (trans->init_handler) trans->init_handler();
	t = trans->next;
	while (t != NULL)
	{
		if ((t->id == trans->id) && (t->type == trans->type)){
			printk("Duplicate IPSEC Transform (id = %d, type = %d)\n",t->id,t->type);
			ipsec_transforms[hash] = trans->next;
			break;
		}
		t = t->next;
	}
#ifdef MODULE
	ipsec_dec_mod_count();
#endif
}

void transform_init(void)
{
	struct ipsec_transform *t;
#ifdef MODULE
	ipsec_inc_mod_count();
#endif

	printk(KERN_INFO "IPSEC Transforms: ");
	for (t=ipsec_transform_base;t!=NULL;)
	{
		struct ipsec_transform *tmp = t->next;
		ipsec_add_transform(t);
		t = tmp;
	}
	printk("\n");
#ifdef MODULE
	ipsec_dec_mod_count();
#endif
}
