/* auth_test.c */
/* AH/ESP, Authentication test transform specific code */
/* This code assembles the authentication portion of the ESP packet
   and the AH part of an AH header but does no actuall authentication */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <net/ip.h>
#include <net/sock.h>
#include <linux/in.h>
#include <linux/in6.h>
#include <linux/skbuff.h>
#include <asm/uaccess.h>
#include <asm/checksum.h>
#include "../sadb.h"
#include "../ipsec.h"
#include "../transform.h"
#include "auth_test.h"

static char addr_s2[18];

#define ALG_NAME AUTHTEST_NAME

struct sk_buff *auth_test_input(struct sk_buff *skb, struct sadb_dst_node *d, struct ipsec_transform *t)
{
        unsigned char *pdata=0, adata[MAX_AUTH_TEST_AD_LENGTH];
	struct iphdr tmpiph;
        unsigned long spi=0;
        struct iphdr *iph = (struct iphdr *)skb->data;
        struct ah_header ah;
	unsigned short ihl = iph->ihl<<2;
        unsigned char ivec[MAX_AUTH_TEST_IVEC_LENGTH];
        int sn_length=0;

	spi = d->sa_info.spi;

	switch (d->sa_info.protocol)
	{
	case IPPROTO_AH:
		bprintk(skb->data,skb->len,"IP Packet before Authenticator Decapsulation"); 

        	pdata = skb_pull(skb,ihl);

        	ah = *(struct ah_header *)pdata;
		sn_length = d->sa_info.sn?4:0;

       		memcpy(adata,pdata+sizeof(ah)+sn_length,d->sa_info.alg.ah.auth_data_length<<2);
        	memset(pdata+sizeof(ah)+sn_length,0,d->sa_info.alg.ah.auth_data_length<<2);

		memcpy(ivec,pdata+sizeof(ah)+(ah.length<<2),d->sa_info.alg.ah.auth_ivec_length);

        	memcpy(&tmpiph,iph,ihl);
        	iph_ah_prep((unsigned char *)&tmpiph);
        	iph->protocol=ah.next_header;
        	iph->tot_len = htons(ntohs(iph->tot_len) - sizeof(ah) - (d->sa_info.alg.ah.auth_data_length<<2) - d->sa_info.alg.ah.auth_ivec_length - sn_length);

        	iph->check = 0;
        	iph->check = ip_fast_csum((unsigned char *)iph,iph->ihl);
        	skb_pull(skb,sizeof(ah)+(d->sa_info.alg.ah.auth_data_length<<2)+d->sa_info.alg.ah.auth_ivec_length+sn_length);

        	skb_push(skb,ihl);

		/* IP header copy MUST be in tail to head order (hence r_memcpy) to avoid the (ihl<<2)-8 bytes of overlap */
        	r_memcpy(skb->data,(unsigned char *)iph,ihl);

        	skb->nh.raw=skb->data;
        	skb->h.raw=skb->nh.raw+ihl;

		bprintk(skb->data,skb->len,"IP Packet after Authenticator Decapsulation"); 
	break;
	case IPPROTO_ESP:
	break;
	}

        return skb;
}

struct sk_buff *auth_test_output(struct sk_buff *skb, struct sadb_dst_node *d, struct ipsec_transform *t)
{
        struct rtable *rt=(struct rtable*)skb->dst;
        struct device *dev = rt->u.dst.dev;
        struct iphdr *iph = (struct iphdr *)skb->data;
        struct iphdr tmpiph;
        unsigned long spi;
        unsigned short dlen,ihl;
        struct sk_buff *tmpskb=0;
        struct ah_header ah;
        unsigned char adata[MAX_AUTH_TEST_AD_LENGTH];
        int error=0,sn_length;

#define errout(e) { error = e; printk(KERN_INFO "%s: Outbound: Protocol Error %d\n", ALG_NAME, e);goto out_error;}

	switch (d->sa_info.protocol)
	{
	case IPPROTO_AH:
		bprintk(skb->data,skb->len,"IP Packet before Authenticator Encapsulation"); 

		sn_length = d->sa_info.sn?4:0;
        	spi = ah.spi = d->sa_info.peer_spi;
        	ah.spi = htonl(ah.spi);
        	ah.reserved = 0;
	
        	ah.length = d->sa_info.alg.ah.auth_data_length + (d->sa_info.sn?1:0);

        	if (d->sa_info.flags & IPSEC_TUNNEL_FLAG)
        	{
                	ah.next_header = IPPROTO_IPIP;
                	dlen = ntohs(iph->tot_len);
			ihl = 20;
        	}
        	else
        	{
                	ah.next_header = iph->protocol;
                	dlen = ntohs(iph->tot_len)-(iph->ihl<<2);
			ihl = iph->ihl<<2;
        	}

		/* allocate skbuff to it's largest size possible */
        	if (!(tmpskb = ipsec_alloc_skb(skb,dlen+((dev->hard_header_len+15)&~15)+ihl+sizeof(ah)+sn_length+(d->sa_info.alg.ah.auth_data_length<<2)+d->sa_info.alg.ah.auth_ivec_length)))errout(ENOBUFS);
	
        	skb_reserve(tmpskb,((dev->hard_header_len+15)&~15)+ihl+sizeof(ah)+sn_length+(d->sa_info.alg.ah.auth_data_length<<2));
        	skb_put(tmpskb,dlen+d->sa_info.alg.ah.auth_ivec_length);

		/* Generate IVEC */
        	generate_ivec(tmpskb->data,d->sa_info.alg.ah.auth_ivec_length);

		/* add IP data */
        	if (d->sa_info.flags & IPSEC_TUNNEL_FLAG)
                	memcpy(tmpskb->data+d->sa_info.alg.ah.auth_ivec_length,skb->data,dlen);
        	else 
                	memcpy(tmpskb->data+d->sa_info.alg.ah.auth_ivec_length,skb->data+ihl,dlen);


        	skb_push(tmpskb,(d->sa_info.alg.ah.auth_data_length<<2)+sn_length+sizeof(ah)+ihl);
        	memcpy(tmpskb->data,skb->data,ihl);
        	memcpy(tmpskb->data+ihl,&ah,sizeof(ah));
		if (d->sa_info.sn)
		{
			unsigned long tmp_sn = ntohl(d->sa_info.sn);
			memcpy(tmpskb->data+ihl+sizeof(ah),&tmp_sn, sn_length);
			d->sa_info.sn++;
		}

        	iph = (struct iphdr *)tmpskb->data;
        	iph->protocol = IPPROTO_AH;

		/* change address if IPIP encapsulation */
		/* fill in new header if flags & IPSEC_TUNNEL_FLAG */
        	if (d->sa_info.flags & IPSEC_TUNNEL_FLAG){

                	ip_rt_put(rt);
                	error = ip_route_output(&rt,d->sa_info.peer_addr.addr_union.ip_addr.s_addr,0,iph->tos,0);
                	if (error)errout(error);
                	iph->saddr = rt->rt_src;
                	iph->daddr = rt->rt_dst;
                	iph->ihl = ihl>>2;
                	iph->id = htons(ip_id_count++);
                	iph->frag_off &= htons(IP_DF);
                	iph->ttl = ip_statistics.IpDefaultTTL;

        	        tmpskb->dev = rt->u.dst.dev;
                	tmpskb->dst = dst_clone(&rt->u.dst);
        	}

        	iph->tot_len = htons(ihl + dlen + sizeof(ah)+(d->sa_info.alg.ah.auth_data_length<<2)+d->sa_info.alg.ah.auth_ivec_length+sn_length);

        	memcpy(&tmpiph,iph,ihl);

        	iph_ah_prep((unsigned char *)&tmpiph);

        	memset(adata,0,(d->sa_info.alg.ah.auth_data_length<<2));

        	memcpy(tmpskb->data+ihl+sizeof(ah)+sn_length,adata,d->sa_info.alg.ah.auth_data_length<<2);

		bprintk(skb->data,skb->len,"IP Packet after Authenticator Encapsulation"); 

        	if (skb) kfree_skb(skb);
        	return tmpskb;
	break;
	case IPPROTO_ESP:
		return skb;
	break;
	}
out_error:
	if (tmpskb) kfree_skb(tmpskb);
	if (skb) kfree_skb(skb);
  	return NULL;

}

int auth_test_sadb_add(struct sadb_dst_node *d)
{

	switch (d->sa_info.protocol)
	{
	case IPPROTO_AH:
                d->sa_info.mss_delta = sizeof(struct ah_header)+ (d->sa_info.alg.ah.auth_data_length<<2)+
                           d->sa_info.alg.ah.auth_ivec_length+(d->sa_info.sn?4:0)+
                           ((d->sa_info.flags & IPSEC_TUNNEL_FLAG)?20:0);
                ipsec_mss_delta = max(ipsec_mss_delta,d->sa_info.mss_delta);
	break;
	case IPPROTO_ESP:
	break;
	default:
                printk(KERN_INFO "%s: SADB: Invalid Protocol: dst %s spi %lx protocol %x\n", ALG_NAME,
                        strcpy(addr_s2,ntoa(d->dst.addr_union.ip_addr.s_addr)), d->sa_info.spi, d->sa_info.protocol);
                return 0;
	break;
	}
	printk_sadb_dst_node(d);

	return 1;
}
