/*
 *	SILCPacket.java		2002/11/06
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.nio.ByteBuffer;
import java.security.SecureRandom;
import java.io.IOException;

/** Describes a SILC Packet.
 * 
 * @author Alistair K Phipps
 * @version 20021115
 */
public class SILCPacket
{
	/** Header to be sent at start of packet */
	private SILCHeader _header;

	/** Padding to be sent as part of packet to pad out to cipher block length multiple */
	private byte[] _ylPadding;

	/** Payloads to be sent in packet - note for some packet types the payload may actually include a list of payloads */
	private SILCPayload _payload;

	/** Constructor to create a packet from specified header information and payload list
	 * @param yType Packet type
	 * @param ySrcIdType Source ID type
	 * @param ylSrcId Source ID
	 * @param yDestIdType Destination ID type
	 * @param ylDestId Destination ID
	 * @param payload Payloads to be sent in packet
	 */
	public SILCPacket( byte yType, byte ySrcIdType, byte[] ylSrcId, byte yDestIdType, byte[] ylDestId, SILCPayload payload )
	{
		// create the header
		// calculate payload length
		int iPayloadLength = SILCHeader.getEmptyLength() + ylSrcId.length + ylDestId.length + payload.toByteList().length;
		assert( iPayloadLength < 65536 ); // must fit in unsigned short
		
		// calculate padding length (between 1 and 16)

		// http://www.silcnet.org/docs/draft-riikonen-silc-pp-07.txt Section 2.7
		// padding_length = 16 - (packet_length mod block_size)
        // if (padding_length < 8)
        // padding_length += block_size

		// byte yPadLength = (byte)( 16 - ( iPayloadLength % 16 ) );
		int padlength = ( 16 - ( iPayloadLength % 16 ) );
		if (padlength<8)
			padlength+=16;
		byte yPadLength = (byte)padlength;
		//byte yPadLength = (byte)( 16 - ( iPayloadLength % 16 ) );


		// XXX assume do not have private message with private message key (not implemented here), packet never broadcast (can only be sent from router, which will not be implemented here) or compressed (not implemented here)
		_header = new SILCHeader( yType, (short)iPayloadLength, ySrcIdType, ylSrcId, yDestIdType, ylDestId, yPadLength, payload.isList(), false, false, false );

		// generate the padding
		SecureRandom sr = new SecureRandom();
		_ylPadding = new byte[ _header.getPadLength() & 0xFF ];
		sr.nextBytes( _ylPadding );
		
		// set up payload
		_payload = payload;
	}

	/** Constructor to create a packet by reading it from the specified byte list
	 */
	public SILCPacket( byte[] yl )
	{
		ByteBuffer yb = ByteBuffer.wrap( yl );
		ByteBuffer ybHeader = yb.duplicate();	// header is part of full packet
		ybHeader.limit( SILCHeader.getEmptyLength() + ( yb.get( 6 ) & 0xFF ) + ( yb.get( 7 ) & 0xFF ) ); // but ends at header length + srcid length + destid length
		_header = new SILCHeader( ybHeader.array() ); // XXX: should probably check this kind if bytebuffer can be converted to array
		// check packet is correct length
		assert( ( ( _header.getPayloadLength() & 0xFFFF ) + ( _header.getPadLength() & 0xFF ) ) == yb.limit() );
		// read padding (we need it to be intact to do hashes of the packet etc later)
		_ylPadding = new byte[ _header.getPadLength() & 0xFF ];
		yb.position( ybHeader.limit() );
		yb.get( _ylPadding );
		// if we have a payload...
		if( yb.limit() >= ybHeader.limit() + ( _header.getPadLength() & 0xFF ) )
		{
			// move to first payload byte
			yb.position( ybHeader.limit() + ( _header.getPadLength() & 0xFF ) );
			ByteBuffer ybPayload = yb.slice();
			byte[] ylPayload = new byte[ ybPayload.limit() ];
			ybPayload.get( ylPayload ); // have to make a copy as .array() just returns the backing array
			switch( _header.getType() )
			{
				case SILCPacketType.SUCCESS:
					_payload = new SILCSuccessPayload( ylPayload );
					break;
				case SILCPacketType.FAILURE:
					_payload = new SILCFailurePayload( ylPayload );
					break;
				case SILCPacketType.REJECT:
					_payload = new SILCRejectPayload( ylPayload );
					break;
				case SILCPacketType.ERROR:
					_payload = new SILCErrorPayload( ylPayload );
					break;
				case SILCPacketType.DISCONNECT:
					_payload = new SILCDisconnectPayload( ylPayload );
					break;
				case SILCPacketType.KEY_EXCHANGE:
					_payload = new SILCKeyExchangeStartPayload( ylPayload );
					break;
				/*case SILCPacketType.KEY_EXCHANGE_1: XXX this should only happen at the server side - not implemented
					_payload = new SILCKeyExchange1Payload( ylPayload );*/
				case SILCPacketType.KEY_EXCHANGE_2:
					_payload = new SILCKeyExchange2Payload( ylPayload );
					break;
				case SILCPacketType.HEARTBEAT:
					_payload = new SILCHeartbeatPayload( ylPayload );
					break;
				case SILCPacketType.CONNECTION_AUTH:
					_payload = new SILCConnectionAuthPayload( ylPayload );
					break;
				case SILCPacketType.NEW_CLIENT:
					_payload = new SILCNewClientPayload( ylPayload );
					break;
				case SILCPacketType.NEW_ID:
					_payload = new SILCIdPayload( ylPayload );
					break;
				case SILCPacketType.PRIVATE_MESSAGE:
					_payload = new SILCGenericMessagePayload( ylPayload );
					break;
				case SILCPacketType.NOTIFY:
					_payload = new SILCNotifyPayload( ylPayload );
					break;
				/*case SILCPacketType.COMMAND:*/	// XXX we should never receive a COMMAND packet - only server does
				case SILCPacketType.COMMAND_REPLY:	// command replies are the same syntax as command payloads
					_payload = new SILCCommandPayload( ylPayload );
					break;
				default:
					_payload = new SILCUnknownPayload( ylPayload );
			}
		}
		else	// no payload
			_payload = null;
	}

	/** convert the packet to a byte list */
	public byte[] toByteList()
	{
		ByteBuffer yb = ByteBuffer.allocate( ( _header.getPayloadLength() & 0xFFFF ) + ( _header.getPadLength() & 0xFF ) );
		yb.put( _header.toByteList() );
		yb.put( _ylPadding );
		yb.put( _payload.toByteList() );
		return yb.array();
	}

	public byte getType()
	{
		return _header.getType();
	}
	
	public SILCPayload getPayload()
	{
		return _payload;
	}

	public byte getSourceIdType()
	{
		return _header.getSourceIdType();
	}

	public byte[] getSourceId()
	{
		return _header.getSourceId();
	}

	public byte getDestIdType()
	{
		return _header.getDestIdType();
	}

	public byte[] getDestId()
	{
		return _header.getDestId();
	}
	
	public String toString()
	{	// don't both outputting the padding data
		return "HEADER\n======\n" + _header.toString() + "\nPAYLOAD\n=======\n" + _payload.toString();
	}
}
