/*
 * RIPEMD-160 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * RIPEMD-160 algorithm by Hans Dobbertin, Antoon Bosselaers and Bart Preneel,
 * see http://www.esat.kuleuven.ac.be/~bosselae/ripemd160.html
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2286
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the RIPEMD-160 routines.
 *
 * Data sizes:
 *
 * data block for RIPEMD-160 transformation	WORD08[64]
 * resulting RIPEMD-160 or HMAC hash		WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * RMD160_FULL defined:
 *
 * 1. Call rmd160init
 * 2. For all data to be hashed call rmd160next
 * 3. To retrieve the hash call rmd160end
 *
 * RMD160_FAST defined:
 *
 * Call rmd160 to get the hash for the specified data
 *
 * RMD160_HMAC_FULL defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hminit
 * 3. For all data to be hashed call rmd160hmnext
 * 4. To retrieve the hash call rmd160hmend
 * 5. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 *
 * RMD160_HMAC_FAST defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 */

#include "common.h"
#include "rmd160.h"

/* rmd160 result initial data */

static WORD32 rmd160start[5]=
{
	0x67452301,0xefcdab89,0x98badcfe,0x10325476,0xc3d2e1f0
};

/* rmd160 offset and rotation table */

static WORD08 rmd160tab[320]=
{
	 0,11, 5, 8, 1,14,14, 9, 2,15, 7, 9, 3,12, 0,11,
	 4, 5, 9,13, 5, 8, 2,15, 6, 7,11,15, 7, 9, 4, 5,
	 8,11,13, 7, 9,13, 6, 7,10,14,15, 8,11,15, 8,11,
	12, 6, 1,14,13, 7,10,14,14, 9, 3,12,15, 8,12, 6,
	 7, 7, 6, 9, 4, 6,11,13,13, 8, 3,15, 1,13, 7, 7,
	10,11, 0,12, 6, 9,13, 8,15, 7, 5, 9, 3,15,10,11,
	12, 7,14, 7, 0,12,15, 7, 9,15, 8,12, 5, 9,12, 7,
	 2,11, 4, 6,14, 7, 9,15,11,13, 1,13, 8,12, 2,11,
	 3,11,15, 9,10,13, 5, 7,14, 6, 1,15, 4, 7, 3,11,
	 9,14, 7, 8,15, 9,14, 6, 8,13, 6, 6, 1,15, 9,14,
	 2,14,11,12, 7, 8, 8,13, 0,13,12, 5, 6, 6, 2,14,
	13, 5,10,13,11,12, 0,13, 5, 7, 4, 7,12, 5,13, 5,
	 1,11, 8,15, 9,12, 6, 5,11,14, 4, 8,10,15, 1,11,
	 0,14, 3,14, 8,15,11,14,12, 9,15, 6, 4, 8, 0,14,
	13, 9, 5, 6, 3,14,12, 9, 7, 5, 2,12,15, 6,13, 9,
	14, 8, 9,12, 5, 6, 7, 5, 6, 5,10,15, 2,12,14, 8,
	 4, 9,12, 8, 0,15,15, 5, 5, 5,10,12, 9,11, 4, 9,
	 7, 6, 1,12,12, 8, 5, 5, 2,13, 8,14,10,12, 7, 6,
	14, 5, 6, 8, 1,12, 2,13, 3,13,13, 6, 8,14,14, 5,
	11,11, 0,15, 6, 8, 3,13,15, 5, 9,11,13, 6,11,11,
};

/*
 * procedure rmd160block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: rmd160 - pointer to the rmd160 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current rmd160 result.
 */

static void CRYPTOCALL rmd160block(WORD32 *rmd160,WORD08 *data)
{
	register WORD32 i;	/* counter			*/
	register WORD32 j;	/* counter and index		*/
	register WORD32 *w;	/* work buffer fast access	*/
	register union
	{
		WORD08 *b;	/* byte rover			*/
		WORD32 *l;	/* 32 bit rover			*/
	} r;
	WORD32 wrk[28];		/* work buffer			*/


	/* get 64 byte message block to work buffer (endian independent) */

	for(w=wrk,r.b=data,i=0;i<16;w++,i++)
	{
		*w=(WORD32)(*r.b++);
		*w|=((WORD32)(*r.b++))<<8;
		*w|=((WORD32)(*r.b++))<<16;
		*w|=((WORD32)(*r.b++))<<24;
	}

	/* get previous rmd160 block result to work buffer */

	for(r.l=rmd160,i=5;i;w++,i--)w[7]=w[1]=BITS32(*r.l++);

	/* do round one and parallel round one */

	for(w=wrk+16,r.b=rmd160tab,i=16;i;i--)
	{
		w[1]+=(w[2]^w[3]^w[4])+w[*r.b++-16];
		j=*r.b++;
		w[1]=((w[1]<<j)|(BITS32(w[1])>>(32-j)))+w[5];
		w[3]=(w[3]<<10)|(BITS32(w[3])>>22);
		w[7]+=(w[8]^(w[9]|~w[10]))+w[*r.b++-16]+0x50a28be6;
		j=*r.b++;
		w[7]=((w[7]<<j)|(BITS32(w[7])>>(32-j)))+w[11];
		w[9]=(w[9]<<10)|(BITS32(w[9])>>22);
		for(w[0]=w[5],w[6]=w[11],j=11;j;j--)w[j]=w[j-1];
	}

	/* do round two and parallel round two */

	for(i=16;i;i--)
	{
		w[1]+=(w[4]^(w[2]&(w[3]^w[4])))+w[*r.b++-16]+0x5a827999;
		j=*r.b++;
		w[1]=((w[1]<<j)|(BITS32(w[1])>>(32-j)))+w[5];
		w[3]=(w[3]<<10)|(BITS32(w[3])>>22);
		w[7]+=(w[9]^(w[10]&(w[8]^w[9])))+w[*r.b++-16]+0x5c4dd124;
		j=*r.b++;
		w[7]=((w[7]<<j)|(BITS32(w[7])>>(32-j)))+w[11];
		w[9]=(w[9]<<10)|(BITS32(w[9])>>22);
		for(w[0]=w[5],w[6]=w[11],j=11;j;j--)w[j]=w[j-1];
	}

	/* do round three and parallel round three */

	for(i=16;i;i--)
	{
		w[1]+=((w[2]|~w[3])^w[4])+w[*r.b++-16]+0x6ed9eba1;
		j=*r.b++;
		w[1]=((w[1]<<j)|(BITS32(w[1])>>(32-j)))+w[5];
		w[3]=(w[3]<<10)|(BITS32(w[3])>>22);
		w[7]+=((w[8]|~w[9])^w[10])+w[*r.b++-16]+0x6d703ef3;
		j=*r.b++;
		w[7]=((w[7]<<j)|(BITS32(w[7])>>(32-j)))+w[11];
		w[9]=(w[9]<<10)|(BITS32(w[9])>>22);
		for(w[0]=w[5],w[6]=w[11],j=11;j;j--)w[j]=w[j-1];
	}

	/* do round four and parallel round four */

	for(i=16;i;i--)
	{
		w[1]+=(w[3]^(w[4]&(w[2]^w[3])))+w[*r.b++-16]+0x8f1bbcdc;
		j=*r.b++;
		w[1]=((w[1]<<j)|(BITS32(w[1])>>(32-j)))+w[5];
		w[3]=(w[3]<<10)|(BITS32(w[3])>>22);
		w[7]+=(w[10]^(w[8]&(w[9]^w[10])))+w[*r.b++-16]+0x7a6d76e9;
		j=*r.b++;
		w[7]=((w[7]<<j)|(BITS32(w[7])>>(32-j)))+w[11];
		w[9]=(w[9]<<10)|(BITS32(w[9])>>22);
		for(w[0]=w[5],w[6]=w[11],j=11;j;j--)w[j]=w[j-1];
	}

	/* do round five and parallel round five */

	for(i=16;i;i--)
	{
		w[1]+=(w[2]^(w[3]|~w[4]))+w[*r.b++-16]+0xa953fd4e;
		j=*r.b++;
		w[1]=((w[1]<<j)|(BITS32(w[1])>>(32-j)))+w[5];
		w[3]=(w[3]<<10)|(BITS32(w[3])>>22);
		w[7]+=(w[8]^w[9]^w[10])+w[*r.b++-16];
		j=*r.b++;
		w[7]=((w[7]<<j)|(BITS32(w[7])>>(32-j)))+w[11];
		w[9]=(w[9]<<10)|(BITS32(w[9])>>22);
		for(w[0]=w[5],w[6]=w[11],j=11;j;j--)w[j]=w[j-1];
	}

	/* create new block result */

	r.l=rmd160;
	w[10]+=w[3]+r.l[1];
	r.l[1]=r.l[2]+w[4]+w[11];
	r.l[2]=r.l[3]+w[5]+w[7];
	r.l[3]=r.l[4]+w[1]+w[8];
	r.l[4]=r.l[0]+w[2]+w[9];
	r.l[0]=w[10];
}

/*
 * procedure rmd160init
 *
 * input: ptr - pointer to RMD160 internal data
 *
 * This procedure initializes the RMD160 internal data structure.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160init(register RMD160DATA *ptr)
{
	register WORD32 i;	/* counter and index */


	/* initialize all RMD160 internal data */

	for(ptr->total=ptr->size=i=0;i<5;i++)ptr->rmd160[i]=rmd160start[i];
}

#endif

/*
 * procedure rmd160next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to RMD160 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the RMD160 internal data structure.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160next(register WORD08 *data,register WORD32 length,
	register RMD160DATA *ptr)
{
	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed */

	for(;ptr->size&&length;length--)
	{
		/* copy current byte to buffer */

		ptr->bfr[(ptr->size)++]=*data++;

		/* if the buffer is full */

		if(ptr->size==64)
		{
			/* reset buffer size */

			ptr->size=0;

			/* hash buffer */

			rmd160block(ptr->rmd160,ptr->bfr);
		}
	}

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)
		rmd160block(ptr->rmd160,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr[(ptr->size)++]=*data++;
}

#endif

/*
 * procedure rmd160end
 *
 * input:  ptr - pointer to RMD160 internal data
 *
 * output: result - the RMD160 hash
 *
 * This procedure finalizes and returns the RMD160 hash.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160end(WORD08 *result,RMD160DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/
	register WORD08 *p08=ptr->bfr+i;	/* byte rover		*/
	register WORD32 *rmd160=ptr->rmd160;	/* current hash		*/
	register WORD32 val;			/* length creation	*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	for(*p08++=0x80,i++;i!=56;*p08++=0,i++)if(i==64)
	{
		i=0;
		p08=ptr->bfr;
		rmd160block(rmd160,p08);
	}

	/* append bits 0 through 31 of total length in bits to buffer */

	for(i=4,val=(ptr->total)<<3;i;val>>=8,i--)*p08++=(WORD08)(val);

	/* append bits 32 through 63 of total length in bits to buffer */

	*p08++=(WORD08)(BITS32(ptr->total)>>29);
	*p08++=0;
	*p08++=0;
	*p08=0;

	/* hash buffer */

	rmd160block(rmd160,ptr->bfr);

	/* return final hash result to caller */

	for(i=5,p08=result;i;rmd160++,i--)
	{
		*p08++=(WORD08)(*rmd160);
		*p08++=(WORD08)((*rmd160)>>8);
		*p08++=(WORD08)((*rmd160)>>16);
		*p08++=(WORD08)((*rmd160)>>24);
	}

	/* assert user data removal data from buffer */

#ifdef RMD160_PARANOID
	for(i=64,p08=ptr->bfr;i;*p08++=0,i--);
#endif
}

#endif

/*
 * procedure rmd160
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the RMD160 hash
 *
 * This procedure hashes the given data.
 */

#ifdef RMD160_FAST

void CRYPTOCALL rmd160(register WORD08 *data,register WORD32 length,
	WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	union
	{
		WORD32 total;	/* data length buffer		*/
		WORD32 *ptr;	/* fast buffer access		*/
	}u;
	WORD32 val;		/* length creation		*/
	WORD32 rmd160[5];	/* RMD160 current result buffer	*/
	WORD08 bfr[64];		/* final block(s) buffer	*/


	/* initialize RMD160 internal data */

	for(u.total=length,i=0;i<5;i++)rmd160[i]=rmd160start[i];

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)rmd160block(rmd160,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	data=bfr+i;
	for(*data++=0x80,i++;i!=56;*data++=0,i++)if(i==64)
	{
		i=0;
		data=bfr;
		rmd160block(rmd160,data);
	}

	/* append bits 0 through 31 of total length in bits to buffer */

	for(i=4,val=u.total<<3;i;val>>=8,i--)*data++=(WORD08)(val);

	/* append bits 32 through 63 of total length in bits to buffer */

	*data++=(WORD08)(BITS32(u.total)>>29);
	*data++=0;
	*data++=0;
	*data=0;

	/* hash buffer */

	rmd160block(rmd160,bfr);

	/* return final hash result to caller */

	for(i=5,data=result,u.ptr=rmd160;i;u.ptr++,i--)
	{
		*data++=(WORD08)(*u.ptr);
		*data++=(WORD08)((*u.ptr)>>8);
		*data++=(WORD08)((*u.ptr)>>16);
		*data++=(WORD08)((*u.ptr)>>24);
	}

	/* assert user data removal data from buffer */

#ifdef RMD160_PARANOID
	for(i=64,data=bfr;i;*data++=0,i--);
#endif

}

#endif

/*
 * procedure rmd160hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef RMD160_HMAC

void CRYPTOCALL rmd160hmkey(WORD08 *key,WORD32 keylength,RMD160HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	RMD160(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		RMD160DATA rmd160data;	/* RMD160 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		rmd160init(&u.rmd160data);
		rmd160next(key,keylength,&u.rmd160data);
		rmd160end(hash,&u.rmd160data);
		key=hash;
		keylength=RMD160_SIZE;
		for(i=0;i<5;i++)u.rmd160data.rmd160[i]=0;
	}

	/* copy initial values to HMAC internal data structure */

	for(i=0;i<5;i++)ptr->irmd160[i]=ptr->ormd160[i]=rmd160start[i];

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	rmd160block(ptr->irmd160,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	rmd160block(ptr->ormd160,u.pad);

	/* clean up */

	for(i=0;i<RMD160_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure rmd160hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the RMD160 internal data
 *
 * This procedure initializes RMD160 data for HMAC use.
 */

#ifdef RMD160_HMAC_FULL

void CRYPTOCALL rmd160hminit(RMD160DATA *ptr,RMD160HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* initialize all RMD160 internal data */

	for(ptr->total=64,ptr->size=i=0;i<5;i++)ptr->rmd160[i]=key->irmd160[i];
}

#endif

/*
 * procedure rmd160hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the RMD160 internal data
 *
 * output: result - the HMAC result (a RMD160 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef RMD160_HMAC_FULL

void CRYPTOCALL rmd160hmend(WORD08 *result,RMD160DATA *ptr,RMD160HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* complete inner RMD160 */

	rmd160end(result,ptr);

	/* initialize outer RMD160 */

	for(ptr->total=64,ptr->size=i=0;i<5;i++)ptr->rmd160[i]=key->ormd160[i];

	rmd160next(result,RMD160_SIZE,ptr);
	rmd160end(result,ptr);
}

#endif

/*
 * procedure rmd160hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a RMD160 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef RMD160_HMAC_FAST

void CRYPTOCALL rmd160hmac(WORD08 *data,WORD32 length,WORD08 *result,
	RMD160HMDATA *key)
{
	register WORD32 i;	/* counter and index */
	RMD160DATA rmd160data;	/* RMD160 internal data */


	/* initialize inner RMD160 */

	for(rmd160data.total=64,rmd160data.size=i=0;i<5;i++)
		rmd160data.rmd160[i]=key->irmd160[i];

	/* hash given data */

	rmd160next(data,length,&rmd160data);

	/* complete inner RMD160 */

	rmd160end(result,&rmd160data);

	/* initialize outer RMD160 */

	for(rmd160data.total=64,rmd160data.size=i=0;i<5;i++)
		rmd160data.rmd160[i]=key->ormd160[i];

	/* complete and return outer RMD160 */

	rmd160next(result,RMD160_SIZE,&rmd160data);
	rmd160end(result,&rmd160data);
}

#endif

