/*
 * SHA1 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * SHA1 algorithm: see FIPS PUB 180-1
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the SHA1 routines.
 *
 * Data sizes:
 *
 * data block for SHA1 transformation	WORD08[64]
 * resulting SHA1 or HMAC hash		WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * SHA1_FULL defined:
 *
 * 1. Call sha1init
 * 2. For all data to be hashed call sha1next
 * 3. To retrieve the hash call sha1end
 *
 * SHA1_FAST defined:
 *
 * Call sha1 to get the hash for the specified data
 *
 * SHA1_HMAC_FULL defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hminit
 * 3. For all data to be hashed call sha1hmnext
 * 4. To retrieve the hash call sha1hmend
 * 5. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 *
 * SHA1_HMAC_FAST defined:
 *
 * 1. Call sha1hmkey once to preprocess the selected key
 * 2. Call sha1hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the SHA1HMDATA structure
 */

#include "common.h"
#include "sha1.h"

/* sha1 result initial data */

static WORD32 sha1start[5]=
{
	0x67452301,0xEFCDAB89,0x98BADCFE,0x10325476,0xC3D2E1F0
};

/*
 * procedure sha1block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: sha1 - pointer to the sha1 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current sha1 result.
 */

static void CRYPTOCALL sha1block(WORD32 *sha1,WORD08 *data)
{
	register WORD32 i;	/* counter and index		*/
	register union
	{
		WORD32 j;	/* counter and index		*/
		WORD08 *b;	/* byte pointer			*/
		WORD32 *l;	/* work buffer pointer		*/
	} u;
	register WORD32 *m;	/* variables fast access	*/
	WORD32 mem[6];		/* work variables memory	*/
	WORD32 w[80];		/* work buffer			*/


	/* copy old result 32 bit clean to work variables */

	for(m=mem+1,u.l=sha1,i=5;i;i--)*m++=BITS32(*u.l++);

	/* get data bytes to work buffer (endian independent) */

	for(u.b=data,m=w,i=0;i<16;m++,i++)
	{
		*m=(WORD32)(*u.b++);
		*m<<=8;
		*m|=(WORD32)(*u.b++);
		*m<<=8;
		*m|=(WORD32)(*u.b++);
		*m<<=8;
		*m|=(WORD32)(*u.b++);
	}

	/* preprocess data bytes in work buffer */

	for(;i<80;m++,i++)
	{
		*m=m[-3]^m[-8]^m[-14]^m[-16];
		*m=BITS32(((*m)<<1)|((*m)>>31));
	}

	/* process work buffer with optimized f(B,C,D)=(B&C)|((~B)&D) */

	for(m=mem,i=0;i<20;i++)
	{
		m[0]=BITS32(((m[1]<<5)|(m[1]>>27))
		    +(((m[3]^m[4])&m[2])^m[4])
		    +m[5]+w[i]+0x5A827999);
		m[2]=(m[2]<<30)|(m[2]>>2);
		for(u.j=5;u.j;u.j--)m[u.j]=m[u.j-1];
	}

	/* process work buffer with f(B,C,D)=B^C^D */

	for(;i<40;i++)
	{
		m[0]=BITS32(((m[1]<<5)|(m[1]>>27))
		    +(m[2]^m[3]^m[4])
		    +m[5]+w[i]+0x6ED9EBA1);
		m[2]=(m[2]<<30)|(m[2]>>2);
		for(u.j=5;u.j;u.j--)m[u.j]=m[u.j-1];
	}

	/* process work buffer with optimized f(B,C,D)=(B&C)|(B&D)^(C&D) */

	for(;i<60;i++)
	{
		m[0]=BITS32(((m[1]<<5)|(m[1]>>27))
		    +((m[2]&m[3])|((m[2]|m[3])&m[4]))
		    +m[5]+w[i]+0x8F1BBCDC);
		m[2]=(m[2]<<30)|(m[2]>>2);
		for(u.j=5;u.j;u.j--)m[u.j]=m[u.j-1];
	}

	/* process work buffer with f(B,C,D)=B^C^D */

	for(;i<80;i++)
	{
		m[0]=BITS32(((m[1]<<5)|(m[1]>>27))
		    +(m[2]^m[3]^m[4])
		    +m[5]+w[i]+0xCA62C1D6);
		m[2]=(m[2]<<30)|(m[2]>>2);
		for(u.j=5;u.j;u.j--)m[u.j]=m[u.j-1];
	}

	/* create next hash result from previous result and work variables */

	for(m++,u.l=sha1,i=5;i;i--)*u.l+++=*m++;
}

/*
 * procedure sha1init
 *
 * input: ptr - pointer to SHA1 internal data
 *
 * This procedure initializes the SHA1 internal data structure.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1init(register SHA1DATA *ptr)
{
	register WORD32 i;	/* counter and index */


	/* initialize all SHA1 internal data */

	for(ptr->total=ptr->size=i=0;i<5;i++)ptr->sha1[i]=sha1start[i];
}

#endif

/*
 * procedure sha1next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to SHA1 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the SHA1 internal data structure.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1next(register WORD08 *data,register WORD32 length,
	register SHA1DATA *ptr)
{
	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed */

	for(;ptr->size&&length;length--)
	{
		/* copy current byte to buffer */

		ptr->bfr[(ptr->size)++]=*data++;

		/* if the buffer is full */

		if(ptr->size==64)
		{
			/* reset buffer size */

			ptr->size=0;

			/* hash buffer */

			sha1block(ptr->sha1,ptr->bfr);
		}
	}

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)
		sha1block(ptr->sha1,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr[(ptr->size)++]=*data++;
}

#endif

/*
 * procedure sha1end
 *
 * input:  ptr - pointer to SHA1 internal data
 *
 * output: result - the SHA1 hash
 *
 * This procedure finalizes and returns the SHA1 hash.
 */

#if defined(SHA1_FULL) || defined(SHA1_HMAC)

void CRYPTOCALL sha1end(WORD08 *result,SHA1DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/
	register WORD08 *p08=ptr->bfr+i;	/* byte rover		*/
	register WORD32 *sha1=ptr->sha1;	/* current hash		*/
	register WORD32 val;			/* length creation	*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	for(*p08++=0x80,i++;i!=56;*p08++=0,i++)if(i==64)
	{
		i=0;
		p08=ptr->bfr;
		sha1block(sha1,p08);
	}

	/* append bits 63 through 0 of total length in bits to buffer */

	for(i=3;i;i--)*p08++=0;
	for(val=ptr->total,i=32;i;i-=8)*p08++=(WORD08)(val>>(i-3));
	*p08=(WORD08)(val<<3);

	/* hash buffer */

	sha1block(sha1,ptr->bfr);

	/* return final hash result to caller */

	for(i=5,p08=result;i;sha1++,i--)
	{
		*p08++=(WORD08)((*sha1)>>24);
		*p08++=(WORD08)((*sha1)>>16);
		*p08++=(WORD08)((*sha1)>>8);
		*p08++=(WORD08)(*sha1);
	}

	/* assert user data removal data from buffer */

#ifdef SHA1_PARANOID
	for(i=64,p08=ptr->bfr;i;*p08++=0,i--);
#endif
}

#endif

/*
 * procedure sha1
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the SHA1 hash
 *
 * This procedure hashes the given data.
 */

#ifdef SHA1_FAST

void CRYPTOCALL sha1(register WORD08 *data,register WORD32 length,
	WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	union
	{
		WORD32 total;	/* data length buffer		*/
		WORD32 *ptr;	/* buffer fast access		*/
	}u;
	WORD32 sha1[5];		/* SHA1 current result buffer	*/
	WORD08 bfr[64];		/* final block(s) buffer	*/


	/* initialize SHA1 internal data */

	for(u.total=length,i=0;i<5;i++)sha1[i]=sha1start[i];

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)sha1block(sha1,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	data=bfr+i;
	for(*data++=0x80,i++;i!=56;*data++=0,i++)if(i==64)
	{
		i=0;
		data=bfr;
		sha1block(sha1,data);
	}

	/* append bits 63 through 0 of total length in bits to buffer */

	for(i=3;i;i--)*data++=0;
	for(i=32;i;i-=8)*data++=(WORD08)(u.total>>(i-3));
	*data=(WORD08)(u.total<<3);

	/* hash buffer */

	sha1block(sha1,bfr);

	/* return final hash result to caller */

	for(i=5,data=result,u.ptr=sha1;i;u.ptr++,i--)
	{
		*data++=(WORD08)((*u.ptr)>>24);
		*data++=(WORD08)((*u.ptr)>>16);
		*data++=(WORD08)((*u.ptr)>>8);
		*data++=(WORD08)(*u.ptr);
	}

	/* assert user data removal data from buffer */

#ifdef SHA1_PARANOID
	for(i=64,data=bfr;i;*data++=0,i--);
#endif

}

#endif

/*
 * procedure sha1hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef SHA1_HMAC

void CRYPTOCALL sha1hmkey(WORD08 *key,WORD32 keylength,SHA1HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	SHA1(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		SHA1DATA sha1data;	/* SHA1 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		sha1init(&u.sha1data);
		sha1next(key,keylength,&u.sha1data);
		sha1end(hash,&u.sha1data);
		key=hash;
		keylength=SHA1_SIZE;
		for(i=0;i<5;i++)u.sha1data.sha1[i]=0;
	}

	/* copy initial values to HMAC internal data structure */

	for(i=0;i<5;i++)ptr->isha1[i]=ptr->osha1[i]=sha1start[i];

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	sha1block(ptr->isha1,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	sha1block(ptr->osha1,u.pad);

	/* clean up */

	for(i=0;i<SHA1_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure sha1hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the SHA1 internal data
 *
 * This procedure initializes SHA1 data for HMAC use.
 */

#ifdef SHA1_HMAC_FULL

void CRYPTOCALL sha1hminit(SHA1DATA *ptr,SHA1HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* initialize all SHA1 internal data */

	for(ptr->total=64,ptr->size=i=0;i<5;i++)ptr->sha1[i]=key->isha1[i];
}

#endif

/*
 * procedure sha1hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the SHA1 internal data
 *
 * output: result - the HMAC result (a SHA1 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef SHA1_HMAC_FULL

void CRYPTOCALL sha1hmend(WORD08 *result,SHA1DATA *ptr,SHA1HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* complete inner SHA1 */

	sha1end(result,ptr);

	/* initialize outer SHA1 */

	for(ptr->total=64,ptr->size=i=0;i<5;i++)ptr->sha1[i]=key->osha1[i];

	sha1next(result,SHA1_SIZE,ptr);
	sha1end(result,ptr);
}

#endif

/*
 * procedure sha1hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a SHA1 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef SHA1_HMAC_FAST

void CRYPTOCALL sha1hmac(WORD08 *data,WORD32 length,WORD08 *result,
	SHA1HMDATA *key)
{
	register WORD32 i;	/* counter and index */
	SHA1DATA sha1data;	/* SHA1 internal data */


	/* initialize inner SHA1 */

	for(sha1data.total=64,sha1data.size=i=0;i<5;i++)
		sha1data.sha1[i]=key->isha1[i];

	/* hash given data */

	sha1next(data,length,&sha1data);

	/* complete inner SHA1 */

	sha1end(result,&sha1data);

	/* initialize outer SHA1 */

	for(sha1data.total=64,sha1data.size=i=0;i<5;i++)
		sha1data.sha1[i]=key->osha1[i];

	/* complete and return outer SHA1 */

	sha1next(result,SHA1_SIZE,&sha1data);
	sha1end(result,&sha1data);
}

#endif

