/*
 * MD5 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * MD5 algorithm by Ron Rivest, see RFC1321
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2202
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the MD5 routines.
 *
 * Data sizes:
 *
 * data block for MD5 transformation	WORD08[64]
 * resulting MD5 or HMAC hash		WORD08[16]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * MD5_FULL defined:
 *
 * 1. Call md5init
 * 2. For all data to be hashed call md5next
 * 3. To retrieve the hash call md5end
 *
 * MD5_FAST defined:
 *
 * Call md5 to get the hash for the specified data
 *
 * MD5_HMAC_FULL defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hminit
 * 3. For all data to be hashed call md5hmnext
 * 4. To retrieve the hash call md5hmend
 * 5. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 *
 * MD5_HMAC_FAST defined:
 *
 * 1. Call md5hmkey once to preprocess the selected key
 * 2. Call md5hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the MD5HMDATA structure
 */

#include "common.h"
#include "md5.h"

/* required constants for all 64 calculations */

static WORD32 md5tab[64]=
{
	0xd76aa478,0xe8c7b756,0x242070db,0xc1bdceee,
	0xf57c0faf,0x4787c62a,0xa8304613,0xfd469501,
	0x698098d8,0x8b44f7af,0xffff5bb1,0x895cd7be,
	0x6b901122,0xfd987193,0xa679438e,0x49b40821,
	0xf61e2562,0xc040b340,0x265e5a51,0xe9b6c7aa,
	0xd62f105d,0x02441453,0xd8a1e681,0xe7d3fbc8,
	0x21e1cde6,0xc33707d6,0xf4d50d87,0x455a14ed,
	0xa9e3e905,0xfcefa3f8,0x676f02d9,0x8d2a4c8a,
	0xfffa3942,0x8771f681,0x6d9d6122,0xfde5380c,
	0xa4beea44,0x4bdecfa9,0xf6bb4b60,0xbebfbc70,
	0x289b7ec6,0xeaa127fa,0xd4ef3085,0x04881d05,
	0xd9d4d039,0xe6db99e5,0x1fa27cf8,0xc4ac5665,
	0xf4292244,0x432aff97,0xab9423a7,0xfc93a039,
	0x655b59c3,0x8f0ccc92,0xffeff47d,0x85845dd1,
	0x6fa87e4f,0xfe2ce6e0,0xa3014314,0x4e0811a1,
	0xf7537e82,0xbd3af235,0x2ad7d2bb,0xeb86d391
};

/* offsets into 32 bit data array for all 64 calculations */

static WORD08 md5off[64]=
{
	0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,
	1,6,11,0,5,10,15,4,9,14,3,8,13,2,7,12,
	5,8,11,14,1,4,7,10,13,0,3,6,9,12,15,2,
	0,7,14,5,12,3,10,1,8,15,6,13,4,11,2,9
};

/* md5 result initial data */

static WORD32 md5start[4]={0x67452301,0xefcdab89,0x98badcfe,0x10325476};

/*
 * procedure md5block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: md5 - pointer to the md5 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current md5 result.
 */

static void CRYPTOCALL md5block(WORD32 *md5,WORD08 *block)
{
	register WORD32 *p32;	/* general purpose usage	*/
	register WORD32 *ptr;	/* md5 work buffer fast access	*/
	register WORD08 *p08;	/* general purpose usage	*/
	register WORD32 i;	/* loop counter			*/
	WORD32 bfr[4];		/* md5 work buffer		*/
	WORD32 data[16];	/* md5 data block buffer	*/


	/* get 64 byte message block into 32 bit array */

	for(i=0,p32=data,p08=block;i<16;i++,p32++)
	{
		*p32=(WORD32)(*p08++);
		*p32|=((WORD32)(*p08++))<<8;
		*p32|=((WORD32)(*p08++))<<16;
		*p32|=((WORD32)(*p08++))<<24;
	}

	/* get previous md5 block result into working buffer */

	for(i=0,ptr=bfr,p32=md5;i<4;i++)*ptr++=*p32++;

	/* do partially unrolled round one */

	for(i=4,ptr=bfr,p32=md5tab,p08=md5off;i;i--)
	{
		ptr[0]+=(ptr[3]^(ptr[1]&(ptr[2]^ptr[3])))+data[*p08++]+*p32++;
		ptr[0]=((ptr[0]<<7)|(BITS32(ptr[0])>>25))+ptr[1];
		ptr[3]+=(ptr[2]^(ptr[0]&(ptr[1]^ptr[2])))+data[*p08++]+*p32++;
		ptr[3]=((ptr[3]<<12)|(BITS32(ptr[3])>>20))+ptr[0];
		ptr[2]+=(ptr[1]^(ptr[3]&(ptr[0]^ptr[1])))+data[*p08++]+*p32++;
		ptr[2]=((ptr[2]<<17)|(BITS32(ptr[2])>>15))+ptr[3];
		ptr[1]+=(ptr[0]^(ptr[2]&(ptr[3]^ptr[0])))+data[*p08++]+*p32++;
		ptr[1]=((ptr[1]<<22)|(BITS32(ptr[1])>>10))+ptr[2];
	}

	/* do partially unrolled round two */

	for(i=4;i;i--)
	{
		ptr[0]+=(ptr[2]^(ptr[3]&(ptr[1]^ptr[2])))+data[*p08++]+*p32++;
		ptr[0]=((ptr[0]<<5)|(BITS32(ptr[0])>>27))+ptr[1];
		ptr[3]+=(ptr[1]^(ptr[2]&(ptr[0]^ptr[1])))+data[*p08++]+*p32++;
		ptr[3]=((ptr[3]<<9)|(BITS32(ptr[3])>>23))+ptr[0];
		ptr[2]+=(ptr[0]^(ptr[1]&(ptr[3]^ptr[0])))+data[*p08++]+*p32++;
		ptr[2]=((ptr[2]<<14)|(BITS32(ptr[2])>>18))+ptr[3];
		ptr[1]+=(ptr[3]^(ptr[0]&(ptr[2]^ptr[3])))+data[*p08++]+*p32++;
		ptr[1]=((ptr[1]<<20)|(BITS32(ptr[1])>>12))+ptr[2];
	}

	/* do partially unrolled round three */

	for(i=4;i;i--)
	{
		ptr[0]+=(ptr[1]^ptr[2]^ptr[3])+data[*p08++]+*p32++;
		ptr[0]=((ptr[0]<<4)|(BITS32(ptr[0])>>28))+ptr[1];
		ptr[3]+=(ptr[0]^ptr[1]^ptr[2])+data[*p08++]+*p32++;
		ptr[3]=((ptr[3]<<11)|(BITS32(ptr[3])>>21))+ptr[0];
		ptr[2]+=(ptr[3]^ptr[0]^ptr[1])+data[*p08++]+*p32++;
		ptr[2]=((ptr[2]<<16)|(BITS32(ptr[2])>>16))+ptr[3];
		ptr[1]+=(ptr[2]^ptr[3]^ptr[0])+data[*p08++]+*p32++;
		ptr[1]=((ptr[1]<<23)|(BITS32(ptr[1])>>9))+ptr[2];
	}

	/* do partially unrolled round four */

	for(i=4;i;i--)
	{
		ptr[0]+=(ptr[2]^(ptr[1]|~ptr[3]))+data[*p08++]+*p32++;
		ptr[0]=((ptr[0]<<6)|(BITS32(ptr[0])>>26))+ptr[1];
		ptr[3]+=(ptr[1]^(ptr[0]|~ptr[2]))+data[*p08++]+*p32++;
		ptr[3]=((ptr[3]<<10)|(BITS32(ptr[3])>>22))+ptr[0];
		ptr[2]+=(ptr[0]^(ptr[3]|~ptr[1]))+data[*p08++]+*p32++;
		ptr[2]=((ptr[2]<<15)|(BITS32(ptr[2])>>17))+ptr[3];
		ptr[1]+=(ptr[3]^(ptr[2]|~ptr[0]))+data[*p08++]+*p32++;
		ptr[1]=((ptr[1]<<21)|(BITS32(ptr[1])>>11))+ptr[2];
	}

	/* create new md5 block result */

	for(i=0,ptr=bfr,p32=md5;i<4;i++)*p32+++=*ptr++;
}

/*
 * procedure md5init
 *
 * input: ptr - pointer to MD5 internal data
 *
 * This procedure initializes the MD5 internal data structure.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5init(register MD5DATA *ptr)
{
	register WORD32 i;	/* counter and index */


	/* initialize all MD5 internal data */

	for(ptr->total=ptr->size=i=0;i<4;i++)ptr->md5[i]=md5start[i];
}

#endif

/*
 * procedure md5next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to MD5 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the MD5 internal data structure.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5next(register WORD08 *data,register WORD32 length,
	register MD5DATA *ptr)
{
	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed */

	for(;ptr->size&&length;length--)
	{
		/* copy current byte to buffer */

		ptr->bfr[(ptr->size)++]=*data++;

		/* if the buffer is full */

		if(ptr->size==64)
		{
			/* reset buffer size */

			ptr->size=0;

			/* hash buffer */

			md5block(ptr->md5,ptr->bfr);
		}
	}

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)
		md5block(ptr->md5,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr[(ptr->size)++]=*data++;
}

#endif

/*
 * procedure md5end
 *
 * input:  ptr - pointer to MD5 internal data
 *
 * output: result - the MD5 hash
 *
 * This procedure finalizes and returns the MD5 hash.
 */

#if defined(MD5_FULL) || defined(MD5_HMAC)

void CRYPTOCALL md5end(WORD08 *result,MD5DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/
	register WORD08 *p08=ptr->bfr+i;	/* byte rover		*/
	register WORD32 *md5=ptr->md5;		/* current hash		*/
	register WORD32 val;			/* length creation	*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	for(*p08++=0x80,i++;i!=56;*p08++=0,i++)if(i==64)
	{
		i=0;
		p08=ptr->bfr;
		md5block(md5,p08);
	}

	/* append bits 0 through 31 of total length in bits to buffer */

	for(i=4,val=(ptr->total)<<3;i;val>>=8,i--)*p08++=(WORD08)(val);

	/* append bits 32 through 63 of total length in bits to buffer */

	*p08++=(WORD08)(BITS32(ptr->total)>>29);
	*p08++=0;
	*p08++=0;
	*p08=0;

	/* hash buffer */

	md5block(md5,ptr->bfr);

	/* return final hash result to caller */

	for(i=4,p08=result;i;md5++,i--)
	{
		*p08++=(WORD08)(*md5);
		*p08++=(WORD08)((*md5)>>8);
		*p08++=(WORD08)((*md5)>>16);
		*p08++=(WORD08)((*md5)>>24);
	}

	/* assert user data removal data from buffer */

#ifdef MD5_PARANOID
	for(i=64,p08=ptr->bfr;i;*p08++=0,i--);
#endif
}

#endif

/*
 * procedure md5
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the MD5 hash
 *
 * This procedure hashes the given data.
 */

#ifdef MD5_FAST

void CRYPTOCALL md5(register WORD08 *data,register WORD32 length,WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	union
	{
		WORD32 total;	/* data length buffer		*/
		WORD32 *ptr;	/* fast buffer access		*/
	}u;
	WORD32 md5[4];		/* MD5 current result buffer	*/
	WORD08 bfr[64];		/* final block(s) buffer	*/


	/* initialize MD5 internal data */

	for(u.total=length,i=0;i<4;i++)md5[i]=md5start[i];

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)md5block(md5,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	data=bfr+i;
	for(*data++=0x80,i++;i!=56;*data++=0,i++)if(i==64)
	{
		i=0;
		data=bfr;
		md5block(md5,data);
	}

	/* append bits 0 through 31 of total length in bits to buffer */

	for(i=4,length=u.total<<3;i;length>>=8,i--)*data++=(WORD08)(length);

	/* append bits 32 through 63 of total length in bits to buffer */

	*data++=(WORD08)(BITS32(u.total)>>29);
	*data++=0;
	*data++=0;
	*data=0;

	/* hash buffer */

	md5block(md5,bfr);

	/* return final hash result to caller */

	for(i=4,data=result,u.ptr=md5;i;u.ptr++,i--)
	{
		*data++=(WORD08)(*u.ptr);
		*data++=(WORD08)((*u.ptr)>>8);
		*data++=(WORD08)((*u.ptr)>>16);
		*data++=(WORD08)((*u.ptr)>>24);
	}

	/* assert user data removal data from buffer */

#ifdef MD5_PARANOID
	for(i=64,data=bfr;i;*data++=0,i--);
#endif

}

#endif

/*
 * procedure md5hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef MD5_HMAC

void CRYPTOCALL md5hmkey(WORD08 *key,WORD32 keylength,MD5HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	MD5(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		MD5DATA md5data;	/* MD5 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		md5init(&u.md5data);
		md5next(key,keylength,&u.md5data);
		md5end(hash,&u.md5data);
		key=hash;
		keylength=MD5_SIZE;
		for(i=0;i<4;i++)u.md5data.md5[i]=0;
	}

	/* copy initial values to HMAC internal data structure */

	for(i=0;i<4;i++)ptr->imd5[i]=ptr->omd5[i]=md5start[i];

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	md5block(ptr->imd5,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	md5block(ptr->omd5,u.pad);

	/* clean up */

	for(i=0;i<MD5_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure md5hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the MD5 internal data
 *
 * This procedure initializes MD5 data for HMAC use.
 */

#ifdef MD5_HMAC_FULL

void CRYPTOCALL md5hminit(MD5DATA *ptr,MD5HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* initialize all MD5 internal data */

	for(ptr->total=64,ptr->size=i=0;i<4;i++)ptr->md5[i]=key->imd5[i];
}

#endif

/*
 * procedure md5hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the MD5 internal data
 *
 * output: result - the HMAC result (a MD5 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef MD5_HMAC_FULL

void CRYPTOCALL md5hmend(WORD08 *result,MD5DATA *ptr,MD5HMDATA *key)
{
	register WORD32 i;	/* counter and index */


	/* complete inner MD5 */

	md5end(result,ptr);

	/* initialize outer MD5 */

	for(ptr->total=64,ptr->size=i=0;i<4;i++)ptr->md5[i]=key->omd5[i];

	md5next(result,MD5_SIZE,ptr);
	md5end(result,ptr);
}

#endif

/*
 * procedure md5hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a MD5 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef MD5_HMAC_FAST

void CRYPTOCALL md5hmac(WORD08 *data,WORD32 length,WORD08 *result,
	MD5HMDATA *key)
{
	register WORD32 i;	/* counter and index */
	MD5DATA md5data;	/* MD5 internal data */


	/* initialize inner MD5 */

	for(md5data.total=64,md5data.size=i=0;i<4;i++)
		md5data.md5[i]=key->imd5[i];

	/* hash given data */

	md5next(data,length,&md5data);

	/* complete inner MD5 */

	md5end(result,&md5data);

	/* initialize outer MD5 */

	for(md5data.total=64,md5data.size=i=0;i<4;i++)
		md5data.md5[i]=key->omd5[i];

	/* complete and return outer MD5 */

	md5next(result,MD5_SIZE,&md5data);
	md5end(result,&md5data);
}

#endif

