/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* net-o-rama: ip over ip (dis)connection functionality                       */
/*============================================================================*/

/*
 * tcpsocket
 *
 * input: anchor - pointer to vpnd global data
 *
 * return: 0 - success
 *	   1 - no socket
 *	   2 - bind failed
 *	   3 - set socket option or socket ioctl failed
 *	   4 - connect/listen failed
 *
 * This procedure establishes a tcp socket in client or server mode.
 */

int tcpsocket(VPN *anchor)
{
	int s;			/* socket handle		*/
	struct sockaddr_in a;	/* connection data		*/
	int x;			/* used for socket ioctl	*/
	int y;			/* used for socket ioctl	*/
	int err;		/* return code			*/
	struct linger l;	/* delay on closing socket	*/
	fd_set w;		/* used to wait for connect	*/
	struct timeval tv;	/* used to expire connect	*/


	/* debug message */

	ENTER("tcpsocket");

	/* reset return value */

	err=0;

	/* create socket, handle errors */

	if((s=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP))==-1)JUMP("socket",err1);

	/* bind socket if port given (allow address reuse), handle errors */

	y=anchor->mode?anchor->client_port:anchor->server_port;

	if(y)
	{
		x=1;
		if(setsockopt(s,SOL_SOCKET,SO_REUSEADDR,&x,sizeof(x)))
			JUMP("setsockopt(SO_REUSEADDR)",err2);
		a.sin_family=AF_INET;
		a.sin_addr.s_addr=
		    inet_addr(anchor->mode?anchor->client_ip:anchor->server_ip);
		a.sin_port=htons(y);
		if(bind(s,(struct sockaddr *)(&a),sizeof(a)))JUMP("bind",err2);
	}

	/* set linger on close time, handle errors */

	l.l_onoff=1;
	l.l_linger=10;
	if(setsockopt(s,SOL_SOCKET,SO_LINGER,&l,sizeof(l)))
		JUMP("setsockopt(SO_LINGER)",err3);

	/*  set send buffer size if requested */

	if(anchor->sendbuf)
	    if(setsockopt(s,SOL_SOCKET,SO_SNDBUF,&(anchor->sendbuf),
		sizeof(anchor->sendbuf)))JUMP("setsockopt(SO_SNDBUF)",err3);

	/* prevent packet coalescence */

	x=1;
	if(setsockopt(s,SOL_TCP,TCP_NODELAY,&x,sizeof(x)))
		JUMP("setsockopt(TCP_NODELAY)",err3);

	/* set type of service and precedence, if given
	   (if any router actually uses these the precedence
	    is restricted to 'priority' below to prevent
	    chaos - do NOT change this! */

	/* set type of service and precedence if requested */

	if(anchor->ipopts)
	{
		x=((anchor->ipopts)<<2)&0x3c;
		if(setsockopt(s,SOL_IP,IP_TOS,&x,sizeof(x)))
			JUMP("setsockopt(IP_TOS)",err3);
	}

	/* set socket to nonblocking, handle errors */

	x=1;
	if(ioctl(s,FIONBIO,&x))JUMP("ioctl(FIONBIO)",err3);

	/* client mode */

	if(anchor->mode)
	{
		/* connect to peer, handle errors */

		a.sin_family=AF_INET;
		a.sin_addr.s_addr=inet_addr(anchor->server_ip);
		a.sin_port=htons(anchor->server_port);

		if(connect(s,(struct sockaddr *)(&a),sizeof(a)))
		{
			/* if connection has not yet been completed */

			if(errno==EINPROGRESS)
			{
				/* set up socket handle for select call */

				FD_ZERO(&w);
				FD_SET(s,&w);

				/* set up timeout value */

				tv.tv_usec=0;
				tv.tv_sec=anchor->connwait;

				/* signal error, if socket is not writeable on
				   return from select */

				if(select(s+1,NULL,&w,NULL,
					anchor->connwait?&tv:NULL)!=1)
						JUMP("select",err4);

				/* get connect completion code, signal error if
				   not possible */

				y=sizeof(x);
				if(getsockopt(s,SOL_SOCKET,SO_ERROR,&x,&y))
					JUMP("getsockopt",err4);

				/* signal error, if connect did not succeed */

				if(x)GOTO("socket error",err4);
			}

			/* signal error in case of all other errors */

			else JUMP("connect",err4);
		}

		/* make peer handle globally available */

		anchor->peer=s;
	}

	/* server mode */

	else
	{
		/* make socket listen, handle errors */

		if(listen(s,5))JUMP("listen",err4);

		/* make handle globally available */

		anchor->server=s;
	}

	/* debug message */

	LEAVE("tcpsocket");

	/* signal success */

	goto err0;

err4:	err++;
err3:	err++;
err2:	err++;
	close(s);
err1:	err++;
err0:	return err;
}

/*
 * accept_socket
 *
 * input:  anchor - pointer to vpnd global data
 *
 * output: addr - the address of the connectiong peer
 *
 * return: 0 - success
 *	   1 - accept of connection failed
 *	   2 - unexpected peer connected
 *
 * This procedure accepts a client connection and
 * checks, that the client is the expected one.
 */

int accept_socket(struct sockaddr_in *addr,VPN *anchor)
{
	struct sockaddr_in a;	/* connection data		*/
	int x;			/* used for socket ioctl	*/
	int err;		/* return code			*/


	/* debug message */

	ENTER("accept_socket");

	/* reset return value */

	err=0;

	/* try to accept the connection */

	x=sizeof(a);
	if((anchor->peer=accept(anchor->server,(struct sockaddr *)(&a),&x))==-1)
		JUMP("accept",err1);

	/* return peer info to caller */

	*addr=a;

	/* if it's not the expected peer */

	if((a.sin_addr.s_addr!=inet_addr(anchor->client_ip)&&
		inet_addr(anchor->client_ip))||
		(anchor->client_port&&a.sin_port!=htons(anchor->client_port)))
		GOTO("illegal peer",err2);

	/* debug message */

	LEAVE("accept_socket");

	/* signal success */

	goto err0;

	/* error handling */

err2:	err++;
	close(anchor->peer);
	anchor->peer=-1;
err1:	err++;
err0:	return err;
}

/*
 * disconnect
 *
 * input: dmode  - 0 for peer socket
 *	         - 1 for server socket
 *	         - 2 for peer and server socket
 *	  anchor - pointer to vpnd global data
 *
 *
 * This procedure disconnects the tcp/ip connection to the remote peer
 * or the listening server socket.
 */

void disconnect(int dmode,VPN *anchor)
{
	/* debug message */

	ENTER("disconnect");

	/* if peer socket and if a connection is established */

	if(!(dmode&1))if(anchor->peer!=-1)
	{
		/* if not in serial line mode */

		if(!anchor->serialmode)
		{
			/* send TCP_FIN */

			if(shutdown(anchor->peer,2))ERRNO("shutdown");

			/* close communication channel */

			if(close(anchor->peer))ERRNO("close");
		}

		/* in serial line mode close serial line */

		else serialhandler(anchor);

		/* execute linkdown process, if there is any */

		doexec(anchor->linkdown,NULL);

		/* reset handle to no peer */

		anchor->peer=-1;
	}

	/* if server socket and if the server socket exists */

	if(dmode)if(anchor->server!=-1)
	{
		/* close server socket */

		if(close(anchor->server))ERRNO("close");

		/* mark server socket as closed */

		anchor->server=-1;
	}

	/* debug message */

	LEAVE("disconnect");
}
