/*
 * RIPEMD-160 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * RIPEMD-160 algorithm by Hans Dobbertin, Antoon Bosselaers and Bart Preneel,
 * see http://www.esat.kuleuven.ac.be/~bosselae/ripemd160.html
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2286
 *
 * License:
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the RIPEMD-160 routines.
 *
 * Data sizes:
 *
 * data block for RIPEMD-160 transformation	WORD08[64]
 * resulting RIPEMD-160 or HMAC hash		WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * RMD160_FULL defined:
 *
 * 1. Call rmd160init
 * 2. For all data to be hashed call rmd160next
 * 3. To retrieve the hash call rmd160end
 *
 * RMD160_FAST defined:
 *
 * Call rmd160 to get the hash for the specified data
 *
 * RMD160_HMAC_FULL defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hminit
 * 3. For all data to be hashed call rmd160hmnext
 * 4. To retrieve the hash call rmd160hmend
 * 5. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 *
 * RMD160_HMAC_FAST defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 */

#include "common.h"
#include "rmd160.h"

/*
 * procedure rmd160block
 *
 * input: block - pointer to the 64 byte message block
 *
 * inout: rmd160 - pointer to the rmd160 working buffer
 *
 * This procedure processes the 64 byte message block
 * and updates the current rmd160 result.
 */

static void CRYPTOCALL rmd160block(WORD32 *rmd160,WORD08 *data)
{
	register WORD32 a;	/* work buffer	*/
	register WORD32 b;	/* work buffer	*/
	register WORD32 c;	/* work buffer	*/
	register WORD32 d;	/* work buffer	*/
	register WORD32 e;	/* work buffer	*/
	WORD32 w[21];		/* work buffer	*/


	/* get 64 byte message block to work buffer */

	a=data[3];
	a<<=8;
	a+=data[2];
	a<<=8;
	a+=data[1];
	a<<=8;
	w[0]=a+data[0];

	a=data[7];
	a<<=8;
	a+=data[6];
	a<<=8;
	a+=data[5];
	a<<=8;
	w[1]=a+data[4];

	a=data[11];
	a<<=8;
	a+=data[10];
	a<<=8;
	a+=data[9];
	a<<=8;
	w[2]=a+data[8];

	a=data[15];
	a<<=8;
	a+=data[14];
	a<<=8;
	a+=data[13];
	a<<=8;
	w[3]=a+data[12];

	a=data[19];
	a<<=8;
	a+=data[18];
	a<<=8;
	a+=data[17];
	a<<=8;
	w[4]=a+data[16];

	a=data[23];
	a<<=8;
	a+=data[22];
	a<<=8;
	a+=data[21];
	a<<=8;
	w[5]=a+data[20];

	a=data[27];
	a<<=8;
	a+=data[26];
	a<<=8;
	a+=data[25];
	a<<=8;
	w[6]=a+data[24];

	a=data[31];
	a<<=8;
	a+=data[30];
	a<<=8;
	a+=data[29];
	a<<=8;
	w[7]=a+data[28];

	a=data[35];
	a<<=8;
	a+=data[34];
	a<<=8;
	a+=data[33];
	a<<=8;
	w[8]=a+data[32];

	a=data[39];
	a<<=8;
	a+=data[38];
	a<<=8;
	a+=data[37];
	a<<=8;
	w[9]=a+data[36];

	a=data[43];
	a<<=8;
	a+=data[42];
	a<<=8;
	a+=data[41];
	a<<=8;
	w[10]=a+data[40];

	a=data[47];
	a<<=8;
	a+=data[46];
	a<<=8;
	a+=data[45];
	a<<=8;
	w[11]=a+data[44];

	a=data[51];
	a<<=8;
	a+=data[50];
	a<<=8;
	a+=data[49];
	a<<=8;
	w[12]=a+data[48];

	a=data[55];
	a<<=8;
	a+=data[54];
	a<<=8;
	a+=data[53];
	a<<=8;
	w[13]=a+data[52];

	a=data[59];
	a<<=8;
	a+=data[58];
	a<<=8;
	a+=data[57];
	a<<=8;
	w[14]=a+data[56];

	a=data[63];
	a<<=8;
	a+=data[62];
	a<<=8;
	a+=data[61];
	a<<=8;
	w[15]=a+data[60];

	/* load previous rmd160 block result */

	a=rmd160[0];
	b=rmd160[1];
	c=rmd160[2];
	d=rmd160[3];
	e=rmd160[4];

	/* round one */

	a+=w[0];
	a+=b^c^d;
	a=(a<<11)|(BITS32(a)>>21);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[1];
	e+=a^b^c;
	e=(e<<14)|(BITS32(e)>>18);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[2];
	d+=e^a^b;
	d=(d<<15)|(BITS32(d)>>17);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[3];
	c+=d^e^a;
	c=(c<<12)|(BITS32(c)>>20);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[4];
	b+=c^d^e;
	b=(b<<5)|(BITS32(b)>>27);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[5];
	a+=b^c^d;
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[6];
	e+=a^b^c;
	e=(e<<7)|(BITS32(e)>>25);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[7];
	d+=e^a^b;
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[8];
	c+=d^e^a;
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[9];
	b+=c^d^e;
	b=(b<<13)|(BITS32(b)>>19);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[10];
	a+=b^c^d;
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[11];
	e+=a^b^c;
	e=(e<<15)|(BITS32(e)>>17);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[12];
	d+=e^a^b;
	d=(d<<6)|(BITS32(d)>>26);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[13];
	c+=d^e^a;
	c=(c<<7)|(BITS32(c)>>25);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[14];
	b+=c^d^e;
	b=(b<<9)|(BITS32(b)>>23);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[15];
	a+=b^c^d;
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	/* round two */

	e+=w[7];
	e+=0x5a827999;
	e+=c^(a&(b^c));
	e=(e<<7)|(BITS32(e)>>25);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[4];
	d+=0x5a827999;
	d+=b^(e&(a^b));
	d=(d<<6)|(BITS32(d)>>26);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[13];
	c+=0x5a827999;
	c+=a^(d&(e^a));
	c=(c<<8)|(BITS32(c)>>24);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[1];
	b+=0x5a827999;
	b+=e^(c&(d^e));
	b=(b<<13)|(BITS32(b)>>19);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[10];
	a+=0x5a827999;
	a+=d^(b&(c^d));
	a=(a<<11)|(BITS32(a)>>21);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[6];
	e+=0x5a827999;
	e+=c^(a&(b^c));
	e=(e<<9)|(BITS32(e)>>23);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[15];
	d+=0x5a827999;
	d+=b^(e&(a^b));
	d=(d<<7)|(BITS32(d)>>25);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[3];
	c+=0x5a827999;
	c+=a^(d&(e^a));
	c=(c<<15)|(BITS32(c)>>17);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[12];
	b+=0x5a827999;
	b+=e^(c&(d^e));
	b=(b<<7)|(BITS32(b)>>25);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[0];
	a+=0x5a827999;
	a+=d^(b&(c^d));
	a=(a<<12)|(BITS32(a)>>20);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[9];
	e+=0x5a827999;
	e+=c^(a&(b^c));
	e=(e<<15)|(BITS32(e)>>17);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[5];
	d+=0x5a827999;
	d+=b^(e&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[2];
	c+=0x5a827999;
	c+=a^(d&(e^a));
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[14];
	b+=0x5a827999;
	b+=e^(c&(d^e));
	b=(b<<7)|(BITS32(b)>>25);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[11];
	a+=0x5a827999;
	a+=d^(b&(c^d));
	a=(a<<13)|(BITS32(a)>>19);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[8];
	e+=0x5a827999;
	e+=c^(a&(b^c));
	e=(e<<12)|(BITS32(e)>>20);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	/* round three */

	d+=w[3];
	d+=0x6ed9eba1;
	d+=(e|~a)^b;
	d=(d<<11)|(BITS32(d)>>21);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[10];
	c+=0x6ed9eba1;
	c+=(d|~e)^a;
	c=(c<<13)|(BITS32(c)>>19);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[14];
	b+=0x6ed9eba1;
	b+=(c|~d)^e;
	b=(b<<6)|(BITS32(b)>>26);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[4];
	a+=0x6ed9eba1;
	a+=(b|~c)^d;
	a=(a<<7)|(BITS32(a)>>25);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[9];
	e+=0x6ed9eba1;
	e+=(a|~b)^c;
	e=(e<<14)|(BITS32(e)>>18);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[15];
	d+=0x6ed9eba1;
	d+=(e|~a)^b;
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[8];
	c+=0x6ed9eba1;
	c+=(d|~e)^a;
	c=(c<<13)|(BITS32(c)>>19);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[1];
	b+=0x6ed9eba1;
	b+=(c|~d)^e;
	b=(b<<15)|(BITS32(b)>>17);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[2];
	a+=0x6ed9eba1;
	a+=(b|~c)^d;
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[7];
	e+=0x6ed9eba1;
	e+=(a|~b)^c;
	e=(e<<8)|(BITS32(e)>>24);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[0];
	d+=0x6ed9eba1;
	d+=(e|~a)^b;
	d=(d<<13)|(BITS32(d)>>19);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[6];
	c+=0x6ed9eba1;
	c+=(d|~e)^a;
	c=(c<<6)|(BITS32(c)>>26);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[13];
	b+=0x6ed9eba1;
	b+=(c|~d)^e;
	b=(b<<5)|(BITS32(b)>>27);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[11];
	a+=0x6ed9eba1;
	a+=(b|~c)^d;
	a=(a<<12)|(BITS32(a)>>20);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[5];
	e+=0x6ed9eba1;
	e+=(a|~b)^c;
	e=(e<<7)|(BITS32(e)>>25);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[12];
	d+=0x6ed9eba1;
	d+=(e|~a)^b;
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	/* round four */

	c+=w[1];
	c+=0x8f1bbcdc;
	c+=e^(a&(d^e));
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[9];
	b+=0x8f1bbcdc;
	b+=d^(e&(c^d));
	b=(b<<12)|(BITS32(b)>>20);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[11];
	a+=0x8f1bbcdc;
	a+=c^(d&(b^c));
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[10];
	e+=0x8f1bbcdc;
	e+=b^(c&(a^b));
	e=(e<<15)|(BITS32(e)>>17);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[0];
	d+=0x8f1bbcdc;
	d+=a^(b&(e^a));
	d=(d<<14)|(BITS32(d)>>18);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[8];
	c+=0x8f1bbcdc;
	c+=e^(a&(d^e));
	c=(c<<15)|(BITS32(c)>>17);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[12];
	b+=0x8f1bbcdc;
	b+=d^(e&(c^d));
	b=(b<<9)|(BITS32(b)>>23);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[4];
	a+=0x8f1bbcdc;
	a+=c^(d&(b^c));
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[13];
	e+=0x8f1bbcdc;
	e+=b^(c&(a^b));
	e=(e<<9)|(BITS32(e)>>23);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[3];
	d+=0x8f1bbcdc;
	d+=a^(b&(e^a));
	d=(d<<14)|(BITS32(d)>>18);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[7];
	c+=0x8f1bbcdc;
	c+=e^(a&(d^e));
	c=(c<<5)|(BITS32(c)>>27);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[15];
	b+=0x8f1bbcdc;
	b+=d^(e&(c^d));
	b=(b<<6)|(BITS32(b)>>26);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[14];
	a+=0x8f1bbcdc;
	a+=c^(d&(b^c));
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[5];
	e+=0x8f1bbcdc;
	e+=b^(c&(a^b));
	e=(e<<6)|(BITS32(e)>>26);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[6];
	d+=0x8f1bbcdc;
	d+=a^(b&(e^a));
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[2];
	c+=0x8f1bbcdc;
	c+=e^(a&(d^e));
	c=(c<<12)|(BITS32(c)>>20);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	/* round five */

	b+=w[4];
	b+=0xa953fd4e;
	b+=c^(d|~e);
	b=(b<<9)|(BITS32(b)>>23);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[0];
	a+=0xa953fd4e;
	a+=b^(c|~d);
	a=(a<<15)|(BITS32(a)>>17);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[5];
	e+=0xa953fd4e;
	e+=a^(b|~c);
	e=(e<<5)|(BITS32(e)>>27);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[9];
	d+=0xa953fd4e;
	d+=e^(a|~b);
	d=(d<<11)|(BITS32(d)>>21);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[7];
	c+=0xa953fd4e;
	c+=d^(e|~a);
	c=(c<<6)|(BITS32(c)>>26);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[12];
	b+=0xa953fd4e;
	b+=c^(d|~e);
	b=(b<<8)|(BITS32(b)>>24);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[2];
	a+=0xa953fd4e;
	a+=b^(c|~d);
	a=(a<<13)|(BITS32(a)>>19);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[10];
	e+=0xa953fd4e;
	e+=a^(b|~c);
	e=(e<<12)|(BITS32(e)>>20);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[14];
	d+=0xa953fd4e;
	d+=e^(a|~b);
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[1];
	c+=0xa953fd4e;
	c+=d^(e|~a);
	c=(c<<12)|(BITS32(c)>>20);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[3];
	b+=0xa953fd4e;
	b+=c^(d|~e);
	b=(b<<13)|(BITS32(b)>>19);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[8];
	a+=0xa953fd4e;
	a+=b^(c|~d);
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[11];
	e+=0xa953fd4e;
	e+=a^(b|~c);
	e=(e<<11)|(BITS32(e)>>21);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[6];
	d+=0xa953fd4e;
	d+=e^(a|~b);
	d=(d<<8)|(BITS32(d)>>24);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[15];
	c+=0xa953fd4e;
	c+=d^(e|~a);
	c=(c<<5)|(BITS32(c)>>27);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[13];
	b+=0xa953fd4e;
	b+=c^(d|~e);
	b=(b<<6)|(BITS32(b)>>26);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	/* store intermediate result */

	w[16]=a;
	w[17]=b;
	w[18]=c;
	w[19]=d;
	w[20]=e;

	/* load previous rmd160 block result */

	a=rmd160[0];
	b=rmd160[1];
	c=rmd160[2];
	d=rmd160[3];
	e=rmd160[4];

	/* parallel round one */

	a+=w[5];
	a+=0x50a28be6;
	a+=b^(c|~d);
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[14];
	e+=0x50a28be6;
	e+=a^(b|~c);
	e=(e<<9)|(BITS32(e)>>23);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[7];
	d+=0x50a28be6;
	d+=e^(a|~b);
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[0];
	c+=0x50a28be6;
	c+=d^(e|~a);
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[9];
	b+=0x50a28be6;
	b+=c^(d|~e);
	b=(b<<13)|(BITS32(b)>>19);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[2];
	a+=0x50a28be6;
	a+=b^(c|~d);
	a=(a<<15)|(BITS32(a)>>17);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[11];
	e+=0x50a28be6;
	e+=a^(b|~c);
	e=(e<<15)|(BITS32(e)>>17);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[4];
	d+=0x50a28be6;
	d+=e^(a|~b);
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[13];
	c+=0x50a28be6;
	c+=d^(e|~a);
	c=(c<<7)|(BITS32(c)>>25);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[6];
	b+=0x50a28be6;
	b+=c^(d|~e);
	b=(b<<7)|(BITS32(b)>>25);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[15];
	a+=0x50a28be6;
	a+=b^(c|~d);
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[8];
	e+=0x50a28be6;
	e+=a^(b|~c);
	e=(e<<11)|(BITS32(e)>>21);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[1];
	d+=0x50a28be6;
	d+=e^(a|~b);
	d=(d<<14)|(BITS32(d)>>18);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[10];
	c+=0x50a28be6;
	c+=d^(e|~a);
	c=(c<<14)|(BITS32(c)>>18);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[3];
	b+=0x50a28be6;
	b+=c^(d|~e);
	b=(b<<12)|(BITS32(b)>>20);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[12];
	a+=0x50a28be6;
	a+=b^(c|~d);
	a=(a<<6)|(BITS32(a)>>26);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	/* parallel round two */

	e+=w[6];
	e+=0x5c4dd124;
	e+=b^(c&(a^b));
	e=(e<<9)|(BITS32(e)>>23);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[11];
	d+=0x5c4dd124;
	d+=a^(b&(e^a));
	d=(d<<13)|(BITS32(d)>>19);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[3];
	c+=0x5c4dd124;
	c+=e^(a&(d^e));
	c=(c<<15)|(BITS32(c)>>17);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[7];
	b+=0x5c4dd124;
	b+=d^(e&(c^d));
	b=(b<<7)|(BITS32(b)>>25);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[0];
	a+=0x5c4dd124;
	a+=c^(d&(b^c));
	a=(a<<12)|(BITS32(a)>>20);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[13];
	e+=0x5c4dd124;
	e+=b^(c&(a^b));
	e=(e<<8)|(BITS32(e)>>24);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[5];
	d+=0x5c4dd124;
	d+=a^(b&(e^a));
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[10];
	c+=0x5c4dd124;
	c+=e^(a&(d^e));
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[14];
	b+=0x5c4dd124;
	b+=d^(e&(c^d));
	b=(b<<7)|(BITS32(b)>>25);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[15];
	a+=0x5c4dd124;
	a+=c^(d&(b^c));
	a=(a<<7)|(BITS32(a)>>25);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[8];
	e+=0x5c4dd124;
	e+=b^(c&(a^b));
	e=(e<<12)|(BITS32(e)>>20);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[12];
	d+=0x5c4dd124;
	d+=a^(b&(e^a));
	d=(d<<7)|(BITS32(d)>>25);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[4];
	c+=0x5c4dd124;
	c+=e^(a&(d^e));
	c=(c<<6)|(BITS32(c)>>26);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[9];
	b+=0x5c4dd124;
	b+=d^(e&(c^d));
	b=(b<<15)|(BITS32(b)>>17);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[1];
	a+=0x5c4dd124;
	a+=c^(d&(b^c));
	a=(a<<13)|(BITS32(a)>>19);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[2];
	e+=0x5c4dd124;
	e+=b^(c&(a^b));
	e=(e<<11)|(BITS32(e)>>21);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	/* parallel round three */

	d+=w[15];
	d+=0x6d703ef3;
	d+=(e|~a)^b;
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[5];
	c+=0x6d703ef3;
	c+=(d|~e)^a;
	c=(c<<7)|(BITS32(c)>>25);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[1];
	b+=0x6d703ef3;
	b+=(c|~d)^e;
	b=(b<<15)|(BITS32(b)>>17);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[3];
	a+=0x6d703ef3;
	a+=(b|~c)^d;
	a=(a<<11)|(BITS32(a)>>21);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[7];
	e+=0x6d703ef3;
	e+=(a|~b)^c;
	e=(e<<8)|(BITS32(e)>>24);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[14];
	d+=0x6d703ef3;
	d+=(e|~a)^b;
	d=(d<<6)|(BITS32(d)>>26);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[6];
	c+=0x6d703ef3;
	c+=(d|~e)^a;
	c=(c<<6)|(BITS32(c)>>26);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[9];
	b+=0x6d703ef3;
	b+=(c|~d)^e;
	b=(b<<14)|(BITS32(b)>>18);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[11];
	a+=0x6d703ef3;
	a+=(b|~c)^d;
	a=(a<<12)|(BITS32(a)>>20);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[8];
	e+=0x6d703ef3;
	e+=(a|~b)^c;
	e=(e<<13)|(BITS32(e)>>19);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[12];
	d+=0x6d703ef3;
	d+=(e|~a)^b;
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[2];
	c+=0x6d703ef3;
	c+=(d|~e)^a;
	c=(c<<14)|(BITS32(c)>>18);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[10];
	b+=0x6d703ef3;
	b+=(c|~d)^e;
	b=(b<<13)|(BITS32(b)>>19);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[0];
	a+=0x6d703ef3;
	a+=(b|~c)^d;
	a=(a<<13)|(BITS32(a)>>19);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[4];
	e+=0x6d703ef3;
	e+=(a|~b)^c;
	e=(e<<7)|(BITS32(e)>>25);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[13];
	d+=0x6d703ef3;
	d+=(e|~a)^b;
	d=(d<<5)|(BITS32(d)>>27);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	/* parallel round four */

	c+=w[8];
	c+=0x7a6d76e9;
	c+=a^(d&(e^a));
	c=(c<<15)|(BITS32(c)>>17);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[6];
	b+=0x7a6d76e9;
	b+=e^(c&(d^e));
	b=(b<<5)|(BITS32(b)>>27);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[4];
	a+=0x7a6d76e9;
	a+=d^(b&(c^d));
	a=(a<<8)|(BITS32(a)>>24);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[1];
	e+=0x7a6d76e9;
	e+=c^(a&(b^c));
	e=(e<<11)|(BITS32(e)>>21);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[3];
	d+=0x7a6d76e9;
	d+=b^(e&(a^b));
	d=(d<<14)|(BITS32(d)>>18);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[11];
	c+=0x7a6d76e9;
	c+=a^(d&(e^a));
	c=(c<<14)|(BITS32(c)>>18);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[15];
	b+=0x7a6d76e9;
	b+=e^(c&(d^e));
	b=(b<<6)|(BITS32(b)>>26);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[0];
	a+=0x7a6d76e9;
	a+=d^(b&(c^d));
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[5];
	e+=0x7a6d76e9;
	e+=c^(a&(b^c));
	e=(e<<6)|(BITS32(e)>>26);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[12];
	d+=0x7a6d76e9;
	d+=b^(e&(a^b));
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[2];
	c+=0x7a6d76e9;
	c+=a^(d&(e^a));
	c=(c<<12)|(BITS32(c)>>20);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[13];
	b+=0x7a6d76e9;
	b+=e^(c&(d^e));
	b=(b<<9)|(BITS32(b)>>23);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[9];
	a+=0x7a6d76e9;
	a+=d^(b&(c^d));
	a=(a<<12)|(BITS32(a)>>20);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[7];
	e+=0x7a6d76e9;
	e+=c^(a&(b^c));
	e=(e<<5)|(BITS32(e)>>27);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[10];
	d+=0x7a6d76e9;
	d+=b^(e&(a^b));
	d=(d<<15)|(BITS32(d)>>17);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[14];
	c+=0x7a6d76e9;
	c+=a^(d&(e^a));
	c=(c<<8)|(BITS32(c)>>24);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	/* parallel round five */

	b+=w[12];
	b+=c^d^e;
	b=(b<<8)|(BITS32(b)>>24);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[15];
	a+=b^c^d;
	a=(a<<5)|(BITS32(a)>>27);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[10];
	e+=a^b^c;
	e=(e<<12)|(BITS32(e)>>20);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[4];
	d+=e^a^b;
	d=(d<<9)|(BITS32(d)>>23);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[1];
	c+=d^e^a;
	c=(c<<12)|(BITS32(c)>>20);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[5];
	b+=c^d^e;
	b=(b<<5)|(BITS32(b)>>27);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[8];
	a+=b^c^d;
	a=(a<<14)|(BITS32(a)>>18);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[7];
	e+=a^b^c;
	e=(e<<6)|(BITS32(e)>>26);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[6];
	d+=e^a^b;
	d=(d<<8)|(BITS32(d)>>24);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[2];
	c+=d^e^a;
	c=(c<<13)|(BITS32(c)>>19);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[13];
	b+=c^d^e;
	b=(b<<6)|(BITS32(b)>>26);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	a+=w[14];
	a+=b^c^d;
	a=(a<<5)|(BITS32(a)>>27);
	a+=e;
	c=(c<<10)|(BITS32(c)>>22);

	e+=w[0];
	e+=a^b^c;
	e=(e<<15)|(BITS32(e)>>17);
	e+=d;
	b=(b<<10)|(BITS32(b)>>22);

	d+=w[3];
	d+=e^a^b;
	d=(d<<13)|(BITS32(d)>>19);
	d+=c;
	a=(a<<10)|(BITS32(a)>>22);

	c+=w[9];
	c+=d^e^a;
	c=(c<<11)|(BITS32(c)>>21);
	c+=b;
	e=(e<<10)|(BITS32(e)>>22);

	b+=w[11];
	b+=c^d^e;
	b=(b<<11)|(BITS32(b)>>21);
	b+=a;
	d=(d<<10)|(BITS32(d)>>22);

	/* create new block result */

	d+=w[18]+rmd160[1];
	rmd160[1]=rmd160[2]+w[19]+e;
	rmd160[2]=rmd160[3]+w[20]+a;
	rmd160[3]=rmd160[4]+w[16]+b;
	rmd160[4]=rmd160[0]+w[17]+c;
	rmd160[0]=d;
}

/*
 * procedure rmd160init
 *
 * input: ptr - pointer to RMD160 internal data
 *
 * This procedure initializes the RMD160 internal data structure.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160init(register RMD160DATA *ptr)
{
	/* initialize all RMD160 internal data */

	ptr->total=ptr->size=0;
	ptr->rmd160[0]=0x67452301;
	ptr->rmd160[1]=0xefcdab89;
	ptr->rmd160[2]=0x98badcfe;
	ptr->rmd160[3]=0x10325476;
	ptr->rmd160[4]=0xc3d2e1f0;
}

#endif

/*
 * procedure rmd160next
 *
 * input: data   - the data to be hashed
 *	  length - the length of the data to be hashed
 *	  ptr    - pointer to RMD160 internal data
 *
 * This procedure hashes the given block of data. Incomplete data blocks
 * are buffered in the RMD160 internal data structure.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160next(register WORD08 *data,register WORD32 length,
	register RMD160DATA *ptr)
{
	register WORD32 i;	/* buffer size/index	*/


	/* adjust total length */

	ptr->total+=length;

	/* while there is a remainder in the buffer and data are to be
	   processed and the buffer is notfull copy current byte to
	   buffer */

	for(i=ptr->size;(i&63)&&length;length--)ptr->bfr.b[i++]=*data++;

	/* if the buffer is full */

	if(i==64)
	{
		/* reset buffer size */

		i=0;

		/* hash buffer */

		rmd160block(ptr->rmd160,ptr->bfr.b);
	}

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)
		rmd160block(ptr->rmd160,data);

	/* copy remainder to buffer */

	for(;length;length--)ptr->bfr.b[i++]=*data++;

	/* save new buffer size/index */

	ptr->size=(WORD08)(i);
}

#endif

/*
 * procedure rmd160end
 *
 * input:  ptr - pointer to RMD160 internal data
 *
 * output: result - the RMD160 hash
 *
 * This procedure finalizes and returns the RMD160 hash.
 */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)

void CRYPTOCALL rmd160end(WORD08 *result,RMD160DATA *ptr)
{
	register WORD32 i=ptr->size;		/* counter		*/


	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	ptr->bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)ptr->bfr.b[i]=0;
		i=0;
		rmd160block(ptr->rmd160,ptr->bfr.b);
	}
	for(;i<56;i++)ptr->bfr.b[i]=0;

	/* append bits 0 through 31 of total length in bits to buffer */

	ptr->bfr.b[56]=(WORD08)((ptr->total)<<3);
	ptr->bfr.b[57]=(WORD08)((ptr->total)>>5);
	ptr->bfr.b[58]=(WORD08)((ptr->total)>>13);
	ptr->bfr.b[59]=(WORD08)((ptr->total)>>21);

	/* append bits 32 through 63 of total length in bits to buffer */

	ptr->bfr.b[60]=(WORD08)(BITS32(ptr->total)>>29);
	ptr->bfr.b[61]=0;
	ptr->bfr.b[62]=0;
	ptr->bfr.b[63]=0;

	/* hash buffer */

	rmd160block(ptr->rmd160,ptr->bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08) (ptr->rmd160[0]);
	result[ 1]=(WORD08)((ptr->rmd160[0])>>8);
	result[ 2]=(WORD08)((ptr->rmd160[0])>>16);
	result[ 3]=(WORD08)((ptr->rmd160[0])>>24);
	result[ 4]=(WORD08) (ptr->rmd160[1]);
	result[ 5]=(WORD08)((ptr->rmd160[1])>>8);
	result[ 6]=(WORD08)((ptr->rmd160[1])>>16);
	result[ 7]=(WORD08)((ptr->rmd160[1])>>24);
	result[ 8]=(WORD08) (ptr->rmd160[2]);
	result[ 9]=(WORD08)((ptr->rmd160[2])>>8);
	result[10]=(WORD08)((ptr->rmd160[2])>>16);
	result[11]=(WORD08)((ptr->rmd160[2])>>24);
	result[12]=(WORD08) (ptr->rmd160[3]);
	result[13]=(WORD08)((ptr->rmd160[3])>>8);
	result[14]=(WORD08)((ptr->rmd160[3])>>16);
	result[15]=(WORD08)((ptr->rmd160[3])>>24);
	result[16]=(WORD08) (ptr->rmd160[4]);
	result[17]=(WORD08)((ptr->rmd160[4])>>8);
	result[18]=(WORD08)((ptr->rmd160[4])>>16);
	result[19]=(WORD08)((ptr->rmd160[4])>>24);

	/* assert user data removal data from buffer */

#ifdef RMD160_PARANOID
	ptr->bfr.l[0]=0;
	ptr->bfr.l[1]=0;
	ptr->bfr.l[2]=0;
	ptr->bfr.l[3]=0;
	ptr->bfr.l[4]=0;
	ptr->bfr.l[5]=0;
	ptr->bfr.l[6]=0;
	ptr->bfr.l[7]=0;
	ptr->bfr.l[8]=0;
	ptr->bfr.l[9]=0;
	ptr->bfr.l[10]=0;
	ptr->bfr.l[11]=0;
	ptr->bfr.l[12]=0;
	ptr->bfr.l[13]=0;
	ptr->bfr.l[14]=0;
	ptr->bfr.l[15]=0;
#endif
}

#endif

/*
 * procedure rmd160
 *
 * input:  data   - the data to be hashed
 *	   length - the length of the data to be hashed
 *
 * output: result - the RMD160 hash
 *
 * This procedure hashes the given data.
 */

#ifdef RMD160_FAST

void CRYPTOCALL rmd160(register WORD08 *data,register WORD32 length,
	WORD08 *result)
{
	register WORD32 i;	/* counter and index		*/
	WORD32 total;		/* data length buffer		*/
	WORD32 rmd160[5];	/* RMD160 current result buffer	*/
	union
	{
	  WORD32 l[16];		/* paranoia memory clear	*/
	  WORD08 b[64];		/* final block(s) buffer	*/
	}bfr;


	/* initialize RMD160 internal data */

	total=length;
	rmd160[0]=0x67452301;
	rmd160[1]=0xefcdab89;
	rmd160[2]=0x98badcfe;
	rmd160[3]=0x10325476;
	rmd160[4]=0xc3d2e1f0;

	/* do all complete blocks */

	for(;length>63;length-=64,data+=64)rmd160block(rmd160,data);

	/* copy remainder to buffer */

	for(i=0;length;length--)bfr.b[i++]=*data++;

	/* fill in first pad (there's always space available in the buffer)
	   and then pad buffer with zeroes until a buffer length of 56 is
	   reached, hash buffer and reset vars, if buffer is full */

	bfr.b[i++]=0x80;
	if(i>56)
	{
		for(;i<64;i++)bfr.b[i]=0;
		i=0;
		rmd160block(rmd160,bfr.b);
	}
	for(;i<56;i++)bfr.b[i]=0;

	/* append bits 0 through 31 of total length in bits to buffer */

	bfr.b[56]=(WORD08)((total)<<3);
	bfr.b[57]=(WORD08)((total)>>5);
	bfr.b[58]=(WORD08)((total)>>13);
	bfr.b[59]=(WORD08)((total)>>21);

	/* append bits 32 through 63 of total length in bits to buffer */

	bfr.b[60]=(WORD08)(BITS32(total)>>29);
	bfr.b[61]=0;
	bfr.b[62]=0;
	bfr.b[63]=0;

	/* hash buffer */

	rmd160block(rmd160,bfr.b);

	/* return final hash result to caller */

	result[ 0]=(WORD08) (rmd160[0]);
	result[ 1]=(WORD08)((rmd160[0])>>8);
	result[ 2]=(WORD08)((rmd160[0])>>16);
	result[ 3]=(WORD08)((rmd160[0])>>24);
	result[ 4]=(WORD08) (rmd160[1]);
	result[ 5]=(WORD08)((rmd160[1])>>8);
	result[ 6]=(WORD08)((rmd160[1])>>16);
	result[ 7]=(WORD08)((rmd160[1])>>24);
	result[ 8]=(WORD08) (rmd160[2]);
	result[ 9]=(WORD08)((rmd160[2])>>8);
	result[10]=(WORD08)((rmd160[2])>>16);
	result[11]=(WORD08)((rmd160[2])>>24);
	result[12]=(WORD08) (rmd160[3]);
	result[13]=(WORD08)((rmd160[3])>>8);
	result[14]=(WORD08)((rmd160[3])>>16);
	result[15]=(WORD08)((rmd160[3])>>24);
	result[16]=(WORD08) (rmd160[4]);
	result[17]=(WORD08)((rmd160[4])>>8);
	result[18]=(WORD08)((rmd160[4])>>16);
	result[19]=(WORD08)((rmd160[4])>>24);

	/* assert user data removal data from buffer */

#ifdef RMD160_PARANOID
	bfr.l[0]=0;
	bfr.l[1]=0;
	bfr.l[2]=0;
	bfr.l[3]=0;
	bfr.l[4]=0;
	bfr.l[5]=0;
	bfr.l[6]=0;
	bfr.l[7]=0;
	bfr.l[8]=0;
	bfr.l[9]=0;
	bfr.l[10]=0;
	bfr.l[11]=0;
	bfr.l[12]=0;
	bfr.l[13]=0;
	bfr.l[14]=0;
	bfr.l[15]=0;
#endif

}

#endif

/*
 * procedure rmd160hmkey
 *
 * input:  key    - the HMAC key
 *	   length - the length of the HMAC key
 *
 * output: ptr - the preprocessed HMAC key hashes
 *
 * This procedure processes a key for HMAC use.
 */

#ifdef RMD160_HMAC

void CRYPTOCALL rmd160hmkey(WORD08 *key,WORD32 keylength,RMD160HMDATA *ptr)
{
	register WORD32 i;		/* counter and index	*/
	RMD160(hash);			/* hash of key		*/
	union
	{
		WORD08 pad[64];		/* padding buffer	*/
		RMD160DATA rmd160data;	/* RMD160 internal data	*/
	}u;


	/* if the key is larger than 64 bytes hash it and use the
	   hash as the key */

	if(keylength>64)
	{
		rmd160init(&u.rmd160data);
		rmd160next(key,keylength,&u.rmd160data);
		rmd160end(hash,&u.rmd160data);
		key=hash;
		keylength=RMD160_SIZE;
		u.rmd160data.rmd160[0]=u.rmd160data.rmd160[1]=
			u.rmd160data.rmd160[2]=u.rmd160data.rmd160[3]=
			u.rmd160data.rmd160[4]=0;
	}

	/* copy initial values to HMAC internal data structure */

	ptr->irmd160[0]=ptr->ormd160[0]=0x67452301;
	ptr->irmd160[1]=ptr->ormd160[1]=0xefcdab89;
	ptr->irmd160[2]=ptr->ormd160[2]=0x98badcfe;
	ptr->irmd160[3]=ptr->ormd160[3]=0x10325476;
	ptr->irmd160[4]=ptr->ormd160[4]=0xc3d2e1f0;

	/* copy key to padding buffer, fill up buffer
	   and pre-hash inner pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x36;
	for(;i<64;i++)u.pad[i]=0x36;
	rmd160block(ptr->irmd160,u.pad);

	/* copy key to padding buffer, fill up buffer
	   and pre-hash outer pad */

	for(i=0;i<keylength;i++)u.pad[i]=key[i]^0x5c;
	for(;i<64;i++)u.pad[i]=0x5c;
	rmd160block(ptr->ormd160,u.pad);

	/* clean up */

	for(i=0;i<RMD160_SIZE;i++)hash[i]=0;
	for(i=0;i<64;i++)u.pad[i]=0;
}

#endif

/*
 * procedure rmd160hminit
 *
 * input:  key - the preprocessed HMAC key
 *
 * output: ptr - the RMD160 internal data
 *
 * This procedure initializes RMD160 data for HMAC use.
 */

#ifdef RMD160_HMAC_FULL

void CRYPTOCALL rmd160hminit(RMD160DATA *ptr,RMD160HMDATA *key)
{
	/* initialize all RMD160 internal data */

	ptr->total=64;
	ptr->size=0;
	ptr->rmd160[0]=key->irmd160[0];
	ptr->rmd160[1]=key->irmd160[1];
	ptr->rmd160[2]=key->irmd160[2];
	ptr->rmd160[3]=key->irmd160[3];
	ptr->rmd160[4]=key->irmd160[4];
}

#endif

/*
 * procedure rmd160hmend
 *
 * input:  key - the preprocessed HMAC key
 *	   ptr - the RMD160 internal data
 *
 * output: result - the HMAC result (a RMD160 hash)
 *
 * This procedure finalizes and returns the HMAC.
 */

#ifdef RMD160_HMAC_FULL

void CRYPTOCALL rmd160hmend(WORD08 *result,RMD160DATA *ptr,RMD160HMDATA *key)
{
	/* complete inner RMD160 */

	rmd160end(result,ptr);

	/* initialize outer RMD160 */

	ptr->total=64;
	ptr->size=0;
	ptr->rmd160[0]=key->ormd160[0];
	ptr->rmd160[1]=key->ormd160[1];
	ptr->rmd160[2]=key->ormd160[2];
	ptr->rmd160[3]=key->ormd160[3];
	ptr->rmd160[4]=key->ormd160[4];

	rmd160next(result,RMD160_SIZE,ptr);
	rmd160end(result,ptr);
}

#endif

/*
 * procedure rmd160hmac
 *
 * input:  data   - the data to be processed
 *	   length - the length of the data to be processed
 *	   key    - the preprocessed HMAC key
 *
 * output: result - the HMAC result (a RMD160 hash)
 *
 * This procedure creates and returns a HMAC.
 */

#ifdef RMD160_HMAC_FAST

void CRYPTOCALL rmd160hmac(WORD08 *data,WORD32 length,WORD08 *result,
	RMD160HMDATA *key)
{
	RMD160DATA rmd160data;	/* RMD160 internal data */


	/* initialize inner RMD160 */

	rmd160data.total=64;
	rmd160data.size=0;
	rmd160data.rmd160[0]=key->irmd160[0];
	rmd160data.rmd160[1]=key->irmd160[1];
	rmd160data.rmd160[2]=key->irmd160[2];
	rmd160data.rmd160[3]=key->irmd160[3];
	rmd160data.rmd160[4]=key->irmd160[4];

	/* hash given data */

	rmd160next(data,length,&rmd160data);

	/* complete inner RMD160 */

	rmd160end(result,&rmd160data);

	/* initialize outer RMD160 */

	rmd160data.total=64;
	rmd160data.size=0;
	rmd160data.rmd160[0]=key->ormd160[0];
	rmd160data.rmd160[1]=key->ormd160[1];
	rmd160data.rmd160[2]=key->ormd160[2];
	rmd160data.rmd160[3]=key->ormd160[3];
	rmd160data.rmd160[4]=key->ormd160[4];

	/* complete and return outer RMD160 */

	rmd160next(result,RMD160_SIZE,&rmd160data);
	rmd160end(result,&rmd160data);
}

#endif

