/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, ast@domdv.de
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* the slaves: data block send and receive                                    */
/*============================================================================*/

/*
 * dosend
 *
 * input:  data   - the data to be sent
 *	   length - the length of the data to be sent
 *	   anchor - pointer to vpnd global data
 *
 * return: 1 - success
 *	   0 - failure
 *
 * This procedure sends data to the remote side.
 * The first byte must be transferred within 5 seconds,
 * all other bytes must be transferred within a one
 * second per byte interval.
 */

int dosend(WORD08 *data,WORD32 length,VPN *anchor)
{
	int l;			/* amount of data written		*/
	fd_set *s;		/* write set for select call		*/
	fd_set *e;		/* exception set for select call	*/
	struct timeval t;	/* time out for select call		*/
	WORD08 xflag;		/* character escape processing state	*/
	WORD08 bfr;		/* character escape work buffer		*/


	/* debug message */

	ENTER("dosend");

#ifdef DEBUG
	if(debug>5)
	{
		for(l=0;l<length;l++)printf("%02x ",data[l]);
		printf("\n");
	}
#endif

	/* set up file descriptor set pointers */

	s=&anchor->s;
	e=&anchor->x;

	/* reset escaping state */

	xflag=0;

	/* while there is data to be sent */

	while(length)
	{
		/* set up timeout for select call */

		t.tv_sec=anchor->txmax;
		t.tv_usec=0;

		/* set up sets for select call */

		FD_ZERO(s);
		FD_SET(anchor->peer,s);
		FD_ZERO(e);
		FD_SET(anchor->peer,e);

		/* wait for data send possible, signal error, if not */

		if((l=select(anchor->peer+1,NULL,s,e,&t))==-1)
		{
			if(errno==EINTR||errno==EAGAIN)
			{
				if(terminate)RET("termination request",0);
				continue;
			}
		}
		else if(!l)RETURN("select timeout",0);

		/* signal error if exception occurred */

		if(FD_ISSET(anchor->peer,e))RETURN("select exception",0);

		/* send data, react to errors */

		if(anchor->serialmode)
		{
			/* in XON/XOFF filter mode act according to 
			   escaping state */

			if(anchor->xfilter)switch(xflag)
			{
			/* in case of a to be escaped character */

			case 0:	if(*data==XON||*data==XOFF||*data==ESC)
				{
					/* set new escaping state */

					xflag=1;

					/* write escape character to buffer */

					bfr=ESC;
				}

				/* done */

				break;

			/* in case of the character to be escaped escape character */

			case 2: bfr=*data^0x80;

				/* done */

				break;
			}

			if((l=write(anchor->peer,xflag?&bfr:data,
				anchor->xfilter?1:length))<0)
			{
				if(errno==EINTR||errno==EWOULDBLOCK)continue;
				RETURN("write",0);
			}
			else if(!l)RET("write (0 bytes)",0);

			/* in escaping state process escaping states */

			if(xflag)
			{
				if(++xflag!=3)continue;
				else xflag=0;
			}
		}
		else if((l=send(anchor->peer,data,length,0))<0)
		{
			if(errno==EINTR||errno==EWOULDBLOCK)continue;
			RETURN("write",0);
		}
		else if(!l)RET("send (0 bytes)",0);

		/* adjust data pointer and remaining length */

		data+=l;
		length-=l;
	}

	/* debug message */

	LEAVE("dosend");

	/* signal success */

	return 1;
}

/*
 * dorecv
 *
 * input:  lng    - the length of the data to be received
 *	   anchor - pointer to vpnd global data
 *
 * output: dta - the data to be received
 *
 * return: 1 - success
 *	   0 - failure
 *
 * This procedure receives data from the remote side.
 * The first byte must be transferred within 5 seconds,
 * all other bytes must be transferred within a one
 * second per byte interval.
 */

int dorecv(WORD08 *dta,WORD32 lng,VPN *anchor)
{
	int l;			/* amount of data read			*/
	fd_set *s;		/* read set for select call		*/
	fd_set *e;		/* exception set for select call	*/
	struct timeval t;	/* time out for select call		*/
	WORD08 xflag;		/* character escape processing state	*/
	WORD32 length=lng;	/* working copy of data length		*/
	WORD08 *data=dta;	/* working copy of data buffer		*/
#ifdef DEBUG
	int len=length;		/* length memory for debug output	*/
	WORD08 *ddta=data;	/* data pointer memory for debug output	*/
#endif


	/* debug message */

	ENTER("dorecv");

	/* set up file descriptor set pointers */

	s=&anchor->s;
	e=&anchor->x;

	/* reset escaping state */

	xflag=0;

	/* while there is data to be read */

	while(length)
	{
		/* set up timeout for select call */

		t.tv_sec=anchor->rxmax;
		t.tv_usec=0;

		/* set up sets for select call */

		FD_ZERO(s);
		FD_SET(anchor->peer,s);
		FD_ZERO(e);
		FD_SET(anchor->peer,e);

		/* wait for data read possible, signal error, if not */

		if((l=select(anchor->peer+1,s,NULL,e,&t))==-1)
		{
			if(errno==EINTR||errno==EAGAIN)
			{
				if(terminate)RET("termination request",0);
				continue;
			}
		}
		else if(!l)RET("select timeout",0);

		/* signal error if exception occurred */

		if(FD_ISSET(anchor->peer,e))RETURN("select exception",0);

		/* serial line mode */

		if(anchor->serialmode)
		{
			/* read data from serial line */

			if((l=read(anchor->peer,data,anchor->xfilter?1:length))
				<0)
			{
				if(errno==EINTR||errno==EWOULDBLOCK)continue;
				RETURN("read",0);
			}
			else if(!l)RET("read (0 bytes)",0);

			/* in case of XON/XOFF filter */

			if(anchor->xfilter)
			{
				/* ignore all XON/XOFF characters */

				if(*data==XON||*data==XOFF)continue;

				/* process escaping state */

				if(!xflag)
				{
					/* if an escape character is received */

					if(*data==ESC)
					{
						/* increase escaping state */

						xflag++;

						/* skip character */

						continue;
					}
				}
				else
				{
					/* in case of an escaped character
					   restore character */

					*data^=0x80;

					/* reset escaping state */

					xflag=0;
				}
			}
		}

		/* read data from socket */

		else if((l=recv(anchor->peer,data,length,0))<0)
		{
			if(errno==EINTR||errno==EWOULDBLOCK)continue;
			RETURN("recv",0);
		}
		else if(!l)RET("recv (0 bytes)",0);

		/* adjust data pointer and remaining length */

		data+=l;
		length-=l;
	}

	/* debug message */

#ifdef DEBUG
	if(debug>5)
	{
		for(l=0;l<len;l++)printf("%02x ",ddta[l]);
		printf("\n");
	}
#endif

	LEAVE("dorecv");

	/* signal success */

	return 1;
}

/*
 * oobsend
 *
 * input:  request - the out of band request to be sent
 *	   anchor  - pointer to vpnd global data
 *
 * return: 1 - the request was sent (or ignored in serial line mode)
 *	   0 - request send error
 */

int oobsend(int request,VPN *anchor)
{
	char data;	/* out of band data */


	/* debug message */

	ENTER("oobsend");

	/* in TCP/IP mode process the request */

	if(!anchor->serialmode)
	{
		if(request==PING)
		{
			/* prepare linkcheck request */

			data=PING;

			/* send out of band data, handle errors */

			if(send(anchor->peer,&data,1,MSG_OOB)!=1)
				RETURN("oobsend",0);
		}

		/* unknown request, signal error */

		else RET("oobsend: illegal request value",0);
	}

	/* debug message */

	LEAVE("oobsend");

	/* signal success */

	return 1;
}

/*
 * oobrecv
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0 - no out of band data
 *	   1 - out of band data received
 *	   2 - linktest reply received
 *
 * This procedure processes a single byte of out of band data
 * if there is any.
 */

int oobrecv(VPN *anchor)
{
	int oob;	/* return value			*/
	char data;	/* out ofband data		*/
	char bug;	/* kernel bug workaround	*/


	/* debug message */

	ENTER("oobrecv");

	/* in TCP/IP mode */

	if(!anchor->serialmode)
	{
		/* if out of band data reception fails */

		if((oob=recv(anchor->peer,&data,1,MSG_OOB))<0)
		{
			/* ignore interrupts and no data condition */

			if(errno==EINTR||errno==EWOULDBLOCK)oob=0;

			/* handle all other errors */

			else RETURN("recv",0);
		}

		/* if out of band data was received */

		else if(oob)
		{
			/* dangerous but required as a fix for
			   Linux kernels < 2.0.36, otherwise
			   select() in the sequencer routine
			   will endlessly report an exception
			   on the peer socket, on the other
			   hand this call may trash the
			   whole interworking thus causing
			   the connection to be closed
			   and reopened */

			if(anchor->oobfix)recv(anchor->peer,&bug,1,0);

			/* debug message */

			DEBUG2(3,"oobrecv, data=%02x\n",data);

			/* process out of band data */

			switch(data)
			{
			/* linktest request */

			case PING:

				/* send linktest reply, ignore errors */

				data=PONG;
				send(anchor->peer,&data,1,MSG_OOB);
				break;

			/* linktest reply */

			case PONG:

				/* set proper return value */

				oob++;
				break;
			}
		}
	}

	/* no out of band processing in serial mode */

	else oob=0;

	/* debug message */

	LEAVE("oobrecv");

	/* signal oob processing */

	return oob;
}
