/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2007 Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include <speex/speex.h>
#include <stdlib.h>
#include "gsmlib/inc/gsm.h"
#include "vpnd.h"

/* convert one word of pcm to one alaw encoded byte */

#define pcm2alaw(in,out)				\
do {							\
	int v,m,s,t;					\
	if((v=in)>=0)m=0xd5;				\
	else {m=0x55;v=-v-8;}				\
	for(s=0,t=0xff;s<8;s++,t=(t<<1)|1)if(v<=t)break;\
	if(s==8)out=0x7f^m;				\
	else if(s<2)out=((s<<4)|((v>>4)&0xf))^m;	\
	else out=((s<<4)|((v>>(s+3))&0xf))^m;		\
} while(0)

/* convert one alaw encoded byte to one word of pcm */

#define alaw2pcm(in,out)				\
do {							\
	unsigned char v;				\
	int s,t=((v=in^0x55)&0xf)<<4;			\
	if(!(s=(v&0x70)>>4))t+=8;			\
	else if(s==1)t+=0x108;				\
	else t=(t+0x108)<<(s-1);			\
	out=(v&0x80)?t:-t;				\
} while(0)

/* convert one word of pcm to one ulaw encoded byte */

#define pcm2ulaw(in,out)				\
do {							\
	int v,m,s,t;					\
	if((v=in)<0){v=0x84-v;m=0x7f;}			\
	else {v+=0x84;m=0xff;}				\
	for(s=0,t=0xff;s<8;s++,t=(t<<1)|1)if(v<=t)break;\
	if(s==8)out=0x7f^m;				\
	else out=((s<<4)|((v>>(s+3))&0xf))^m;		\
} while(0)

/* convert one alaw encoded byte to one word of pcm */

#define ulaw2pcm(in,out)				\
do {							\
	unsigned char v=~in;				\
	int t=(((v&0xf)<<3)+0x84)<<((v&0x70)>>4);	\
	out=(v&0x80)?(0x84-t):(t-0x84);			\
} while(0)

/* recode working context */

typedef struct
{
	int mode;	/* processing mode		*/
	int fsize;	/* uncompressed frame size	*/
	void *s;	/* encode/decode handle		*/
	SpeexBits b;	/* speex bit area		*/
} SPX;

/*
 * alaw2gsm
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet on input, gsm packet on output
 *
 * return: the gsm packet length or 0 if encoding failure
 *
 * This procedure recodes an alaw encoded audio packet to an
 * gsm encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int alaw2gsm(void *handle,char *data,int len,int max)
{
	int rlen;		/* return length		*/
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* abort if buffer length or output size exceeded */

	if(len>1024||max<33)return 0;

	/* get compression context */

	spx=(SPX *)(handle);

	/* if frame size matches input packet size */

	if(spx->fsize==len)
	{
		/* convert alaw packet to pcm */

		for(rlen=0;rlen<len;rlen++)alaw2pcm(data[rlen],conv[rlen]);

		/* return gsm encoded packet */

		gsm_encode(spx->s,conv,(unsigned char *)data);

		/* set up encoded packet length */

		rlen=33;
	}

	/* fail if frame size mismatch */

	else rlen=0;

	/* return new paket length or zero */

	return rlen;
}

/*
 * gsm2alaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet on input, gsm packet on output
 *
 * return: the alaw packet length or 0 if decoding failure
 *
 * This procedure recodes an gsm encoded audio packet to an
 * alaw encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int gsm2alaw(void *handle,char *data,int len,int max)
{
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* get compression context */

	spx=(SPX *)(handle);

	/* abort if buffer or output length exceeded */

	if(spx->fsize>max||spx->fsize>1024)return 0;

	/* decode gsm to pcm */

	gsm_decode(spx->s,(unsigned char *)data,conv);

	/* encode pcm to alaw */

	for(len=0;len<spx->fsize;len++)pcm2alaw(conv[len],data[len]);

	/* return new paket length or zero */

	return spx->fsize;
}

/*
 * ulaw2gsm
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - ulaw packet on input, gsm packet on output
 *
 * return: the gsm packet length or 0 if encoding failure
 *
 * This procedure recodes an ulaw encoded audio packet to an
 * gsm encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int ulaw2gsm(void *handle,char *data,int len,int max)
{
	int rlen;		/* return length		*/
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* abort if buffer length or output size exceeded */

	if(len>1024||max<33)return 0;

	/* get compression context */

	spx=(SPX *)(handle);

	/* if frame size matches input packet size */

	if(spx->fsize==len)
	{
		/* convert ulaw packet to pcm */

		for(rlen=0;rlen<len;rlen++)ulaw2pcm(data[rlen],conv[rlen]);

		/* return gsm encoded packet */

		gsm_encode(spx->s,conv,(unsigned char *)data);

		/* set up encoded packet length */

		rlen=33;
	}

	/* fail if frame size mismatch */

	else rlen=0;

	/* return new paket length or zero */

	return rlen;
}

/*
 * gsm2ulaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - ulaw packet on input, gsm packet on output
 *
 * return: the ulaw packet length or 0 if decoding failure
 *
 * This procedure recodes an gsm encoded audio packet to an
 * ulaw encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int gsm2ulaw(void *handle,char *data,int len,int max)
{
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* get compression context */

	spx=(SPX *)(handle);

	/* abort if buffer or output length exceeded */

	if(spx->fsize>max||spx->fsize>1024)return 0;

	/* decode gsm to pcm */

	gsm_decode(spx->s,(unsigned char *)data,conv);

	/* encode pcm to ulaw */

	for(len=0;len<spx->fsize;len++)pcm2ulaw(conv[len],data[len]);

	/* return new paket length or zero */

	return spx->fsize;
}

/*
 * alaw2speex
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet on input, speex packet on output
 *
 * return: the speex packet length or 0 if encoding failure
 *
 * This procedure recodes an alaw encoded audio packet to an
 * speex encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int alaw2speex(void *handle,char *data,int len,int max)
{
	int rlen;		/* return length		*/
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* abort if buffer length exceeded */

	if(len>1024)return 0;

	/* get compression context */

	spx=(SPX *)(handle);

	/* if frame size matches input packet size */

	if(spx->fsize==len)
	{
		/* convert alaw packet to pcm */

		for(rlen=0;rlen<len;rlen++)alaw2pcm(data[rlen],conv[rlen]);

		/* reset speex bit structure */

		speex_bits_reset(&spx->b);

		/* encode pcm to speex */

		speex_encode_int(spx->s,conv,&spx->b);

		/* fail if output buffer exceeded */

		if(speex_bits_nbytes(&spx->b)>max)rlen=0;

		/* return speex encoded packet */

		else rlen=speex_bits_write(&spx->b,data,max);
	}

	/* fail if frame size mismatch */

	else rlen=0;

	/* return new paket length or zero */

	return rlen;
}

/*
 * speex2alaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet on input, speex packet on output
 *
 * return: the alaw packet length or 0 if decoding failure
 *
 * This procedure recodes an speex encoded audio packet to an
 * alaw encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int speex2alaw(void *handle,char *data,int len,int max)
{
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* get compression context */

	spx=(SPX *)(handle);

	/* reset speex bit structure */

	speex_bits_reset(&spx->b);

	/* abort if buffer or output length exceeded */

	if(spx->fsize>max||spx->fsize>1024)return 0;

	/* decode speex to pcm */

	speex_bits_read_from(&spx->b,data,len);
	speex_decode_int(spx->s,&spx->b,conv);

	/* encode pcm to alaw */

	for(len=0;len<spx->fsize;len++)pcm2alaw(conv[len],data[len]);

	/* return new paket length or zero */

	return spx->fsize;
}

/*
 * ulaw2speex
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet on input, speex packet on output
 *
 * return: the speex packet length or 0 if encoding failure
 *
 * This procedure recodes an ulaw encoded audio packet to an
 * speex encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int ulaw2speex(void *handle,char *data,int len,int max)
{
	int rlen;		/* return length		*/
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* abort if buffer length exceeded */

	if(len>1024)return 0;

	/* get compression context */

	spx=(SPX *)(handle);

	/* if frame size matches input packet size */

	if(spx->fsize==len)
	{
		/* convert ulaw packet to pcm */

		for(rlen=0;rlen<len;rlen++)ulaw2pcm(data[rlen],conv[rlen]);

		/* reset speex bit structure */

		speex_bits_reset(&spx->b);

		/* encode pcm to speex */

		speex_encode_int(spx->s,conv,&spx->b);

		/* fail if output buffer exceeded */

		if(speex_bits_nbytes(&spx->b)>max)rlen=0;

		/* return speex encoded packet */

		else rlen=speex_bits_write(&spx->b,data,max);
	}

	/* fail if frame size mismatch */

	else rlen=0;

	/* return new paket length or zero */

	return rlen;
}

/*
 * speex2ulaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - ulaw packet on input, speex packet on output
 *
 * return: the ulaw packet length or 0 if decoding failure
 *
 * This procedure recodes an speex encoded audio packet to an
 * ulaw encoded audio packet. The packet must have a length
 * which is a multiple of 20ms.
 */

static int speex2ulaw(void *handle,char *data,int len,int max)
{
	SPX *spx;		/* compression context		*/
	short conv[1024];	/* intermediate pcm buffer	*/


	/* get compression context */

	spx=(SPX *)(handle);

	/* reset speex bit structure */

	speex_bits_reset(&spx->b);

	/* abort if buffer or output length exceeded */

	if(spx->fsize>max||spx->fsize>1024)return 0;

	/* decode speex to pcm */

	speex_bits_read_from(&spx->b,data,len);
	speex_decode_int(spx->s,&spx->b,conv);

	/* encode pcm to ulaw */

	for(len=0;len<spx->fsize;len++)pcm2ulaw(conv[len],data[len]);

	/* return new paket length or zero */

	return spx->fsize;
}

/*
 * recode_init
 *
 * input:  mode - 1=gsm decompression
 *		  2=gsm compression
 *		  3=speex decompression
 *		  4=speex compression
 *
 *	   old  - the previous compression context or NULL
 *
 * return: the new compression context or NULL
 *
 * This procedure creates a compression context.
 */

void *recode_init(int mode,void *old)
{
	int i;			/* general purpose usage	*/
	SPX *spx;		/* compression context		*/


	/* if there is an old context */

	if((spx=(SPX *)(old)))switch(spx->mode)
	{
	/* gsm encoder/decoder */

	case 1:
	case 2:	/* destroy gsm context */

		gsm_destroy(spx->s);

		/* done */

		break;

	/* speex decoder */

	case 3:	/* destroy speex bit area */

		speex_bits_destroy(&spx->b);

		/* destroy speex decoder */

		speex_decoder_destroy(spx->s);

		/* done */

		break;

	/* speex encoder */

	case 4:	/* destroy speex bit area */

		speex_bits_destroy(&spx->b);

		/* destroy speex encoder */

		speex_encoder_destroy(spx->s);

		/* done */

		break;
	}

	/* if there is no old context allocate new one, abort on error */

	else if(!(spx=malloc(sizeof(SPX))))return NULL;

	/* remember compression/decompression mode */

	spx->mode=mode;

	/* initialize speex bit area */

	speex_bits_init(&spx->b);

	/* act according to request */

	switch(mode)
	{
	/* gsm compression/decompression */

	case 1:
	case 2:	/* create gsm context */

		spx->s=gsm_create();

		/* setup frame size */

		spx->fsize=160;

		/* done */

		break;

	/* speex decompression */

	case 3:	/* create decompression context */

		spx->s=speex_decoder_init(&speex_nb_mode);

		/* enable perceptual enhancer */

		i=1;
		speex_decoder_ctl(spx->s,SPEEX_SET_ENH,&i);

		/* get decompressed frame size */

		speex_decoder_ctl(spx->s,SPEEX_GET_FRAME_SIZE,&spx->fsize);

		/* done */

		break;

	/* speex compression */

	case 4:	/* create compression context */

		spx->s=speex_encoder_init(&speex_nb_mode);

		/* set speech quality */

		i=8;
		speex_encoder_ctl(spx->s,SPEEX_SET_QUALITY,&i);

		/* set cpu resource usage */

		i=2;
		speex_encoder_ctl(spx->s,SPEEX_SET_COMPLEXITY,&i);

		/* enable voice activation detection */

		i=1;
		speex_encoder_ctl(spx->s,SPEEX_SET_VAD,&i);

		/* enable discontinous transmission status */

		i=1;
		speex_encoder_ctl(spx->s,SPEEX_SET_DTX,&i);

		/* get uncompressed frame size */

		speex_encoder_ctl(spx->s,SPEEX_GET_FRAME_SIZE,&spx->fsize);

		/* done */

		break;
	}

	/* return new context */

	return (void *)(spx);
}

/*
 * recode_exit
 *
 * input:  p - the compression context
 *
 * return: always NULL
 *
 * This procedure destroys a compression context.
 */

void *recode_exit(void *p)
{
	SPX *spx;		/* compression context */


	/* if there is a compression context */

	if((spx=(SPX *)(p)))
	{
		/* act according to context type */

		switch(spx->mode)
		{
			/* gsm encoder/decoder */

			case 1:
			case 2:	/* destroy gsm context */

				gsm_destroy(spx->s);

				/* done */

				break;

			/* speex decoder */

			case 3:	/* destroy speex bit area */

				speex_bits_destroy(&spx->b);

				/* destroy speex decoder */

				speex_decoder_destroy(spx->s);

				/* done */

				break;

			/* speex encoder */

			case 4:	/* destroy speex bit area */

				speex_bits_destroy(&spx->b);

				/* destroy speex encoder */

				speex_encoder_destroy(spx->s);

				/* done */

				break;
		}

		/* free compression context */

		free(spx);
	}

	/* signal success */

	return NULL;
}

/*
 * recode_alaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - alaw packet and encoded packet
 *
 * return: the resulting packet length or 0 if failure
 *
 * This procedure recodes an audio packet to or from alaw.
 */

int recode_alaw(void *handle,char *data,int len,int max)
{
	SPX *spx=(SPX *)(handle);	/* compression context */


	/* act according to compression mode */

	switch(spx->mode)
	{
	/* gsm decode */

	case 1:	return gsm2alaw(handle,data,len,max);

	/* gsm encode */

	case 2:	return alaw2gsm(handle,data,len,max);

	/* speex decode */

	case 3:	return speex2alaw(handle,data,len,max);

	/* speex encode */

	case 4:	return alaw2speex(handle,data,len,max);

	/* unhandled */

	default:return len;
	}
}

/*
 * recode_ulaw
 *
 * input:  handle - compression context
 *	   len    - the input packet length
 *         max    - the data buffer size
 *
 * inout:  data - ulaw packet and encoded packet
 *
 * return: the resulting packet length or 0 if failure
 *
 * This procedure recodes an audio packet to or from ulaw.
 */

int recode_ulaw(void *handle,char *data,int len,int max)
{
	SPX *spx=(SPX *)(handle);	/* compression context */


	/* act according to compression mode */

	switch(spx->mode)
	{
	/* gsm decode */

	case 1:	return gsm2ulaw(handle,data,len,max);

	/* gsm encode */

	case 2:	return ulaw2gsm(handle,data,len,max);

	/* speex decode */

	case 3:	return speex2ulaw(handle,data,len,max);

	/* speex encode */

	case 4:	return ulaw2speex(handle,data,len,max);

	/* unhandled */

	default:return len;
	}
}
